/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/Base/klVec3.h,v 1.8 2004/02/10 18:19:30 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLVEC3T_HEADERFILE__
#define __KLVEC3T_HEADERFILE__


// 3D vector implementation with templated basic data type
//
// FLOAT specifies the basic data type
// see klFloat_fixed for a sample implementation
//
template<class FLOAT>
class klVec3T
{
	enum
	{
		INT1SHIFT = 31 - 16		// 16-bits fixed point precision
	};

public:
	klVec3T()														{}
	klVec3T(float nV0, float nV1, float nV2)						{  v[0].setFloat(nV0);	v[1].setFloat(nV1);	v[2].setFloat(nV2);  }
	klVec3T(const FLOAT& nV0, const FLOAT& nV1, const FLOAT& nV2)	{  v[0] = nV0;  v[1] = nV1;  v[2] = nV2;  }
	klVec3T(const klVec3T& nOther)								{  v[0] = nOther.v[0];  v[1] = nOther.v[1];  v[2] = nOther.v[2];  }
	klVec3T(const klVec4T<FLOAT>& nOther)							{  v[0] = nOther[0];  v[1] = nOther[1];  v[2] = nOther[2];  }

	void setFixed(const int* nV)		{  v[0].setFixed(nV[0]);	v[1].setFixed(nV[1]);		v[2].setFixed(nV[2]);	}
	void setInt(const int* nV)			{  v[0].setInt(nV[0]);		v[1].setInt(nV[1]);			v[2].setInt(nV[2]);		}
	void setFloat(const float* nV)		{  v[0].setFloat(nV[0]);	v[1].setFloat(nV[1]);		v[2].setFloat(nV[2]);	}
	void setDouble(const double* nV)	{  v[0].setDouble(nV[0]);	v[1].setDouble(nV[1]);		v[2].setDouble(nV[2]);  }
	void setNative(const FLOAT* nV)		{  v[0] = nV[0];			v[1] = nV[1];				v[2] = nV[2];			}
	void setInt1(const int* nV)			{  v[0].setFixed(nV[0]>>INT1SHIFT);
										   v[1].setFixed(nV[1]>>INT1SHIFT);
										   v[2].setFixed(nV[2]>>INT1SHIFT);  }

	void setFixed(int nIdx, int nV)		{  v[nIdx].setFixed(nV);  }
	void setInt(int nIdx, int nV)		{  v[nIdx].setInt(nV);  }
	void setFloat(int nIdx, float nV)	{  v[nIdx].setFloat(nV);  }
	void setDouble(int nIdx, double nV)	{  v[nIdx].setDouble(nV);  }

	void setFixed(int nV0, int nV1, int nV2)			{  v[0].setFixed(nV0);	v[1].setFixed(nV1);		v[2].setFixed(nV2);  }
	void setInt(int nV0, int nV1, int nV2)				{  v[0].setInt(nV0);	v[1].setInt(nV1);		v[2].setInt(nV2);  }
	void setFloat(float nV0, float nV1, float nV2)		{  v[0].setFloat(nV0);  v[1].setFloat(nV1);		v[2].setFloat(nV2);  }
	void setDouble(double nV0, double nV1, double nV2)	{  v[0].setDouble(nV0);	v[1].setDouble(nV1);	v[2].setDouble(nV2);  }

	void getByte(unsigned char* nV)	{  for(int i=0; i<3; i++)  nV[i] = v[i].getByte();  }
	void getByte(int* nV)			{  for(int i=0; i<3; i++)  nV[i] = v[i].getByte();  }
	void getFixed(int* nV)			{  for(int i=0; i<3; i++)  nV[i] = v[i].getFixed();  }
	void getInt(int* nV)			{  for(int i=0; i<3; i++)  nV[i] = v[i].getInt();  }
	void getInt1(int* nV)			{  for(int i=0; i<3; i++)  nV[i] = v[i].getInt()<<INT1SHIFT;  }
	void getFloat(float* nV)		{  for(int i=0; i<3; i++)  nV[i] = v[i].getFloat();  }
	void getDouble(double* nV)		{  for(int i=0; i<3; i++)  nV[i] = v[i].getDouble();  }

	const FLOAT& operator[](int nIdx) const  {  return v[nIdx];  }
	FLOAT& operator[](int nIdx)  {  return v[nIdx];  }

	inline void normalize();

	FLOAT getLength() const;
	FLOAT getInvLength() const;

	inline void transform(const klMatrixT<FLOAT>& nM, const klVec3T<FLOAT>& nV);

	inline klVec3T& multiplyComonentWise(const klVec3T& nV0, const klVec3T& nV1);

	void multiplyBy255()		{  v[0].multiplyBy255();  v[1].multiplyBy255();  v[2].multiplyBy255();  }

	inline FLOAT dot(const klVec3T& nOther);

	// overloaded operators
	//
	inline klVec3T& operator+=(const klVec3T& nO);
	inline klVec3T& operator-=(const klVec3T& nO);

	inline klVec3T& operator*=(const FLOAT& nV);

	bool operator==(const klVec3T& nO) const  {  return v[0]==nO.v[0] && v[1]==nO.v[1] && v[2]==nO.v[2];  }

	klVec3T operator-() const  {  return klVec3T(-v[0], -v[1], -v[2]);  }

	// unfortunately template declarations are handled quite differentely on various compilers...
	//
#ifdef _IS_WINDOWS_
	friend inline const klVec3T operator+(const klVec3T& left, const klVec3T& right);
	friend inline const klVec3T operator-(const klVec3T& left, const klVec3T& right);
#else
#ifdef GCC_295X
	friend const klVec3T<FLOAT> operator+ (const klVec3T& left, const klVec3T& right);
	friend const klVec3T<FLOAT> operator- (const klVec3T& left, const klVec3T& right);
#else
	friend const klVec3T<FLOAT> operator+<> (const klVec3T& left, const klVec3T& right);
	friend const klVec3T<FLOAT> operator-<> (const klVec3T& left, const klVec3T& right);
#endif // GCC_2.95.X
#endif // _IS_WINDOWS_
protected:
	FLOAT v[3];
};


template<class FLOAT> inline klVec3T<FLOAT>&
klVec3T<FLOAT>::operator*=(const FLOAT& nV)
{
	v[0]*=nV;  v[1]*=nV;  v[2]*=nV;
	return *this;
}


template<class FLOAT> inline klVec3T<FLOAT>&
klVec3T<FLOAT>::operator+=(const klVec3T& nO)
{
	v[0]+=nO.v[0];  v[1]+=nO.v[1];  v[2]+=nO.v[2];
	return *this;
}


template<class FLOAT> inline klVec3T<FLOAT>&
klVec3T<FLOAT>::operator-=(const klVec3T& nO)
{
	v[0]-=nO.v[0];  v[1]-=nO.v[1];  v[2]-=nO.v[2];
	return *this;
}


template<class FLOAT> inline const klVec3T<FLOAT>
operator+(const klVec3T<FLOAT>& left, const klVec3T<FLOAT>& right)
{
	return klVec3T<FLOAT>(left.v[0]+right.v[0], left.v[1]+right.v[1], left.v[2]+right.v[2]);
}


template<class FLOAT> inline const klVec3T<FLOAT>
operator-(const klVec3T<FLOAT>& left, const klVec3T<FLOAT>& right)
{
	return klVec3T<FLOAT>(left.v[0]-right.v[0], left.v[1]-right.v[1], left.v[2]-right.v[2]);
}


template<class FLOAT> inline void
klVec3T<FLOAT>::normalize()
{
	FLOAT invLen = getInvLength();
	*this *= invLen;
}


template<class FLOAT> inline void
klVec3T<FLOAT>::transform(const klMatrixT<FLOAT>& nM, const klVec3T<FLOAT>& nV)
{
	v[0] = nM[0] * nV[0] + nM[4] * nV[1] + nM[8] *  nV[2] + nM[12];
	v[1] = nM[1] * nV[0] + nM[5] * nV[1] + nM[9] *  nV[2] + nM[13];
	v[2] = nM[2] * nV[0] + nM[6] * nV[1] + nM[10] * nV[2] + nM[14];
}


template<class FLOAT> inline klVec3T<FLOAT>&
klVec3T<FLOAT>::multiplyComonentWise(const klVec3T<FLOAT>& nV0, const klVec3T<FLOAT>& nV1)
{
	v[0] = nV0.v[0] * nV1.v[0];
	v[1] = nV0.v[1] * nV1.v[1];
	v[2] = nV0.v[2] * nV1.v[2];

    return *this;
}



template<class FLOAT> inline FLOAT
klVec3T<FLOAT>::dot(const klVec3T<FLOAT>& nOther)
{
	return v[0]*nOther.v[0] + v[1]*nOther.v[1] + v[2]*nOther.v[2];
}


template<class FLOAT> inline FLOAT
klVec3T<FLOAT>::getLength() const
{
	return FLOAT(sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]));
}


template<class FLOAT> inline FLOAT
klVec3T<FLOAT>::getInvLength() const
{
	FLOAT ret = getLength();
	ret.inverse();

	return ret;
}


//////////////////////////////////////////////
//
//  Generic Fixed-Point partitial specializations
//

// calculating the squared length does not work with normal fixed point
// since we have a maximum value of +/- 32768 that we can store in a fixed point
// variable. this value would be exceeded as soon as the length of the vector is
// larger than 181.
//

#ifdef _USE_FIXED_GENERIC_
template<> inline klFloat_fixed<class klFixedBase_generic>
klVec3T<klFloat_fixed<class klFixedBase_generic> >::getLength() const
{
	klFloat_fixed<class klFixedBase_generic> ret;

	__int64 v0 = v[0].getFixed(), v1 = v[1].getFixed(), v2 = v[2].getFixed();
	__int64 sqLen = (v0*v0+v1*v1+v2*v2)>>klFixedBase_generic::PBITS;
	ret.setDouble(::sqrt(sqLen/65536.0f));

	return ret;
}
#endif //_USE_FIXED_GENERIC_


//////////////////////////////////////////////
//
//  GPP optimized partitial specializations
//
#ifdef _USE_FIXED_GPP_


template<> inline klFloat_fixed<class klFixedBase_gpp>
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::getInvLength() const
{
	klFloat_fixed<class klFixedBase_gpp> ret;
	unsigned int sqLen, invLen;

	gppVec3DLengthSq_16_32s((GPP_VEC3D*)this, &sqLen);
	gppInvSqrtHP_16_32s(sqLen, &invLen);
	ret.setFixed(invLen);

	return ret;
}


template<> inline klFloat_fixed<class klFixedBase_gpp>
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::dot(const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& nOther)
{
	klFloat_fixed<class klFixedBase_gpp> ret;
	int i;

	gppVec3DDot_16_32s((GPP_VEC3D*)this, (GPP_VEC3D*)(&nOther), &i);
	ret.setFixed(i);

	return ret;
}


template<> inline klVec3T<klFloat_fixed<class klFixedBase_gpp> >&
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::operator*=(const klFloat_fixed<class klFixedBase_gpp>& nV)
{
	gppVec3DScale_16_32s((GPP_VEC3D*)this, nV.getFixed());
	return *this;
}


template<> inline void
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::normalize()
{
	// gppVec3DNormalize_16_32s works from GPP40 upwards, otherwise use the code below
	gppVec3DNormalize_16_32s((GPP_VEC3D*)this);

	//FLOAT invLen = getInvLength();
	//*this *= invLen;

}


template<> inline klFloat_fixed<class klFixedBase_gpp>
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::getLength() const
{
	klFloat_fixed<class klFixedBase_gpp> ret;
	unsigned int len;

	gppVec3DLength_16_32s((GPP_VEC3D*)this, &len);
	ret.setFixed(len);

	return ret;
}


template<> inline klVec3T<klFloat_fixed<class klFixedBase_gpp> >&
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::operator+=(const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& nO)
{
	gppVec3DAdd_n_32s((GPP_VEC3D*)this, (GPP_VEC3D*)(&nO), (GPP_VEC3D*)this);
	return *this;
}


template<> inline klVec3T<klFloat_fixed<class klFixedBase_gpp> >&
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::operator-=(const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& nO)
{
	gppVec3DSub_n_32s((GPP_VEC3D*)this, (GPP_VEC3D*)(&nO), (GPP_VEC3D*)this);
	return *this;
}


inline const klVec3T<klFloat_fixed<class klFixedBase_gpp> >
operator+(const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& left, const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& right)
{
	klVec3T<klFloat_fixed<class klFixedBase_gpp> > ret;

	gppVec3DAdd_n_32s((GPP_VEC3D*)(&left), (GPP_VEC3D*)(&right), (GPP_VEC3D*)(&ret));
	return ret;
}


inline const klVec3T<klFloat_fixed<class klFixedBase_gpp> >
operator-(const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& left, const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& right)
{
	klVec3T<klFloat_fixed<class klFixedBase_gpp> > ret;

	gppVec3DSub_n_32s((GPP_VEC3D*)(&left), (GPP_VEC3D*)(&right), (GPP_VEC3D*)(&ret));
	return ret;
}


template<> inline void
klVec3T<klFloat_fixed<class klFixedBase_gpp> >::transform(const klMatrixT<klFloat_fixed<class klFixedBase_gpp> >& nM, const klVec3T<klFloat_fixed<class klFixedBase_gpp> >& nV)
{
	int tmpV[4];

	gppVec3DTransform_16_32s((GPP_VEC3D*)(&nV), (GPP_VEC4D*)tmpV, (int*)&nM);
	v[0].setFixed(tmpV[0]);
	v[1].setFixed(tmpV[1]);
	v[2].setFixed(tmpV[2]);
}


#endif //_USE_FIXED_GPP_


#endif //__KLVEC3T_HEADERFILE__
