 /**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Id: klPtrList.h,v 1.3 2004/01/13 15:40:57 drgoldie Exp $
 *
\**************************************************************************/

 
#ifndef __KLPTRLIST_HEADERFILE__
#define __KLPTRLIST_HEADERFILE__

#ifdef _IS_WINDOWS_
#pragma warning( disable: 4311 4312 4514)
#endif

/// Template pointer list

#include <string.h>

#ifndef NULL
#define NULL    0
#endif


template <class TYPE>
class klPtrList
{
public:  
	 klPtrList();
	 klPtrList(const klPtrList<TYPE> &other);
	 virtual 
		~klPtrList();


	/// Sets how many elements the list grows when growing is necessary
	bool		
		setStandardGrowingSize(int size) const;


	/// Returns the number of objects currently stored in the list
	int			
		getSize() const { return actualItemCount;};


	/// Returns true if the list is empty
	bool		
		isEmpty() const { return actualItemCount==0;};


	/// Checks if the index is valid
	bool		
		checkIndex(int index) const;


	/// Adds one element to the head of the list (SLOW)
	virtual bool		
		addHead(TYPE* item);


	/// Adds one element to the tail of the list
	virtual bool		
		addTail(TYPE* item);


	/// Inserts one element after the given index
	virtual bool		
		insertAfter(int index,TYPE* item);


	/// Inserts one element before the given index
	virtual bool		
		insertBefore(int index,TYPE* item);


	/// Overwrites one element
	virtual bool		
		setAt(int index,TYPE* item);


	/// Replaces the indices of the 2 items
	virtual bool		
		replaceIndices(int nIndexA,int nIndexB);


	/// Replaces the first found element which is equal to 'old'  with 'item'
	virtual bool		
		replace(TYPE* old,TYPE* item);


	/// Replaces all elements which are equal to 'old' with 'item'
	virtual bool		
		replaceAll(TYPE* old,TYPE* item);


	/// Returns the first element of the list
	TYPE*		
		getHead();


	/// Returns the last element of the list
	TYPE*		
		getTail();


	/// Returns the element at the given position
	TYPE*		
		getAt(int index) const;


	/// Removing the first element of the list (SLOW)
	bool		
		removeHead();


	/// Removing the last element of the list
	bool		
		removeTail();


	/// Removes the first element 'item'
	bool		
		removeItem(TYPE* item);


	/// Removes the element a position 'index'
	virtual bool		
		removeIndex(int index);


	/// Empties the list.
	virtual bool		
		removeAll();


	/// Finds the first element which is equal to 'item'
	int			
		find(const TYPE* item);


	/// Finds the first element which is equal to 'item' and positioned after 'after'
	int			
		findAfter(const TYPE* item,int after);


	/// Finds the first element which is equal to 'item' and positioned before 'before'
	int			
		findBefore(const TYPE* item,int before);


	/// Returns a part of the list
	virtual klPtrList*	
		getSubList(int from,int to);


	/// Gives direct access to the list members
    TYPE* operator[](int index) const 
		{ return getAt(index); }


	/// Copies a list from another list
    virtual inline klPtrList<TYPE>&
		operator=(const klPtrList<TYPE>& other);


	/// Adds the elements of another list to this list
    virtual inline klPtrList<TYPE>&
		operator+=(const klPtrList<TYPE>& other);

protected:
	static int	standardGrowingSize;
	int			actualBlockSize;
	int			actualItemCount;
	TYPE**		array;

	bool		enlargeList(int size=-1);
	bool		reserveBlockSize(int size);
};

template <class TYPE>
 int klPtrList<TYPE>::standardGrowingSize = 16;


template <class TYPE> inline
klPtrList<TYPE>::klPtrList<TYPE>()
{
	array				= NULL;
	actualBlockSize	= 0;
	actualItemCount	= 0;

	enlargeList();
}

template <class TYPE> inline
klPtrList<TYPE>::klPtrList<TYPE>(const klPtrList<TYPE> &other)
{
	//klPtrList<TYPE>();
	array				= NULL;
	actualBlockSize	= 0;
	actualItemCount	= 0;

	enlargeList();

	*this = other;
}

template <class TYPE> inline
klPtrList<TYPE>::~klPtrList()
{
    // bool ret = 
    removeAll();

	if (array!=NULL)
		delete []array;
	array = NULL;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::enlargeList(int size)
{
	TYPE** tempList;
	if (size == -1)
		size = standardGrowingSize;

	tempList = new TYPE*[actualBlockSize + size];
	int i;
	for (i=0; i < actualBlockSize;++i)
		tempList[i] = array[i];
	for (; i < ( actualBlockSize+size );++i)
		tempList[i] = NULL;
	if (array!=NULL)
		delete array;
	array = tempList;
	actualBlockSize += size;
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::setStandardGrowingSize(int size) const
{
	if (size<1)
		return false;
	standardGrowingSize = size;
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::reserveBlockSize(int size)
{
	if (actualBlockSize < actualItemCount+size)
		return enlargeList();
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::checkIndex(int index) const
{
	if (index<0 || index > actualItemCount-1)
		return false;
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::addHead(TYPE* item)
{
	if (!reserveBlockSize(1))
		return false;

	memmove((void*)(reinterpret_cast<unsigned>(array)+sizeof(TYPE*)),
			array,actualItemCount*sizeof(TYPE*));
	array[0] = item;
	++actualItemCount;

	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::addTail(TYPE* item)
{
	if (!reserveBlockSize(1))
		return false;
	array[actualItemCount] = item;
	++actualItemCount;
		
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::insertAfter(int index,TYPE* item)
{
	if (!checkIndex(index))
		return false;
	if (index == actualItemCount-1)
		return addTail(item);
	else
	{
		if (!reserveBlockSize(1))
			return false;

		memmove((void*)(reinterpret_cast<unsigned>(array)+(index+2)*sizeof(TYPE*)),
				(void*)(reinterpret_cast<unsigned>(array)+(index+1)*sizeof(TYPE*)),
						(actualItemCount-index-1)*sizeof(TYPE*));
		array[index+1] = item;
		++actualItemCount;
	}
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::insertBefore(int index,TYPE* item)
{
	if (!checkIndex(index))
		return false;
	if (index == 0)
		return addHead(item);
	else
	{
		if (!reserveBlockSize(1))
			return false;

		memmove((void*)(reinterpret_cast<unsigned>(array)+(index+1)*sizeof(TYPE*)),
				(void*)(reinterpret_cast<unsigned>(array)+(index)*sizeof(TYPE*)),
						(actualItemCount-index)*sizeof(TYPE*));
		array[index] = item;
		++actualItemCount;
	}
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::setAt(int index,TYPE* item)
{
	if (!checkIndex(index))
		return false;

	array[index] = item;
	return true;
}

template <class TYPE> inline bool		
klPtrList<TYPE>::replaceIndices(int nIndexA,int nIndexB)
{
	if (!checkIndex(nIndexA))
		return false;
	if (!checkIndex(nIndexB))
		return false;
	TYPE *item = array[nIndexA];
	array[nIndexA] = array[nIndexB];
	array[nIndexB] = item;
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::replace(TYPE* old,TYPE* item)
{
	bool ret = false;
	for (int i=0; i < actualItemCount; ++i)
		if (array[i] == old)
		{
			array[i] = item;
			ret = true;
			break;
		}
	return ret;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::replaceAll(TYPE* old,TYPE* item)
{
	bool ret = false;
	for (int i=0; i < actualItemCount; ++i)
		if (array[i] == old)
		{
			array[i] = item;
			ret = true;
		}
	return ret;
}

template <class TYPE> inline TYPE* 
klPtrList<TYPE>::getHead()
{
	TYPE* ret = NULL;
	if (actualItemCount>0)
	{
		return NULL;
	}

	ret = array[0];
	return ret;
}

template <class TYPE> inline TYPE* 
klPtrList<TYPE>::getTail()
{
	TYPE* ret = NULL;
	if (actualItemCount<1)
		return NULL;

	ret = array[actualItemCount-1];
	return ret;
}

template <class TYPE> inline TYPE* 
klPtrList<TYPE>::getAt(int index) const
{
	TYPE* ret = NULL;
	if (!checkIndex(index))
	{
		return NULL;
	}

	ret = array[index];
	return ret;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::removeHead()
{
	return removeIndex(0);
}

template <class TYPE> inline bool 
klPtrList<TYPE>::removeTail()
{
	return removeIndex(actualItemCount-1);
}

template <class TYPE> inline bool 
klPtrList<TYPE>::removeItem(TYPE* item)
{
	int index;
	index = find(item);
	if (index < 0)
		return false;
	return removeIndex(index);
}

template <class TYPE> inline bool 
klPtrList<TYPE>::removeIndex(int index)
{
	if (!checkIndex(index))
		return false;
	memmove((void*)(reinterpret_cast<unsigned>(array)+(index)*sizeof(TYPE*)),
			(void*)(reinterpret_cast<unsigned>(array)+(index+1)*sizeof(TYPE*)),
					(actualItemCount-index-1)*sizeof(TYPE*));
	--actualItemCount;
	return true;
}

template <class TYPE> inline bool 
klPtrList<TYPE>::removeAll()
{
	actualItemCount = 0;
	return true;
}

template <class TYPE> inline int 
klPtrList<TYPE>::find(const TYPE* item)
{
	int ret = -1;

	for (int i=0; i < actualItemCount; ++i)
		if (item == array[i])
		{
			ret = i;
			break;
		}
	return ret;
}

template <class TYPE> inline int 
klPtrList<TYPE>::findAfter(const TYPE* item,int after)
{
	int ret = -1;
	int index = after+1;
	if (!checkIndex(index))
		return false;

	for (int i=index; i < actualItemCount; ++i)
		if (item == array[i])
		{
			ret = i;
			break;
		}
	return ret;
}

template <class TYPE> inline int 
klPtrList<TYPE>::findBefore(const TYPE* item,int before)
{
	int ret = -1;
	int index = before-1;
	if (!checkIndex(index))
		return false;

	for (int i=0; i < before; ++i)
		if (item == array[i])
		{
			ret = i;
			break;
		}
	return ret;
}

template <class TYPE> inline klPtrList<TYPE>* 
klPtrList<TYPE>::getSubList(int from,int to)
{
	if (!checkIndex(from))
	{
		return NULL;
	}
	if (!checkIndex(to))
	{
		return NULL;
	}
	if (from < to)
	{
		return NULL;
	}
	klPtrList<TYPE> *ret = new klPtrList<TYPE>;

	for (int i=from; i < to; ++i)
		ret->addTail(array[i]);
	return ret;
}

template <class TYPE> inline klPtrList<TYPE>&
klPtrList<TYPE>::operator=(const klPtrList<TYPE>& other)
{
	removeAll();
	for (int i=0;i<other.getSize();++i)
		addTail(other[i]);
	return *this;
}

template <class TYPE> inline klPtrList<TYPE>&
klPtrList<TYPE>::operator+=(const klPtrList<TYPE>& other)
{
	for (int i=0;i<other.getSize();++i)
		addTail(other[i]);
	return *this;
}


#endif //__KLPTRLIST_HEADERFILE__
