/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/Base/klMatrix.h,v 1.3 2004/01/13 14:40:28 tuhtah Exp $
 *
\**************************************************************************/


#ifndef __KLMATRIXT_HEADERFILE__
#define __KLMATRIXT_HEADERFILE__

#ifdef _IS_WINDOWS_
  #include <FLOAT.h>
  #include <memory.h>
#else
  #include <string.h>
#endif // _IS_WINDOWS_

// Matrix implementation with templated basic data type
//
// FLOAT specifies the basic data type
// see klFloat_fixed for a sample implementation
//
template<class FLOAT>
class klMatrixT
{
public:
	klMatrixT()												{}
	klMatrixT(const klMatrixT& nOther)						{  memcpy(v, nOther.v, 16*sizeof(float));  }

	void setFixed(const int* nV)		{  for(int i=0; i<16; i++)  v[i].setFixed(nV[i]);  }
	void setInt(const int* nV)			{  for(int i=0; i<16; i++)  v[i].setInt(nV[i]);  }
	void setFloat(const float* nV)		{  for(int i=0; i<16; i++)  v[i].setFloat(nV[i]);  }
	void setDouble(const double* nV)	{  for(int i=0; i<16; i++)  v[i].setDouble(nV[i]);  }

	int getFixed(int nIdx) const		{  return v[nIdx].getFixed();  }
	int getInt(int nIdx) const			{  return v[nIdx].getInt();  }
	float getFloat(int nIdx) const		{  return v[nIdx].getFloat();  }
	double getDouble(int nIdx) const	{  return v[nIdx].getDouble();  }

	void getFloat(float* nV) const		{  for(int i=0; i<16; i++)  nV[i] = v[i].getFloat();  }

	void makeIdent();
	void makeRotate(const FLOAT& nAngle, const FLOAT& x, const FLOAT& y, const FLOAT& z);
	void makeOrtho(const FLOAT& nLeft, const FLOAT& nRight,
				   const FLOAT& nBottom, const FLOAT& nTop,
				   const FLOAT& nNearval, const FLOAT& nFarval);
	void makeFrustum(const FLOAT& nLeft, const FLOAT& nRight, const FLOAT& nBottom,
					 const FLOAT& nTop, const FLOAT& nNearval, const FLOAT& nFarval);

	bool invert(const klMatrixT& nOther);
	bool invert()  {  klMatrixT cpy(*this);  return invert(cpy);  }

	void transpose(const klMatrixT& nOther);
	void transpose()  {  klMatrixT cpy(*this);  transpose(cpy);  }

	void rotate(const FLOAT& nAngle, const FLOAT& x, const FLOAT& y, const FLOAT& z);
	void translate(const FLOAT& nX, const FLOAT& nY, const FLOAT& nZ);
	void scale(const FLOAT& nX, const FLOAT& nY, const FLOAT& nZ);

	// overloaded operators
	//
	const FLOAT& operator[](int nIdx) const  {  return v[nIdx];  }
	FLOAT& operator[](int nIdx)  {  return v[nIdx];  }

	klMatrixT& operator=(const klMatrixT& nOther)  {  memcpy(v, nOther.v, 16*sizeof(FLOAT));  return *this;  }

	klMatrixT& operator*=(const klMatrixT& nOther)	{  multiply(*this, nOther);  return *this;  }
	inline void multiply(const klMatrixT& nLeft, const klMatrixT& nRight);

protected:
	FLOAT&	getAt(int nRow, int nCol)		{  return v[nRow+(nCol<<2)];  }
	FLOAT	getAt(int nRow, int nCol) const	{  return v[nRow+(nCol<<2)];  }

	FLOAT v[16];
};


//
// TODO: provide GPP optimized version (will not help a lot since usually we don't do a lot of matrix multiplies...)
//
template<class FLOAT> inline void
klMatrixT<FLOAT>::multiply(const klMatrixT& nLeft, const klMatrixT& nRight)
{
	for(int i = 0; i < 4; i++) {
		FLOAT	f0=nLeft.getAt(i,0),
				f1=nLeft.getAt(i,1),
				f2=nLeft.getAt(i,2),
				f3=nLeft.getAt(i,3);

		getAt(i,0) = f0 * nRight.getAt(0,0) + f1 * nRight.getAt(1,0) + f2 * nRight.getAt(2,0) + f3 * nRight.getAt(3,0);
		getAt(i,1) = f0 * nRight.getAt(0,1) + f1 * nRight.getAt(1,1) + f2 * nRight.getAt(2,1) + f3 * nRight.getAt(3,1);
		getAt(i,2) = f0 * nRight.getAt(0,2) + f1 * nRight.getAt(1,2) + f2 * nRight.getAt(2,2) + f3 * nRight.getAt(3,2);
		getAt(i,3) = f0 * nRight.getAt(0,3) + f1 * nRight.getAt(1,3) + f2 * nRight.getAt(2,3) + f3 * nRight.getAt(3,3);
   }
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::makeIdent()
{
	v[0]  = v[5]  = v[10] = v[15] = 1.0f;
	v[1]  = v[2]  = v[3]  = v[4]  = v[6]  = v[7]  = v[8]  = v[9]  = v[11] = v[12] = v[13] = v[14] = 0.0f;
}


template<class FLOAT> inline bool
klMatrixT<FLOAT>::invert(const klMatrixT& nOther)
{
   FLOAT	pos, neg, t, det;

   // calculate the determinant of upper left 3x3 submatrix and
   // determine if the matrix is singular.
   pos = neg = 0.0f;
   t =  nOther.getAt(0,0) * nOther.getAt(1,1) * nOther.getAt(2,2);
   if(t >= 0.0f) pos += t; else neg += t;

   t =  nOther.getAt(1,0) * nOther.getAt(2,1) * nOther.getAt(0,2);
   if(t >= 0.0f) pos += t; else neg += t;

   t =  nOther.getAt(2,0) * nOther.getAt(0,1) * nOther.getAt(1,2);
   if(t >= 0.0f) pos += t; else neg += t;

   t = -nOther.getAt(2,0) * nOther.getAt(1,1) * nOther.getAt(0,2);
   if(t >= 0.0f) pos += t; else neg += t;

   t = -nOther.getAt(1,0) * nOther.getAt(0,1) * nOther.getAt(2,2);
   if(t >= 0.0f) pos += t; else neg += t;

   t = -nOther.getAt(0,0) * nOther.getAt(2,1) * nOther.getAt(1,2);
   if(t >= 0.0f) pos += t; else neg += t;

   det = pos + neg;

   if (det*det < 0.001f)
      return false;

   det.inverse();

   // start with an identity matrix
   makeIdent();

   // do rotation/scaling part
   getAt(0,0) = (  (nOther.getAt(1,1)*nOther.getAt(2,2) - nOther.getAt(2,1)*nOther.getAt(1,2) )*det);
   getAt(0,1) = (- (nOther.getAt(0,1)*nOther.getAt(2,2) - nOther.getAt(2,1)*nOther.getAt(0,2) )*det);
   getAt(0,2) = (  (nOther.getAt(0,1)*nOther.getAt(1,2) - nOther.getAt(1,1)*nOther.getAt(0,2) )*det);
   getAt(1,0) = (- (nOther.getAt(1,0)*nOther.getAt(2,2) - nOther.getAt(2,0)*nOther.getAt(1,2) )*det);
   getAt(1,1) = (  (nOther.getAt(0,0)*nOther.getAt(2,2) - nOther.getAt(2,0)*nOther.getAt(0,2) )*det);
   getAt(1,2) = (- (nOther.getAt(0,0)*nOther.getAt(1,2) - nOther.getAt(1,0)*nOther.getAt(0,2) )*det);
   getAt(2,0) = (  (nOther.getAt(1,0)*nOther.getAt(2,1) - nOther.getAt(2,0)*nOther.getAt(1,1) )*det);
   getAt(2,1) = (- (nOther.getAt(0,0)*nOther.getAt(2,1) - nOther.getAt(2,0)*nOther.getAt(0,1) )*det);
   getAt(2,2) = (  (nOther.getAt(0,0)*nOther.getAt(1,1) - nOther.getAt(1,0)*nOther.getAt(0,1) )*det);

   // do translation part
   getAt(0,3) = - (nOther.getAt(0,3) * getAt(0,0) + nOther.getAt(1,3) * getAt(0,1) + nOther.getAt(2,3) * getAt(0,2) );
   getAt(1,3) = - (nOther.getAt(0,3) * getAt(1,0) + nOther.getAt(1,3) * getAt(1,1) + nOther.getAt(2,3) * getAt(1,2) );
   getAt(2,3) = - (nOther.getAt(0,3) * getAt(2,0) + nOther.getAt(1,3) * getAt(2,1) + nOther.getAt(2,3) * getAt(2,2) );

   return true;
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::transpose(const klMatrixT& nOther)
{
	v[0] = nOther.v[0];
	v[1] = nOther.v[4];
	v[2] = nOther.v[8];
	v[3] = nOther.v[12];
	v[4] = nOther.v[1];
	v[5] = nOther.v[5];
	v[6] = nOther.v[9];
	v[7] = nOther.v[13];
	v[8] = nOther.v[2];
	v[9] = nOther.v[6];
	v[10] = nOther.v[10];
	v[11] = nOther.v[14];
	v[12] = nOther.v[3];
	v[13] = nOther.v[7];
	v[14] = nOther.v[11];
	v[15] = nOther.v[15];
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::makeRotate(const FLOAT& nAngle, const FLOAT& nX, const FLOAT& nY, const FLOAT& nZ)
{
	FLOAT	xx, yy, zz, xy, yz, zx, xs, ys, zs, one_c,
			x = nX, y = nY, z = nZ, angle = nAngle*0.01745329f,
			s = sin(angle), c = cos(angle);

	makeIdent();

	if(x!=0 && y==0 && z==0)		// rotation only around x-axis
	{
		getAt(1,1) = c;
		getAt(2,2) = c;
		if (y < 0.0F)
		{
	        getAt(1,2) = s;
			getAt(2,1) = -s;
		}
		else
		{
			getAt(1,2) = -s;
			getAt(2,1) = s;
		}
	}
	else if(x==0 && y!=0 && z==0)		// rotation only around y-axis
	{			
		getAt(0,0) = c;
		getAt(2,2) = c;
		if (y < 0.0F)
		{
			getAt(0,2) = -s;
			getAt(2,0) = s;
		}
		else
		{
			getAt(0,2) = s;
			getAt(2,0) = -s;
		}
	}
    else if(x==0 && y==0 && z!=0)		// rotation only around z-axis
	{
		getAt(0,0) = c;
		getAt(1,1) = c;
		if (z < 0.0F)
		{
			getAt(0,1) = s;
			getAt(1,0) = -s;
		}
		else
		{
			getAt(0,1) = -s;
			getAt(1,0) = s;
		}
	}
	else								// otherweise do full rotation matrix
	{
		FLOAT len = sqrt(x * x + y * y + z * z);

		if(len <= 0.001f)				// too small axis ?
			return;

		len.inverse();
		x *= len;
		y *= len;
		z *= len;

		xx = x * x;
		yy = y * y;
		zz = z * z;
		xy = x * y;
		yz = y * z;
		zx = z * x;
		xs = x * s;
		ys = y * s;
		zs = z * s;
		one_c = 1 - c;

		getAt(0,0) = (one_c * xx) + c;
		getAt(0,1) = (one_c * xy) - zs;
		getAt(0,2) = (one_c * zx) + ys;
		getAt(1,0) = (one_c * xy) + zs;
		getAt(1,1) = (one_c * yy) + c;
		getAt(1,2) = (one_c * yz) - xs;
		getAt(2,0) = (one_c * zx) - ys;
		getAt(2,1) = (one_c * yz) + xs;
		getAt(2,2) = (one_c * zz) + c;
	}
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::rotate(const FLOAT& nAngle, const FLOAT& x, const FLOAT& y, const FLOAT& z)
{
	klMatrixT other;

	other.makeRotate(nAngle, x, y, z);
	(*this) *= other;
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::translate(const FLOAT& nX, const FLOAT& nY, const FLOAT& nZ)
{
	v[12] = v[0] * nX + v[4] * nY + v[8]  * nZ + v[12];
	v[13] = v[1] * nX + v[5] * nY + v[9]  * nZ + v[13];
	v[14] = v[2] * nX + v[6] * nY + v[10] * nZ + v[14];
	v[15] = v[3] * nX + v[7] * nY + v[11] * nZ + v[15];
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::scale(const FLOAT& nX, const FLOAT& nY, const FLOAT& nZ)
{
	v[0] *= nX;   v[4] *= nY;   v[8]  *= nZ;
	v[1] *= nX;   v[5] *= nY;   v[9]  *= nZ;
	v[2] *= nX;   v[6] *= nY;   v[10] *= nZ;
	v[3] *= nX;   v[7] *= nY;   v[11] *= nZ;
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::makeOrtho(const FLOAT& nLeft, const FLOAT& nRight,
						   const FLOAT& nBottom, const FLOAT& nTop,
						   const FLOAT& nNearval, const FLOAT& nFarval)
{
	float	left=nLeft.getFloat(), right=nRight.getFloat(), bottom=nBottom.getFloat(),
			top=nTop.getFloat(), nearval=nNearval.getFloat(), farval=nFarval.getFloat();
	float	x, y, z;
	float	tx, ty, tz;

	x = 2.0f / (right-left);
	y = 2.0f / (top-bottom);
	z = -2.0f / (farval-nearval);
	tx = -(right+left) / (right-left);
	ty = -(top+bottom) / (top-bottom);
	tz = -(farval+nearval) / (farval-nearval);

	getAt(0,0) = x;     getAt(0,1) = 0.0F;  getAt(0,2) = 0.0F;  getAt(0,3) = tx;
	getAt(1,0) = 0.0F;  getAt(1,1) = y;     getAt(1,2) = 0.0F;  getAt(1,3) = ty;
	getAt(2,0) = 0.0F;  getAt(2,1) = 0.0F;  getAt(2,2) = z;     getAt(2,3) = tz;
	getAt(3,0) = 0.0F;  getAt(3,1) = 0.0F;  getAt(3,2) = 0.0F;  getAt(3,3) = 1.0F;
}


template<class FLOAT> inline void
klMatrixT<FLOAT>::makeFrustum(const FLOAT& nLeft, const FLOAT& nRight, const FLOAT& nBottom,
							 const FLOAT& nTop, const FLOAT& nNearval, const FLOAT& nFarval)
{
	float	left=nLeft.getFloat(), right=nRight.getFloat(), bottom=nBottom.getFloat(),
			top=nTop.getFloat(), nearval=nNearval.getFloat(), farval=nFarval.getFloat();
	float x, y, a, b, c, d;

	x = (2.0f * nearval) / (right - left);
	y = (2.0f * nearval) / (top - bottom);
	a = (right + left) / (right - left);
	b = (top + bottom) / (top - bottom);
	c = -(farval + nearval) / ( farval - nearval);
	d = -(2.0f * farval * nearval) / (farval - nearval);

	getAt(0,0) = x;     getAt(0,1) = 0.0f;  getAt(0,2) = a;      getAt(0,3) = 0.0f;
	getAt(1,0) = 0.0f;  getAt(1,1) = y;     getAt(1,2) = b;      getAt(1,3) = 0.0f;
	getAt(2,0) = 0.0f;  getAt(2,1) = 0.0f;  getAt(2,2) = c;      getAt(2,3) = d;
	getAt(3,0) = 0.0f;  getAt(3,1) = 0.0f;  getAt(3,2) = -1.0f;  getAt(3,3) = 0.0f;
} 


#endif //__KLMATRIXT_HEADERFILE__
