 /**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Id: klList.h,v 1.3 2004/01/13 15:40:57 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLLIST_HEADERFILE__
#define __KLLIST_HEADERFILE__

#include <string.h>

#ifdef _IS_WINDOWS_
#pragma warning( disable: 4514 )
#endif

/// klList is a template class for storing simple objects directly in the list (no pointers)
/**
 *  See also: klPtrList
 */
template <class TYPE>
class klList
{
public:   
	klList();
	klList(const klList<TYPE> &other);

	 virtual 
		~klList();

	/// Returns the number of objects currently stored in the list
	int			
		getSize() const  { return actualItemCount; }


	/// Returns true if the list is empty
	bool		
		isEmpty() const  { return actualItemCount==0; }


	/// Checks if the index is valid
	bool		
		checkIndex(int index) const;


	/// Sets how many elements the list grows when growing is done
	void
		setGrowingSize(int size)  { growingSize = size; }


	/// Lets the list grow 'size' elemens
	bool
		enlargeList(int size=-1);


	/// Enlarge the lists by a given size and set the lists size
	bool
		enlargeListSet(int size=-1);


	/// Enlarges/Reduces the list to a given size.
	int			
		setSize(int nSize);


	/// Adds one element to the head of the list (SLOW)
	virtual int		
		addHead(const TYPE& item);


	/// Adds one element to the tail of the list
	virtual int		
		addTail(const TYPE& item);


	/// Adds the last element of the list to the list again
	virtual int		
		addTailAgain();


	/// Adds 'nSize' elements to the tail of the list
	virtual int		
		addTail(const TYPE* item,int nSize);


	/// Inserts one element after the given index (SLOW)
	virtual int		
		insertAfter(int index,const TYPE& item);


	/// Inserts one element before the given index (SLOW)
	virtual int		
		insertBefore(int index,const TYPE& item);


	/// Takes 'nSize' elements starting at the given address and copies them into the list
	virtual bool		
		setList(const TYPE* item,int nSize);


	/// Overwrites one element
	virtual bool		
		setAt(int index,const TYPE& item);


	/// Replaces the indices of the 2 items
	virtual bool		
		replaceIndices(int nIndexA,int nIndexB);


	/// Replaces the first found element which is equal to 'old'  with 'item'
	virtual bool		
		replace(const TYPE& old,const TYPE& item);


	/// Replaces all elements which are equal to 'old' with 'item'
	virtual bool		
		replaceAll(const TYPE& old,const TYPE& item);


	/// Returns the first element of the list
	/**
	 *  Returns false if the list is empty
	 */
	bool		
		getHead(TYPE& item) const;


	/// Returns the first element of the list
	TYPE&
		getHead() const;


	/// Returns the last element of the list
	/**
	 *  Returns false if the list is empty
	 */
	bool		
		getTail(TYPE& item) const;


	/// Returns the last element of the list
	TYPE&
		getTail() const;


	/// Returns the element at the given position
	/**
	 *  Returns false if the list is empty
	 */
	bool		
		getAt(int index,TYPE& item) const;


	/// Returns the element at the given position
	TYPE&
		getAt(int index) const;


	/// Removing the first element of the list (SLOW)
	bool		
		removeHead();


	/// Removing the last element of the list
	bool		
		removeTail();


	/// Removes the first element in the list which is equal to 'item'
	bool
		removeItem(const TYPE& item);


	/// Removes the element a position 'index'
	virtual bool
		removeIndex(int index);


	/// Empties the list.
	virtual bool		
		removeAll();


	/// Finds the first element which is equal to 'item'
	int			
		find(const TYPE& item);


	/// Finds the first element which is equal to 'item' and positioned after 'after'
	int			
		findAfter(const TYPE& item,int after);


	/// Finds the first element which is equal to 'item' and positioned before 'before'
	int			
		findBefore(const TYPE& item,int before);


	/// Returns a part of the list
	virtual klList*	
		getSubList(int from,int to);

	/// Gives direct access to the list members
    TYPE& 
		operator[](unsigned index) const { return getAt(index); }


	/// Copies a list from another list
    inline klList<TYPE>&
		operator=(const klList<TYPE>& other);


	/// Compares two lists
	/**
	 *  Each item is compared. (therefore TYPE must have an operator==).
	 *  If all items are ident, true is returned.
	 */
	friend inline bool 
		operator==(const klList&, const klList&)  { return false; }


protected:
	int			growingSize;
	int			actualBlockSize;
	int			actualItemCount;
	TYPE*		array;

	bool		reserveBlockSize(int size);
};


template <class TYPE> inline
klList<TYPE>::klList<TYPE>()
{
	array			= NULL;
	actualBlockSize	= 0;
	actualItemCount	= 0;
	growingSize		= 16;

	enlargeList();
}

template <class TYPE> inline
klList<TYPE>::klList<TYPE>(const klList<TYPE> &other)
{
	klList<TYPE>();
	*this = other;
}

template <class TYPE> inline
klList<TYPE>::~klList()
{
    // bool ret =
    removeAll();

	if(array)
		delete [] array;
}

template <class TYPE> inline bool 
klList<TYPE>::enlargeList(int size)
{
	TYPE* tempList;
	if (size == -1)
		size = growingSize;

	tempList = new TYPE[actualBlockSize + size];
//		memcpy(tempList,array,actualBlockSize*sizeof(TYPE));
	for (int i=0; i < actualItemCount;++i)
		tempList[i] = array[i];
	//for (; i < ( actualBlockSize+size );++i)
	//	tempList[i] = NULL;
	if (array!=NULL)
		delete [] array;
	array = tempList;

	actualBlockSize += size;

	return true;
}

template <class TYPE> inline bool 
klList<TYPE>::enlargeListSet(int size)
{
	if (enlargeList(size))
	{
		actualItemCount = actualBlockSize;
		return true;
	}
	return false;
}

template <class TYPE> inline int			
klList<TYPE>::setSize(int nSize)
{
	if(array!=NULL)
		delete [] array;
	int size;
	if (nSize<1)
		size = growingSize;
	else
		size = (nSize-(nSize%growingSize)) + growingSize;
	array = new TYPE[size];
	actualItemCount = nSize;
	return actualItemCount;
}

template <class TYPE> inline bool 
klList<TYPE>::reserveBlockSize(int size)
{
	if (actualBlockSize < actualItemCount+size)
		return enlargeList();
	return true;
}

template <class TYPE> inline bool 
klList<TYPE>::checkIndex(int index) const
{
	if (index<0 || index > actualItemCount-1)
		return false;
	return true;
}

template <class TYPE> inline int
klList<TYPE>::addHead(const TYPE& item)
{
	if (!reserveBlockSize(1))
		return -1;

	memmove((void*)(reinterpret_cast<unsigned>(array)+sizeof(TYPE)),
			array,actualItemCount*sizeof(TYPE));
	array[0] = item;
	++actualItemCount;

	return 0;
}

template <class TYPE> inline int 
klList<TYPE>::addTail(const TYPE& item)
{ 
	if (!reserveBlockSize(1)) 
		return -1;

	array[actualItemCount] = item;
	++actualItemCount;
		
	return (actualItemCount-1);
}

template <class TYPE> inline int 
klList<TYPE>::addTail(const TYPE* item,int nSize)
{ 
	if (!reserveBlockSize(nSize)) 
		return -1;

	for (int i=0;i<nSize;++i)
		array[actualItemCount+i] = item[i];
	actualItemCount += nSize;

	return (actualItemCount-1);
}


template <class TYPE> inline int 
klList<TYPE>::addTailAgain()
{
	if(actualItemCount>0)
	{
		TYPE item = array[actualItemCount-1];		// need to copy since a list grow could screw the reference
		return addTail(item);
	}
	else
		return -1;
}


template <class TYPE> inline int 
klList<TYPE>::insertAfter(int index,const TYPE& item)
{
	if (!checkIndex(index))
	{
		return -1;
	}
	if (index == actualItemCount-1)
		return addTail(item);
	else
	{
		if (!reserveBlockSize(1))
			return -1;

		memmove((void*)(reinterpret_cast<unsigned>(array)+(index+2)*sizeof(TYPE)),
				(void*)(reinterpret_cast<unsigned>(array)+(index+1)*sizeof(TYPE)),
						(actualItemCount-index-1)*sizeof(TYPE));
		array[index+1] = item;
		++actualItemCount;
	}
	return (index+1);
}

template <class TYPE> inline int 
klList<TYPE>::insertBefore(int index,const TYPE& item)
{
	if (!checkIndex(index))
	{
		return -1;
	}
	if (index == 0)
		return addHead(item);
	else
	{
		if (!reserveBlockSize(1))
			return -1;

		memmove((void*)(reinterpret_cast<unsigned>(array)+(index+1)*sizeof(TYPE)),
				(void*)(reinterpret_cast<unsigned>(array)+(index)*sizeof(TYPE)),
						(actualItemCount-index)*sizeof(TYPE));
		array[index] = item;
		++actualItemCount;
	}
	return (index-1);
}

template <class TYPE> inline bool 
klList<TYPE>::setList(const TYPE* item,int nSize)
{
	if(array!=NULL)
		delete [] array;
	int size;
	if (nSize<1)
		size = growingSize;
	else
		size = (nSize-(nSize%growingSize)) + growingSize;
	array = new TYPE[size];
	memmove((void*)reinterpret_cast<unsigned>(array),
			(void*)reinterpret_cast<unsigned>(item),
					(nSize)*sizeof(TYPE));
	actualItemCount = nSize;
	return true;
}

template <class TYPE> inline bool 
klList<TYPE>::setAt(int index,const TYPE& item)
{
	if (!checkIndex(index))
		return false;

	array[index] = item;
	return true;
}

template <class TYPE> inline bool		
klList<TYPE>::replaceIndices(int nIndexA,int nIndexB)
{
	if (!checkIndex(nIndexA))
		return false;
	if (!checkIndex(nIndexB))
		return false;
	TYPE item = array[nIndexA];
	array[nIndexA] = array[nIndexB];
	array[nIndexB] = item;
	return true;
}

template <class TYPE> inline bool 
klList<TYPE>::replace(const TYPE& old,const TYPE& item)
{
	bool ret = false;
	for (int i=0; i < actualItemCount; ++i)
		if (array[i] == old)
		{
			array[i] = item;
			ret = true;
			break;
		}
	return ret;
}

template <class TYPE> inline bool 
klList<TYPE>::replaceAll(const TYPE& old,const TYPE& item)
{
	bool ret = false;
	for (int i=0; i < actualItemCount; ++i)
		if (array[i] == old)
		{
			array[i] = item;
			ret = true;
		}
	return ret;
}

template <class TYPE> inline bool 
klList<TYPE>::getHead(TYPE& item) const
{
	if (actualItemCount>0)
	{
		return false;
	}

	item = array[0];
	return true;
}

template <class TYPE> inline TYPE&
klList<TYPE>::getHead() const
{
	//if(actualItemCount<1)
	//	raiseError("Item not found");

	return array[0];
}

template <class TYPE> inline bool 
klList<TYPE>::getTail(TYPE& item) const
{
	if (actualItemCount<1)
	{
		return false;
	}

	item = array[actualItemCount-1];
	return true;
}

template <class TYPE> inline TYPE&
klList<TYPE>::getTail()  const
{
	//if(actualItemCount<1)
	//	raiseError("Item not found");

	return array[actualItemCount-1];
}

template <class TYPE> inline bool
klList<TYPE>::getAt(int index,TYPE& item) const
{
	if (!checkIndex(index))
	{
		return false;
	}

	item = array[index];
	return true;
}

template <class TYPE> inline TYPE&
klList<TYPE>::getAt(int index) const
{
	//if (!checkIndex(index))
	//	raiseError("IndexError");

	return array[index];
}

template <class TYPE> inline bool 
klList<TYPE>::removeHead()
{
	return removeIndex(0);
}

template <class TYPE> inline bool 
klList<TYPE>::removeTail()
{
	//return removeIndex(actualItemCount-1);
	if(actualItemCount>0)
	{
		actualItemCount--;
		return true;
	}
	else
		return false;
}

template <class TYPE> inline bool 
klList<TYPE>::removeItem(const TYPE& item)
{
	int index;
	index = find(item);
	if (index < 0)
		return false;
	return removeIndex(index);
}

template <class TYPE> inline bool 
klList<TYPE>::removeIndex(int index)
{
	if (!checkIndex(index))
		return false;

	memmove((void*)(reinterpret_cast<unsigned>(array)+(index)*sizeof(TYPE)),
			(void*)(reinterpret_cast<unsigned>(array)+(index+1)*sizeof(TYPE)),
					(actualItemCount-index-1)*sizeof(TYPE));
	--actualItemCount;
	return true;
}

template <class TYPE> inline bool 
klList<TYPE>::removeAll()
{
	actualItemCount = 0;
	return true;
}

template <class TYPE> inline int 
klList<TYPE>::find(const TYPE& item)
{
	int ret = -1;

	for (int i=0; i < actualItemCount; ++i)
		if (item == array[i])
		{
			ret = i;
			break;
		}
	return ret;
}

template <class TYPE> inline int 
klList<TYPE>::findAfter(const TYPE& item,int after)
{
	int ret = -1;
	int index = after+1;
	if (!checkIndex(index))
		return -1;

	for (int i=index; i < actualItemCount; ++i)
		if (item == array[i])
		{
			ret = i;
			break;
		}
	return ret;
}

template <class TYPE> inline int 
klList<TYPE>::findBefore(const TYPE& item,int before)
{
	int ret = -1;
	int index = before-1;
	if (!checkIndex(index))
		return -1;

	for (int i=0; i < before; ++i)
		if (item == array[i])
		{
			ret = i;
			break;
		}
	return ret;
}

template <class TYPE> inline klList<TYPE>* 
klList<TYPE>::getSubList(int from,int to)
{
	if (!checkIndex(from))
	{
		return NULL;
	}
	if (!checkIndex(to))
	{
		return NULL;
	}
	if (from < to)
	{
		return NULL;
	}
	klList<TYPE> *ret = new klList<TYPE>;

	for (int i=from; i < to; ++i)
		ret->addTail(array[i]);
	return ret;
}


template <class TYPE> inline klList<TYPE>&
klList<TYPE>::operator=(const klList<TYPE>& other)
{
	removeAll();
	if (array!=NULL)
		delete [] array;
	if (other.array == NULL || other.actualItemCount==0)
	{
		array = NULL;
		actualBlockSize = 0;
		actualItemCount = 0;
		enlargeList();
	}
	else
	{
		array = new TYPE[other.actualItemCount];
		memmove((void*)(reinterpret_cast<unsigned>(array)),
				(void*)(reinterpret_cast<unsigned>(other.array)),
						(other.actualItemCount)*sizeof(TYPE));
		actualItemCount = other.actualItemCount;
		actualBlockSize = actualItemCount;
	}
/*
	for (int i=0;i<other.GetSize();++i)
		AddTail(other[i]);
*/
	return *this;
}



#endif //__KLLIST_HEADERFILE__
