/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE


#include "juce_Uuid.h"


//==============================================================================
Uuid::Uuid()
{
    fromString (generateUuid());
}

Uuid::Uuid (const String& uuid)
{
    fromString (uuid);
}

Uuid::Uuid (const unsigned char* bytes)
{
    *this = bytes;
}

Uuid::Uuid (const Uuid& other)
{
    *this = other.uuidBytes;
}

void Uuid::fromString (const String& uuid)
{
    const String uuidRaw (uuid.removeCharacters (T("-")));

    const int64 uuidH = uuidRaw.substring (0,16).getHexValue64();
    const int64 uuidL = uuidRaw.substring (16,32).getHexValue64();

    for (int i = 0; i < 8; ++i)
    {
        uuidBytes [i]      = (unsigned char)((uuidH >> ((7 - i) * 8)) & 0xff);
        uuidBytes [i + 8]  = (unsigned char)((uuidL >> ((7 - i) * 8)) & 0xff);;
    }
}

Uuid& Uuid::operator= (const Uuid& other)
{
    if (this != &other)
        *this = other.uuidBytes;

    return *this;
}

Uuid& Uuid::operator= (const unsigned char* bytes)
{
    if (bytes)
        memcpy (uuidBytes, bytes, UUID_BYTES);
    else
        zeromem (uuidBytes, UUID_BYTES);

    return *this;
}

Uuid& Uuid::operator= (const String& uuid)
{
    fromString (uuid);
    return *this;
}

Uuid::operator const unsigned char*() const
{
    return uuidBytes;
}

void Uuid::copyToBytes (unsigned char* bytes) const
{
    memcpy (bytes, uuidBytes, UUID_BYTES);
}

const String Uuid::toString() const
{
    String uuidStr;

    uuidStr << String::toHexString (uuidBytes, 4, 0)
            << T("-") << String::toHexString (uuidBytes + 4, 2, 0)
            << T("-") << String::toHexString (uuidBytes + 6, 2, 0)
            << T("-") << String::toHexString (uuidBytes + 8, 2, 0)
            << T("-") << String::toHexString (uuidBytes + 10, 6, 0);

    return uuidStr;
}

bool Uuid::operator== (const Uuid& other) const
{
    return ! memcmp (uuidBytes, other.uuidBytes, UUID_BYTES);
}

bool Uuid::isNull() const
{
    for (int i = 0; i < UUID_BYTES; i++)
        if (uuidBytes[i])
            return false;

    return true;
}

bool Uuid::fromStream (InputStream& in)
{
    const int len = in.readByte();

    if (len == 0)
    {
        *this = 0;
    }
    else if (len == UUID_BYTES)
    {
        if (in.read (uuidBytes, UUID_BYTES) == UUID_BYTES)
            return true;
    }

    return false;
}

void Uuid::toStream (OutputStream& out) const
{
    if (isNull())
    {
        out.writeByte (0);
    }
    else
    {
        out.writeByte (UUID_BYTES);
        out.write (uuidBytes, UUID_BYTES);
    }
}

END_JUCE_NAMESPACE
