/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_SIMPLELISTBOX_JUCEHEADER__
#define __JUCE_SIMPLELISTBOX_JUCEHEADER__

#include "juce_ListBox.h"


//==============================================================================
/**
    A subclass of this is used to drive a SimpleListBox.

    @see SimpleListBox
*/
class JUCE_API  SimpleListBoxModel
{
public:
    //==============================================================================
    /** Destructor. */
    virtual ~SimpleListBoxModel()  {}

    //==============================================================================
    /** This has to return the number of items in the list.

        @see ListBox::getNumRows()
    */
    virtual int getNumRows() = 0;

    /** This method must be implemented to draw a row of the list.
    */
    virtual void paintListBoxItem (int rowNumber,
                                   Graphics& g,
                                   int width, int height,
                                   bool rowIsSelected) = 0;

    /** This can be overridden to react to the user clicking on a row.

        @see listBoxItemDoubleClicked
    */
    virtual void listBoxItemClicked (int row, const MouseEvent& e);

    /** This can be overridden to react to the user double-clicking on a row.

        @see listBoxItemClicked
    */
    virtual void listBoxItemDoubleClicked (int row, const MouseEvent& e);

    /** Override this to be informed when rows are selected or deselected.

        @see ListBox::selectedRowsChanged()
    */
    virtual void selectedRowsChanged (int lastRowselected);

    /** Override this to be informed when the delete key is pressed.

        @see ListBox::deleteKeyPressed()
    */
    virtual void deleteKeyPressed (int lastRowSelected);

    /** Override this to be informed when the return key is pressed.

        @see ListBox::returnKeyPressed()
    */
    virtual void returnKeyPressed (int lastRowSelected);

    /** To allow rows from your table to be dragged-and-dropped, implement this method.

        If this returns a non-empty name then when the user drags a row of the table,
        the listbox will try to find a DragAndDropContainer in its parent hierarchy, and
        will use it to trigger a drag-and-drop operation, using this string as the
        source description, and the listbox itself as the source component.

        @see DragAndDropContainer::startDragging
    */
    virtual const String getDragSourceDescription (const SparseSet<int>& selectedRows);
};


//==============================================================================
/**
    A simplified ListBox, suitable for most applications.

    The ListBox class is powerful, but a bit of a hassle because you need to
    create a special type of component for the rows, which has to intercept mouse
    clicks to select rows, etc.  The SimpleListBox lets you just define a method
    to paint the rows and gets told about mouse clicks and keypresses. It won't let
    you add subcomponents to a row, but for most purposes that's not a big problem.

    To use it, create a SimpleListBox and pass it a suitable subclass of SimpleListBoxModel
    to use to draw itself.

    For selecting rows, changing colours, etc., use the methods in the ListBox parent class.

    @see SimpleListBoxModel, ListBox
*/
class JUCE_API  SimpleListBox  : public ListBox
{
public:
    //==============================================================================
    /** Creates a SimpleListBox.

        @param componentName    the name to give the component
        @param model            the model to use - this may be null, and
                                can be changed later with the setModel() method
    */
    SimpleListBox (const String& componentName,
                   SimpleListBoxModel* const model);

    /** Destructor. */
    ~SimpleListBox();

    //==============================================================================
    /** Changes the current model being used to draw the list.

        The model can be set to null, which will just create an empty list.
    */
    void setModel (SimpleListBoxModel* const newModel);


    //==============================================================================
    /** @internal */
    int getNumRows();

    juce_UseDebuggingNewOperator

protected:
    /** @internal */
    Component* createRowComponent();
    /** @internal */
    void updateRowComponent (Component*, int, bool);
    /** @internal */
    void selectedRowsChanged (int);
    /** @internal */
    void returnKeyPressed (int);
    /** @internal */
    void deleteKeyPressed (int);

private:
    friend class SimpleListBoxRowComponent;
    SimpleListBoxModel* model;

    SimpleListBox (const SimpleListBox&);
    const SimpleListBox& operator= (const SimpleListBox&);
};


#endif   // __JUCE_SIMPLELISTBOX_JUCEHEADER__
