/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "win32_headers.h"

#ifdef _MSC_VER
  #pragma warning (disable: 4514)
  #pragma warning (push)
  #include <crtdbg.h>
#endif

#ifdef __BORLANDC__
  #define __MT__
#endif

#include <process.h>
#include "../../../src/juce_core/basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE

#include "../../../src/juce_core/text/juce_String.h"
#include "../../../src/juce_core/threads/juce_CriticalSection.h"
#include "../../../src/juce_core/threads/juce_WaitableEvent.h"
#include "../../../src/juce_core/threads/juce_Thread.h"
#include "../../../src/juce_core/threads/juce_Process.h"


extern HWND juce_messageWindowHandle;

#ifdef _MSC_VER
  #pragma warning (pop)
//  #pragma warning (disable : 4127)    // "Conditional expression is constant" warning
#endif


//==============================================================================
CriticalSection::CriticalSection() throw()
{
    // (just to check the MS haven't changed this structure and broken things...)
    static_jassert (sizeof (CRITICAL_SECTION) <= 24);

    InitializeCriticalSection ((CRITICAL_SECTION*) internal);
}

CriticalSection::~CriticalSection() throw()
{
    DeleteCriticalSection ((CRITICAL_SECTION*) internal);
}

void CriticalSection::enter() const throw()
{
    EnterCriticalSection ((CRITICAL_SECTION*) internal);
}

bool CriticalSection::tryEnter() const throw()
{
    return TryEnterCriticalSection ((CRITICAL_SECTION*) internal) != FALSE;
}

void CriticalSection::exit() const throw()
{
    LeaveCriticalSection ((CRITICAL_SECTION*) internal);
}

//==============================================================================
WaitableEvent::WaitableEvent() throw()
{
    internal = CreateEvent (0, FALSE, FALSE, 0);
}

WaitableEvent::~WaitableEvent() throw()
{
    CloseHandle (internal);
}

bool WaitableEvent::wait (const int timeOutMillisecs) const throw()
{
    return WaitForSingleObject (internal, timeOutMillisecs) == WAIT_OBJECT_0;
}

void WaitableEvent::signal() const throw()
{
    SetEvent (internal);
}

void WaitableEvent::reset() const throw()
{
    ResetEvent (internal);
}

//==============================================================================
void JUCE_API juce_threadEntryPoint (void*);

static unsigned int __stdcall threadEntryProc (void* userData)
{
    AttachThreadInput (GetWindowThreadProcessId (juce_messageWindowHandle, 0),
                       GetCurrentThreadId(),
                       TRUE);

    juce_threadEntryPoint (userData);

    _endthread();
    return 0;
}

void* juce_createThread (void* userData)
{
    unsigned int threadId;

    return (void*) _beginthreadex (0, 0,
                                   &threadEntryProc,
                                   userData,
                                   0, &threadId);
}

void juce_killThread (void* handle)
{
    if (handle != 0)
    {
#ifdef JUCE_DEBUG
        OutputDebugString (_T("** Warning - Forced thread termination **\n"));
#endif
        TerminateThread (handle, 0);
    }
}

int Thread::getCurrentThreadId()
{
    return (int) GetCurrentThreadId();
}

// priority 1 to 10 where 5=normal, 1=low
void juce_setThreadPriority (void* threadHandle, int priority)
{
    int pri = THREAD_PRIORITY_TIME_CRITICAL;

    if (priority < 1)
        pri = THREAD_PRIORITY_IDLE;
    else if (priority < 2)
        pri = THREAD_PRIORITY_LOWEST;
    else if (priority < 5)
        pri = THREAD_PRIORITY_BELOW_NORMAL;
    else if (priority < 7)
        pri = THREAD_PRIORITY_NORMAL;
    else if (priority < 9)
        pri = THREAD_PRIORITY_ABOVE_NORMAL;
    else if (priority < 10)
        pri = THREAD_PRIORITY_HIGHEST;

    if (threadHandle == 0)
        threadHandle = GetCurrentThread();

    SetThreadPriority (threadHandle, pri);
}

void Thread::setCurrentThreadAffinityMask (const uint32 affinityMask)
{
    SetThreadAffinityMask (GetCurrentThread(), affinityMask);
}

static HANDLE sleepEvent = 0;

void juce_initialiseThreadEvents()
{
    sleepEvent = CreateEvent (0, 0, 0, 0);
}

void Thread::yield()
{
    Sleep (0);
}

void Thread::sleep (int millisecs)
{
    if (millisecs >= 10)
    {
        Sleep (millisecs);
    }
    else
    {
        jassert (sleepEvent != 0);

        // unlike Sleep() this is guaranteed to return to the current thread after
        // the time expires, so we'll use this for short waits, which are more likely
        // to need to be accurate
        WaitForSingleObject (sleepEvent, millisecs);
    }
}

//==============================================================================
static int lastProcessPriority = -1;

// called by WindowDriver because Windows does wierd things to process priority
// when you swap apps, and this forces an update when the app is brought to the front.
void repeatLastProcessPriority()
{
    if (lastProcessPriority >= 0) // (avoid changing this if it's not been explicitly set by the app..)
    {
        DWORD p;

        switch (lastProcessPriority)
        {
        case Process::LowPriority:
            p = IDLE_PRIORITY_CLASS;
            break;

        case Process::NormalPriority:
            p = NORMAL_PRIORITY_CLASS;
            break;

        case Process::HighPriority:
            p = HIGH_PRIORITY_CLASS;
            break;

        case Process::RealtimePriority:
            p = REALTIME_PRIORITY_CLASS;
            break;

        default:
            jassertfalse // bad priority value
            return;
        }

        SetPriorityClass (GetCurrentProcess(), p);
    }
}

void Process::setPriority (ProcessPriority prior)
{
    if (lastProcessPriority != (int) prior)
    {
        lastProcessPriority = (int) prior;
        repeatLastProcessPriority();
    }
}

//==============================================================================
void Process::raisePrivilege()
{
    jassertfalse // xxx not implemented
}

void Process::lowerPrivilege()
{
    jassertfalse // xxx not implemented
}

void Process::terminate()
{
#if JUCE_DEBUG && defined (_MSC_VER) && JUCE_CHECK_MEMORY_LEAKS
    _CrtDumpMemoryLeaks();
#endif

    // bullet in the head in case there's a problem shutting down..
    ExitProcess (0);
}

void* Process::loadDynamicLibrary (const String& name)
{
    void* result = 0;

    JUCE_TRY
    {
        result = (void*) LoadLibrary (name);
    }
    JUCE_CATCH_ALL

    return result;
}

void Process::freeDynamicLibrary (void* h)
{
    JUCE_TRY
    {
        if (h != 0)
            FreeLibrary ((HMODULE) h);
    }
    JUCE_CATCH_ALL
}

void* Process::getProcedureEntryPoint (void* h, const String& name)
{
    return (h != 0) ? (void*) GetProcAddress ((HMODULE) h, name)
                    : 0;
}

//==============================================================================
void* juce_getGlobalLock (const String& name, int timeoutMillis)
{
    void* h = CreateMutex (0, TRUE, name);

    if (h != 0 && GetLastError() == ERROR_ALREADY_EXISTS)
    {
        if (timeoutMillis == 0
             || WaitForSingleObject (h, (timeoutMillis < 0) ? INFINITE : timeoutMillis)
                   == WAIT_TIMEOUT)
        {
            ReleaseMutex (h);
            CloseHandle (h);
            h = 0;
        }
    }

    return h;
}

void juce_releaseGlobalLock (void* handle)
{
    if (handle != 0)
    {
        ReleaseMutex (handle);
        CloseHandle (handle);
    }
}

END_JUCE_NAMESPACE
