/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module DBPROP.CPP | IDBProperties and IDBInfo interface implementations
//
//

// Includes ------------------------------------------------------------------
#include "hfiles.h"
#include "headers.h"

// Code ----------------------------------------------------------------------

//  IDBInfo specific interface methods
// CImpIDBInfo::GetKeywords  --------------------------------------------
//
// @mfunc Returns information about keywords used in text commands
//
// @rdesc HRESULT
//		@flag S_OK | Keywords successfully returned, NULL because no keywords
//      @flag E_INVALIDARG  | ppwszKeywords was NULL
//      @flag E_UNEXPECTED	| Can not be called unless initialized
//
STDMETHODIMP    CImpIDBInfo::GetKeywords
    (
	    LPWSTR*		ppwszKeywords
    )
{
	INTERFACE_METHOD_START( "IDBInfo::GetKeywords" );

	// Initialization
	if( ppwszKeywords )
		*ppwszKeywords = NULL;
	
	// check if we are initialized
	if (!m_pObj->m_fDSOInitialized)
		return E_UNEXPECTED;

    // check params
    if (NULL == ppwszKeywords)
		return E_INVALIDARG;

	// Keywords for Pervalise 7.0. Keyword for SSQL 3 are here, too.
	OLECHAR Keywords[] = 
		L"ACCELERATED,AFTER,ATOMIC,AUTOINC,BEFORE,BEGINS,BFLOAT,BINARYNULL,BIND,BLANK,"
		L"CALL,CHAIN,CONDITION,CONTAINS,CREATETAB,CS,CURDATE,CURRENCY,CURTIME,DATABASE,"
		L"DATEFORMAT,DCOMPRESS,DECIMALNULL,DICTIONARY,DO,EACH,ELSEIF,EX,EXCLUSIVE,EXIT,"
		L"EXIT,HANDLER,HANDLER,IF,INCLUDE,INDEX,INOUT,LEAVE,LENGTH,LOGICAL,LOGIN,LOOP,"
		L"LSTRING,LTRIM,LVAR,MASK,MILLISECOND,MOD,MODIFY,MONEY,NEW,NORMAL,NOTE,NOTFOUND,"
		L"NUMERCSA,NUMERICSTS,OLD,OPENMODE,OUT,OWNER,OWNERACCESS,PAGESIZE,PREALLOCATE,"
		L"PROC,RANGE,READONLY,REFERENCING,RELEASE,REPLACE,RESIGNAL,ROW,RTRIM,SAVEPOINT,"
		L"SECURITY,SEG,SIGNAL,SQLEXCEPTION,SQLEXCEPTIONS,SQLWARNING,SQLWARNINGS,START,"
		L"STATEMENT,STRINGNULL,SUBSTR,THRESHOLD,TYPE,UNDO,UNDO,UNSIGNED,UPDATABLE,"
		L"VERIFY,WEEKDAY,WHILE,ZSTRING";
	
		// Or (according to MSDASQL for Pervaseive's ODBS): 
		// ACCELERATED,AUTOINC,BEGINS,BFLOAT,BINARYNULL,BLANK,CONTAINS,CREATETAB,
		// CS,CURDATE,CURTIME,DCOMPRESS,DECIMALNULL,EX,EXCLUSIVE,LENGTH,LOGICAL,LOGIN,
		// LSTRING,LTRIM,LVAR,MASK,MILLISECOND,MOD,MODIFY,MONEY,NORMAL,NOTE,NUMERICSA,
		// NUMERICSTS,OPENMODE,OWNER,OWNERACCESS,PAGESIZE,PREALLOCATE,PROC ,RANGE,READONLY,
		// REPLACE,RTRIM,SECURITY,SEG,START,STATEMENT,STRINGNULL,SUBSTR ,THRESHOLD,
		// VERIFY,WEEKDAY,ZSTRING
	
	// create buffer 
	*ppwszKeywords = (LPWSTR) g_pIMalloc->Alloc(sizeof(Keywords));
	if( *ppwszKeywords == NULL )
		return E_OUTOFMEMORY;

	// Copy keywords
	wcscpy( *ppwszKeywords, Keywords );
	
	return S_OK;

    INTERFACE_METHOD_END();
}

// CImpIDBInfo::GetLiteralInfo  -----------------------------------------
//
// @mfunc Returns information about literals used in text command
//
// @rdesc HRESULT
//		@flag S_OK			| cLiterals was 0
//      @flag E_INVALIDARG  | cLiterals not equal to 0 and rgLiterals was NULL or
//							| pcLiteralInfo, prgLiteralInfo, or ppCharBuffer was NULL
//      @flag E_UNEXPECTED	| Can not be called unless initialized
//		@flag DB_E_ERRORSOCCURRED | None of the requested literals are supported
//
STDMETHODIMP    CImpIDBInfo::GetLiteralInfo
    (
	    ULONG           cLiterals,      //@parm IN | Number of literals being asked about
		const DBLITERAL rgLiterals[],   //@parm IN | Array of literals about which to return
				                        //           information
		ULONG*          pcLiteralInfo,  //@parm OUT | Number of literals for which info is
			                            //            returned
		DBLITERALINFO** prgLiteralInfo, //@parm OUT | Array of info structures
		WCHAR**         ppCharBuffer    //@parm OUT | Buffer for characters
    )
{
	INTERFACE_METHOD_START( "IDBInfo::GetLiteralInfo" );

	// Initialize 
	if( pcLiteralInfo )
		*pcLiteralInfo = 0;
	if( prgLiteralInfo )
		*prgLiteralInfo = NULL;
	if( ppCharBuffer )
		*ppCharBuffer = NULL;

	// check if we are initialized
	if (!m_pObj->m_fDSOInitialized)
		return E_UNEXPECTED;

	// check params
	if( cLiterals != 0 && rgLiterals == NULL )
		return E_INVALIDARG;
	
    if (!pcLiteralInfo || !prgLiteralInfo || !ppCharBuffer)
        return E_INVALIDARG;
    
	// Buffer with all used strings. 
	WCHAR wszAllStrings[]  = L".\0" L"%\0" L"_\0" L"[\0" L"]\0" L"\0"
							 L"!\"%&'()*+,-./:;<=>?@[\\]^{|} ~\0"
							 L"0123456789!\"%&'()*+,-./:;<=>?@[\\]^{|}~ \0"
							 L"\",.;*<>?|\0"
							 L"0123456789\",.;*<>?|\0"; // specsymbols  <<<< %"\  >>>>>
	
	// Supported literal infos
	const nLiteralInfosSupported = 18+2; // Should correlate with LiteralInfos[]
	
	// Allocate Memory for literal information
	if( cLiterals == 0 ) 
	{
		cLiterals = nLiteralInfosSupported; // Get all literals
		rgLiterals = NULL; // ignore literal IDs
	}

	// Allocate memory for literal information - DBLITERALINFO array
	*prgLiteralInfo = (DBLITERALINFO*)g_pIMalloc->Alloc(cLiterals * sizeof(DBLITERALINFO));
	if (!*prgLiteralInfo)
		return E_OUTOFMEMORY;

	*ppCharBuffer = (WCHAR*)g_pIMalloc->Alloc( sizeof(wszAllStrings) );
	if (!*ppCharBuffer)
	{
		g_pIMalloc->Free( *prgLiteralInfo );
		*prgLiteralInfo = NULL;
		return E_OUTOFMEMORY;
	}

	// Fill string buffer
	memcpy( *ppCharBuffer, wszAllStrings, sizeof( wszAllStrings ) );

	// Relative strings
	LPOLESTR pwszPoint = *ppCharBuffer;
	LPOLESTR pwszPercent = pwszPoint + wcslen( pwszPoint ) + 1;
	LPOLESTR pwszUnderline = pwszPercent + wcslen( pwszPercent ) + 1;
	LPOLESTR pwszLeftSqBracket = pwszUnderline + wcslen( pwszUnderline ) + 1;
	LPOLESTR pwszRightSqBracket = pwszLeftSqBracket + wcslen( pwszLeftSqBracket ) + 1;
	LPOLESTR pwszQuote = pwszRightSqBracket + wcslen( pwszRightSqBracket ) + 1;
	LPOLESTR pwszInvalidChars = pwszQuote + wcslen( pwszQuote ) + 1;
	LPOLESTR pwszInvalidFirstChars = pwszInvalidChars + wcslen( pwszInvalidChars ) + 1;
	LPOLESTR pwszInvalidCharsShort = pwszInvalidFirstChars + wcslen( pwszInvalidFirstChars ) + 1;
	LPOLESTR pwszInvalidFirstCharsShort = pwszInvalidCharsShort + wcslen( pwszInvalidCharsShort ) + 1;
	
	// String pointers are reltlive now
	// Supported literals and their information. Not all are supported now.
	DBLITERALINFO LiteralInfos[ nLiteralInfosSupported ] = {
		{ NULL, NULL, NULL, DBLITERAL_BINARY_LITERAL, TRUE, ~(ULONG)0 },
		{ NULL, pwszInvalidCharsShort, pwszInvalidFirstCharsShort, DBLITERAL_CATALOG_NAME, TRUE, MAXDDFPATH },
		{ pwszPoint, NULL, NULL, DBLITERAL_CATALOG_SEPARATOR, TRUE, 1 },
		{ NULL, NULL/*?*/, NULL/*?*/, DBLITERAL_CHAR_LITERAL, TRUE, ~(ULONG)0 },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_COLUMN_ALIAS, TRUE, 20 },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_COLUMN_NAME, TRUE, 20 },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_CORRELATION_NAME, TRUE, 20 },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_CURSOR_NAME, TRUE, 18 },

		//{ pwszLeftSqBracket, NULL, NULL, DBLITERAL_ESCAPE_PERCENT_PREFIX, TRUE, 1 },
		//{ pwszRightSqBracket, NULL, NULL, DBLITERAL_ESCAPE_PERCENT_SUFFIX, TRUE, 1 },
		//{ pwszLeftSqBracket, NULL, NULL, DBLITERAL_ESCAPE_UNDERSCORE_PREFIX, TRUE, 1 },
		//{ pwszRightSqBracket, NULL, NULL, DBLITERAL_ESCAPE_UNDERSCORE_SUFFIX, TRUE, 1 },
		
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_INDEX_NAME, TRUE, 20 },
		{ pwszPercent, NULL, NULL, DBLITERAL_LIKE_PERCENT, TRUE, 1 },
		{ pwszUnderline, NULL, NULL, DBLITERAL_LIKE_UNDERSCORE, TRUE, 1 },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_PROCEDURE_NAME, TRUE, 30 },
		
		{ NULL, pwszInvalidCharsShort, pwszInvalidFirstCharsShort, DBLITERAL_SCHEMA_NAME, TRUE, MAXOWNERNAME },
		{ pwszPoint, NULL, NULL, DBLITERAL_SCHEMA_SEPARATOR, TRUE, 1 },
		
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_TABLE_NAME, TRUE, 20 },
		{ NULL, NULL, NULL, DBLITERAL_TEXT_COMMAND, TRUE, 32767 },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_USER_NAME, TRUE, MAXUSERNAME },
		{ NULL, pwszInvalidChars, pwszInvalidFirstChars, DBLITERAL_VIEW_NAME, TRUE, 20 },
		{ pwszQuote, NULL, NULL, DBLITERAL_QUOTE_PREFIX, TRUE, 0 },
		{ pwszQuote, NULL, NULL, DBLITERAL_QUOTE_SUFFIX, TRUE, 0 }
	};

	// Process each of the DBLITERAL values that are in the restriction array
	*pcLiteralInfo = cLiterals;

	ULONG ulDex, ulSucceeded, i;
	if( rgLiterals != NULL )  // Set values to appropriate positions
		for( ulDex = ulSucceeded = 0; ulDex < cLiterals; ulDex++)
		{
			// Serching for in array
			for( i = 0; i < nLiteralInfosSupported; i++ )
				if( LiteralInfos[ i ].lt == rgLiterals[ulDex] )
				{
					(*prgLiteralInfo)[ulDex] = LiteralInfos[ i ]; // OK with string pointers!
					(*prgLiteralInfo)[ulDex].lt = rgLiterals[ulDex];
					ulSucceeded++; // Success for current element
					TRACE4( "Got literal %d -- %S -- %S", rgLiterals[ulDex], (*prgLiteralInfo)[ulDex].pwszLiteralValue, (*prgLiteralInfo)[ulDex].pwszInvalidStartingChars );
					break; 
				}
			
			// Not found
			if( i == nLiteralInfosSupported )
			{
				// fSupported will be set to FALSE, too.
				ZeroMemory( (*prgLiteralInfo) + ulDex, sizeof(DBLITERALINFO) );	
				// Restore literal number
				(*prgLiteralInfo)[ulDex].lt = rgLiterals[ulDex]; 
			}
		}
	else // set values somewhere
		for(ulSucceeded=0; ulSucceeded < nLiteralInfosSupported; ulSucceeded++) 
			(*prgLiteralInfo)[ulSucceeded] = LiteralInfos[ulSucceeded]; // OK with string pointers!

	// Clear string buffer if no infos found. Do not clear infos buffer! - acccording to specifications
	if( ulSucceeded == 0 )
	{
		g_pIMalloc->Free( *ppCharBuffer );
		*ppCharBuffer = NULL;
	}
	
	// Three variations what to return
	return (ulSucceeded == cLiterals) ? S_OK : (ulSucceeded ? DB_S_ERRORSOCCURRED : DB_E_ERRORSOCCURRED );

	INTERFACE_METHOD_END();
}



//  IDBProperties specific interface methods

// CImpIDBProperties::GetPropertyInfo  -----------------------------------------
//
// @mfunc Returns information about rowset and data source properties supported
// by the provider
//
// @rdesc HRESULT
//      @flag S_OK          | The method succeeded
//      @flag E_INVALIDARG  | pcPropertyInfo or prgPropertyInfo was NULL
//      @flag E_OUTOFMEMORY | Out of memory
//
STDMETHODIMP    CImpIDBProperties::GetPropertyInfo
    (
	    ULONG				cPropertySets,		//@parm IN  | Number of properties being asked about
	    const DBPROPIDSET	rgPropertySets[],	//@parm IN  | Array of cPropertySets properties about
				                                //            which to return information
	    ULONG*				pcPropertyInfoSets,	//@parm OUT | Number of properties for which information
			                                    //            is being returned
	    DBPROPINFOSET**		prgPropertyInfoSets,//@parm OUT | Buffer containing default values returned
		WCHAR**				ppDescBuffer		//@parm OUT	| Buffer containing property descriptions
    )
{
	INTERFACE_METHOD_START( "IDBProperties::GetPropertyInfo" );

    assert( m_pObj );
    assert( m_pObj->m_pUtilProp );

    // just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->GetPropertyInfo(
									m_pObj->m_fDSOInitialized,
									cPropertySets, 
									rgPropertySets,
									pcPropertyInfoSets, 
									prgPropertyInfoSets,
									ppDescBuffer);

	INTERFACE_METHOD_END();
}


// IDBProperties::GetProperties ----------------------------------------------------
//
// @mfunc Returns current settings of all properties in the DBPROPFLAGS_DATASOURCE 
//			property group
//
// @rdesc HRESULT
//      @flag S_OK          | The method succeeded
//      @flag E_INVALIDARG  | pcProperties or prgPropertyInfo was NULL
//      @flag E_OUTOFMEMORY | Out of memory
//
STDMETHODIMP CImpIDBProperties::GetProperties
    (
	    ULONG				cPropertySets,		//@parm IN | count of restiction guids
		const DBPROPIDSET	rgPropertySets[],	//@parm IN | restriction guids
		ULONG*              pcPropertySets,		//@parm OUT | count of property sets returned
		DBPROPSET**			prgPropertySets		//@parm OUT | property sets information returned
    )
{
	INTERFACE_METHOD_START( "IDBProperties::GetProperties" );
    
	assert( m_pObj );
    assert( m_pObj->m_pUtilProp );

	// What to get
	DWORD dwBitMask = PROPSET_INIT;

	// set BitMask
	if ( m_pObj->m_fDSOInitialized )
		dwBitMask |= (PROPSET_DSOINFO | PROPSET_DATASOURCE);

	// Check Arguments
	HRESULT hr = m_pObj->m_pUtilProp->GetPropertiesArgChk(dwBitMask, cPropertySets, 
								rgPropertySets, pcPropertySets, prgPropertySets);
	if ( FAILED(hr) )
		return hr;

    // Just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->GetProperties(
									dwBitMask,
									cPropertySets, 
									rgPropertySets,
									pcPropertySets, 
									prgPropertySets );

	INTERFACE_METHOD_END();
}


// CImpIDBProperties::SetProperties  --------------------------------------------
//
// @mfunc Set properties in the DBPROPFLAGS_DATASOURCE property group
//
// @rdesc HRESULT
//      @flag S_OK          | The method succeeded
//      @flag E_INVALIDARG  | cProperties was not equal to 0 and rgProperties was NULL
//
STDMETHODIMP    CImpIDBProperties::SetProperties
    (
	    ULONG		cPropertySets,
		DBPROPSET	rgPropertySets[]
	)
{
	INTERFACE_METHOD_START( "IDBProperties::SetProperties" );

	HRESULT hr			= E_FAIL;
	DWORD	dwBitMask   = PROPSET_DSOINFO;

	assert( m_pObj );
    assert( m_pObj->m_pUtilProp );

	// Quick return if the Count of Properties is 0
	if( cPropertySets == 0 )
		return S_OK;

	// Check Arguments for use by properties
	hr = m_pObj->m_pUtilProp->SetPropertiesArgChk(cPropertySets, rgPropertySets);
	if( FAILED(hr) )
		return hr;

	// set BitMask
	if ( m_pObj->m_fDSOInitialized )
		dwBitMask |= PROPSET_INIT;

    // just pass this call on to the utility object that manages our properties
    return m_pObj->m_pUtilProp->SetProperties(
									dwBitMask,
									cPropertySets, 
									rgPropertySets);

	INTERFACE_METHOD_END();
}


