/*
 * Copyright (c) 2007, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Reflection;
using System.Windows.Forms;

namespace MySQL.Utilities
{
  public interface ISplashScreen
  {
    void SetSplashScreenStatusInfo(string NewStatusInfo);
  }

  public class MySQLApplicationContext : ApplicationContext
  {
    private static MySQLApplicationContext currentContext;

    public MySQLApplicationContext()
    {
      if (currentContext == null)
        currentContext = this;
    }

    public MySQLApplicationContext(Form AppMainForm)
      : this()
    {
      this.MainForm = AppMainForm;
    }

    public static MySQLApplicationContext CurrentContext
    {
      get { return currentContext; }
    }

  }

  public class SplashScreen
  {
    private static Form form = null;
    private static ISplashScreen splashScreenInterface = null;
    private static string tempStatus = string.Empty;

    /// <summary>
    /// Display the splash form
    /// </summary>
    public static void Show(Type splashFormType)
    {
      if (splashFormType == null)
        return; // Simply don't show a splash if its creation failed.

      CreateInstance(splashFormType);
     // form.TopMost = true;
      form.Show();
      form.Update();
    }



    /// <summary>
    /// Set the status message
    /// </summary>
    public static string Status
    {
      set
      {
        if (splashScreenInterface == null || form == null)
        {
          tempStatus = value;
          return;
        }

        form.Invoke(
            new SplashStatusChangedHandle(
              delegate(string str) 
              { splashScreenInterface.SetSplashScreenStatusInfo(str); }),
            new object[] { value });
      }

    }

    /// <summary>
    /// Closes the splash screen.
    /// </summary>
    public static void CloseSplash()
    {
      if (form != null)
      {
        if (form.InvokeRequired)
          form.Invoke((Action)(() => form.Close()));
        else
          form.Close();
        form = null;
      }
    }

    private static void CreateInstance(Type FormType)
    {

      object obj = FormType.InvokeMember(null,
        BindingFlags.DeclaredOnly |
        BindingFlags.Public | 
        BindingFlags.NonPublic |
        BindingFlags.Instance | 
        BindingFlags.CreateInstance, null, null, null);

      form = obj as Form;

      splashScreenInterface = obj as ISplashScreen;
      if (form == null || splashScreenInterface == null)
      {
        throw new Exception(
          "The type passed in must inherit from System.Windows.Forms.Form and must implement the ISplashScreen interface.");
      }

      if (!string.IsNullOrEmpty(tempStatus))
        splashScreenInterface.SetSplashScreenStatusInfo(tempStatus);
    }

    private delegate void SplashStatusChangedHandle(string NewStatusInfo);

  }
}
