/* 
 * Copyright (c) 2010, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Implementation of the mforms splitter control for Windows.
 */

#include "stdafx.h"

#include "mforms/mforms.h"
#include "wf_view.h"
#include "wf_splitter.h"

using namespace MySQL::Forms;
using namespace System::Windows::Forms;

//--------------------------------------------------------------------------------------------------

SplitterImpl::SplitterImpl(mforms::Splitter *self)
  : ViewImpl(self)
{
  pendingSplitterDistance = -1;
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::safe_assign_splitter_distance(SplitContainer ^container, int distance)
{
  // If the splitter is within any of the min size areas of the two panels set it to the center
  // between both. If that still fails it can only mean the container is smaller than the sum of
  // the min sizes. In that case don't assign a splitter position at all.
  int size = container->Orientation == Orientation::Horizontal ? container->Height : container->Width;
  if (distance < container->Panel1MinSize ||
    distance > size - container->Panel2MinSize)
    distance = (size - container->Panel2MinSize - container->Panel1MinSize) / 2;

  try
  {
    if (distance > 0)
      container->SplitterDistance = distance;
  }
  catch (...)
  {
    // Ignore stupid splitter distance errors. The SplitContainer really should adjust values
    // outside the valid range to a meaningful position.
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::safe_assign_min_size(SplitContainer ^container, int min_size, bool first)
{
  // Similar as for the splitter distance the split container will throw an exception if the
  // min sizes don't match the size.
  int size = (container->Orientation == Orientation::Horizontal) ? container->Height : container->Width;
  if (first)
  {
    if (size - container->Panel2MinSize - container->SplitterWidth <= min_size)
    {
      int new_size = container->Panel2MinSize + container->SplitterWidth + min_size;
      if (container->Orientation == Orientation::Horizontal) 
        container->Height = new_size;
      else
        container->Width = new_size;
    }

    try
    {
      container->Panel1MinSize = min_size;
    }
    catch (...)
    {
    }
  }
  else
  {
    if (size - container->Panel1MinSize - container->SplitterWidth <= min_size)
    {
      int new_size = container->Panel1MinSize + container->SplitterWidth + min_size;
      if (container->Orientation == Orientation::Horizontal) 
        container->Height = new_size;
      else
        container->Width = new_size;
    }

    try
    {
      container->Panel2MinSize = min_size;
    }
    catch (...)
    {
    }
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::set_position(int position)
{
  SplitContainer^ container = get_control<SplitContainer>();
  if (container->IsHandleCreated && container->Visible)
    safe_assign_splitter_distance(container, position);
  else
    pendingSplitterDistance = position;
}

//--------------------------------------------------------------------------------------------------

int SplitterImpl::get_position()
{
  SplitContainer^ container = get_control<SplitContainer>();
  if (pendingSplitterDistance == -1)
    return container->SplitterDistance;
  else
    return pendingSplitterDistance;
}

//--------------------------------------------------------------------------------------------------

bool SplitterImpl::create(mforms::Splitter *self, bool horizontal)
{
  SplitterImpl^ splitter = gcnew SplitterImpl(self);

  if (splitter != nullptr)
  {
    SplitContainer^ container= ViewImpl::create<SplitContainer>(self, splitter);
    container->BackColor = Color::Transparent;
    container->Paint += gcnew PaintEventHandler(&SplitterImpl::DoPaint);
    container->SplitterMoved += gcnew SplitterEventHandler(&SplitterImpl::DoSplitterMoved);
    container->Size = Size(100, 100);
    container->TabStop = false;

    // Note: orientation is that of the splitter, not the container layout.
    container->Orientation = horizontal? Orientation::Vertical : Orientation::Horizontal;
    container->SplitterWidth = 6;

    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

/**
 * Adds a new child window to the splitter. Behavior is as follows:
 * - If the left panel is empty add to this.
 * - If not and the right panel is empty add to this.
 * - If no panel is empty then we have an error.
 */
void SplitterImpl::add(mforms::Splitter *self, mforms::View *child, int min_size, bool fixed)
{
  SplitterImpl^ splitter = (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
  {
    ViewImpl^ view = (ViewImpl^)ObjectImpl::FromUnmanaged(child);
    Control^ control = view->get_control<Control>();

    SplitContainer^ container = splitter->get_control<SplitContainer>();
    if (container != nullptr && control != nullptr)
    {
      if (container->Panel1->Controls->Count == 0)
      {
        container->Panel1->Controls->Add(control);

        splitter->safe_assign_min_size(container, min_size, true);
        if (fixed)
          container->FixedPanel = FixedPanel::Panel1;
        control->Dock = DockStyle::Fill;
        view->set_resize_mode(AutoResizeMode::ResizeBoth);
      }
      else
        if (container->Panel2->Controls->Count == 0)
        {
          container->Panel2->Controls->Add(control);
          splitter->safe_assign_min_size(container, min_size, false);
          if (fixed)
            container->FixedPanel = FixedPanel::Panel2;
          control->Dock = DockStyle::Fill;
          view->set_resize_mode(AutoResizeMode::ResizeBoth);
        }
        else
          throw std::logic_error("mforms splitter error: adding more than 2 child controls is not allowed.");

      // Hide the panel that has no child control.
      container->Panel1Collapsed = container->Panel1->Controls->Count == 0;
      container->Panel2Collapsed = container->Panel2->Controls->Count == 0;
    }
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::remove(mforms::Splitter *self, mforms::View *child)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
  {
    ViewImpl^ view= (ViewImpl^)ObjectImpl::FromUnmanaged(child);
    Control^ control= view->get_control<Control>();

    // Since there can only be one child window on each panel we do a Clear() instead removing only the
    // individual control, fixing so possibly invalid setups (if they ever occur).
    // Additionally, collapse the panel that is now empty unless both panels are empty then.
    SplitContainer^ container = splitter->get_control<SplitContainer>();
    if (container->Panel1->Controls->Contains(control))
    {
      container->Panel1->Controls->Clear();
      if (container->Panel2->Controls->Count > 0)
        container->Panel1Collapsed = true;
    }
    else
      if (container->Panel2->Controls->Contains(control))
      {
        container->Panel2->Controls->Clear();
        if (container->Panel1->Controls->Count > 0)
          container->Panel2Collapsed = true;
      }
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::set_position(mforms::Splitter *self, int position)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
    splitter->set_position(position);
}

//--------------------------------------------------------------------------------------------------

int SplitterImpl::get_position(mforms::Splitter *self)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
    return splitter->get_position();
  return 0;
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::set_expanded(mforms::Splitter *self, bool first, bool expand)
{
  SplitterImpl^ splitter = (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
  {
    SplitContainer^ container = splitter->get_control<SplitContainer>();
    if (first)
      container->Panel1Collapsed = !expand;
    else
      container->Panel2Collapsed = !expand;
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * We don't use the paint event for drawing but to apply a pending splitter position setting
 * as it is way too unreliable to set it directly (depends on visibility state, size, created handles etc.).
 */
void SplitterImpl::DoPaint(System::Object ^sender, PaintEventArgs ^e)
{
  SplitContainer^ container = dynamic_cast<SplitContainer^>(sender);
  mforms::Splitter* backend = ViewImpl::get_backend_control<mforms::Splitter>(container);
  SplitterImpl^ splitter = (SplitterImpl^) ObjectImpl::FromUnmanaged(backend);
  if (splitter->pendingSplitterDistance > -1)
  {
    splitter->safe_assign_splitter_distance(container, splitter->pendingSplitterDistance);
    splitter->pendingSplitterDistance = -1;
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::DoSplitterMoved(System::Object^ sender, SplitterEventArgs^ e)
{
  SplitContainer^ container= dynamic_cast<SplitContainer^>(sender);
  mforms::Splitter* backend= ViewImpl::get_backend_control<mforms::Splitter>(container);
  if (backend != NULL)
    backend->position_changed();
}

//--------------------------------------------------------------------------------------------------
