#!/usr/bin/env python

#---------------------------------------------------------------
# stream.py
# socket IO wrapper, supporting BT e TCP, on linux and Symbian S60 
# (C) 2006-2007 M.Andreoli
#------------------------------------------------------------------
# stream.py encapsulate a simple file send/receive protol and can
# be used as stdin and stdout stream in python programming
#-----------------------------------------------------------------

"""
USAGE

The server:
		s=Stream('BT')
		s.accept(service="XYZ")
		s.write(' ... data ....')
		s.sendfile('/tmp/x11.jpg')
		ok=s.read()
		if ok=='END':
			end_server()
			s.close(); s.shutdown()
The client:
		s.Stream('BT')
		s.connect(service="XYZ")
		data=s.read()
		s.recfile('/tmp/x11.jpg')
		s.write('END')
		sleep_some_time()
		s.close()

Variants:
		s.accept(port=13)
		s.connect(host='...', port=13)
"""



import sys,os
import socket
import time
import string
import traceback

stack='symbian'

try:
	import bluetooth
	stack='bluez'
except:
	stack='symbian'

# count how many chunks we can have spliting a 
# file in blocksize (with rest)

def chunkcount(filename,blocksize):
    size=os.path.getsize(filename)
    blocks=size/blocksize
    rest=size % blocksize
    if rest!=0: blocks+=1
    return blocks 

# internal function

def trace(msg=''):

    error_type, error, traceback = sys.exc_info()
    if error==None:
        error=""

    sys.stderr.write("TRACE: %s: %s\n" % (msg, error) )
    raise Exception,"TRACE: %s" % msg

#-----------------
# the main class
#------------------

# create objects of type stream, and work using both
# transports: bluetooth and TCP/IP

class Stream:
    """
    usage: Stream(mode='BT|TCP',verbose)
    """
    def __init__(self,mode,verbose=0):
        self.MAX_BUFFER=5120
        self.ip=''		# mode=TCP 
        self.client=None
        self.server=None
        self.peer=('None',0)
    	self.verbose=verbose
	self.log=lambda x: sys.stderr.write(x+"\n")
	self.mode=mode
	self.host=''
	self.port=None
	

    def debug(self,msg):
        if self.verbose:
		sys.stderr.write("STREAM: "+msg+"\n")
    #-------------------
    # low level IO
    #-------------------
    def close(self):
	if self.client == None: return
	try:
        	self.client.close()
		self.client=None
    		self.debug("%s closed connection" % repr(self.peer))
		self.peer=('None',0)
	except:
    		self.debug("problem closing connection")
    def shutdown(self):
	self.close()
	if self.server!=None:
		self.server.close()
    def read_(self,n=1):
        return self.client.recv(n)
    def write_(self,str):
	try:
        	return self.client.sendall(str)
	except:
		n=len(str)
		sent=0
		while sent<n:
        		sent+=self.client.send(str[sent:])
		return sent
	
    def readline_(self):
        s=[]
        while 1:
            c=self.client.recv(1)
            if c=='':
                continue
            s.append(c)
            if c=='\n':
                break
        return ''.join(s)
    def raw_input_(self,prompt=""):
        self.write_(prompt)
        return self.readline_()
    def flush(self):
	#self.log("flush() called")
	#self.write_('')
	pass
    #-------------------
    # high level IO
    #-------------------

	# Simple Packet's Protocol adopted here:
	# each packet has this simple structure: header+data
	# LY  where L=|Y|, and |L|=1 if L<=128
	# CLY	C-128=|L| and L=|Y|  if L>128
	# Es:
	# 	char(2) 'ab'
 	# 	char(130) '28' 'asdfakdfkaf 28 characters .....' 

    def readline(self):
	    return self.read()
    def raw_input(self,prompt=None):
	    if prompt: self.write(prompt)
	    return self.readline()
    def write(self,data=''):
	n=len(data)
	if n==0:
		return 0
	#sys.stderr.write("write(): [%s]\n" % data)
	if n<=128:
		self.write_(chr(n))
		return self.write_(data)
	else:
		nn=str(n); l=len(nn)
		self.write_(chr(128+l))
		self.write_(nn)
        	sentbytes=0        
        	while sentbytes<n:
            		m=min(n,self.MAX_BUFFER)
            		self.write_(data[sentbytes:sentbytes+m])
            		sentbytes+=m
        	self.debug("Sent %d bytes"%sentbytes)
    		return sentbytes

    def read(self):
	c=ord( self.read_(1) )
	if c<=128:
		return self.read_(c)
	l=c-128
	n=int( self.read_(l) )	

        recvbytes=0
        data=[]
        while recvbytes<n:
            m=min(n-recvbytes,self.MAX_BUFFER)
            recvstring=self.read_(m)
            recvbytes+=len(recvstring)
            data.append(recvstring)
        self.debug("Received %d bytes"%recvbytes )
	#sys.stderr.write("read(): [%s]\n" % ''.join(data))
        return ''.join(data) 

# send a file over the stream
# Self made OBEX protocol :-)
#       chunks | chunk | chunk | .....


    def sendfile(self,filename,blocksize=1024*20):
        ok=True
        try:
                f=open(filename,'rb')
        except:
                trace("opening %s" % filename)
		self.write('0')
                return False

        blocks=chunkcount(filename,blocksize)
	self.debug("sendfile: blocks %d" %blocks)
        sent=0
	self.write(str(blocks))
        for n in range(blocks):

                content=f.read(blocksize)

                try:
                        self.write(content)
                        sent+=len(content)
                except:
                        ok=False
                        trace("writing block %d/%d  of %s" % (n,blocks,filename) )
                        continue

        self.flush()
        f.close()
        return ok

# receive a file from the stream
# (for the protocol, see sendfile)

    def recfile(self,filename):
        size=0; nb=0
        ok=True
	
        try:
                #f=open(filename,'wb+')
                f=open(filename,'wb')
        except:
                print "impossible to create [%s]; data will be discarded" %filename
                ok=False
	blocks=int( self.read() )
        for b in range(blocks):
                try:
                        content=self.read()
                except:
                        ok=False
                        break
                size+=len(content)
                nb+=1
		if ok:
                	f.write(content)
	if ok:
        	f.close()
	#self.debug("recfile: blocks received %d" %nb)
        return ok

# create the BT service and set also 
# self.port

    def adv_service(self,service):
	if self.mode !='BT':
		trace("adv_service: only BT")

	if sys.platform.find("symbian")<0:
		port=bluetooth.PORT_ANY
		if self.port!=None: port=self.port
                self.server.bind(("",port))
                self.server.listen(1)
                self.port = self.server.getsockname()[1]
                uuid = "94f39d29-7d6d-437d-973b-fba39e345f4d"
                bluetooth.advertise_service( self.server, service,
                service_id = uuid,
                service_classes = [  uuid, bluetooth.SERIAL_PORT_CLASS ],
                profiles = [ bluetooth.SERIAL_PORT_PROFILE ] )
        else:
		self.port=socket.bt_rfcomm_get_available_server_channel(self.server)
		self.server.bind(("", self.port))
		self.server.listen(1)
        	socket.bt_advertise_service(unicode(service), 
				self.server, True, socket.RFCOMM)

        	#socket.set_security(self.server, socket.AUTH)

        self.log("BT service %s:%s advertised" %(service,self.port))

# in accept() service= and port= are
# mutually exclusive
			
    def accept(self,service=None,port=None):

	self.port=port

	# create the server socket
	self.server=self.create_socket()

	# the service= take priority

	if service!=None:
		try:
			self.adv_service(service)
		except:
			trace("impossible to create the service %s"%service)
	else:
		if self.port!= None:	
			       # bring up WLAN for Nokia
		        if sys.platform.find("symbian")>=0 and self.mode=='TCP':
                		id=socket.select_access_point()
                		ap=socket.access_point(id)
                		ap.start()
                		self.ip="%s" % ap.ip()
                		self.log("phone ip %s" % self.ip)
                		self.log("NB: if connections fails, try again.\n\n")

			addr=(self.ip, int(self.port) )
			self.server.bind(addr)
			self.server.listen(1)
		else:
			trace("accept(): port=None") 
		
	self.log( "%s mode, platform: %s, port %d"
			%(self.mode,sys.platform,self.port))
	self.log( "Waiting for connections ...");
	try:
		(self.client,self.peer) = self.server.accept()
	except:
		self.log("client closed connection")
		raise
	self.log( "Connection from "+repr(self.peer) )
   	return True 

# (helper) create a socket and return it
# logics based on: self.host, self.port, self.mode

    def create_socket(self):
        if self.mode=='BT':
        	if sys.platform.find("symbian")>=0:
                       	sock = socket.socket(socket.AF_BT,socket.SOCK_STREAM)
        	else:
			# we support only bluez clients
                       	sock=bluetooth.BluetoothSocket( bluetooth.RFCOMM )
        elif self.mode=='TCP':
                sock=socket.socket(socket.AF_INET,socket.SOCK_STREAM)
	else:
		raise Exception, "mode=None in create_socket()"

	return sock 

# only return the first target (host,port)
# NB: is service=None,the output is architetture-depending
  
    def find_service(self,service=None):
	if self.mode!='BT':
		trace("find_service(): only BT")

        if sys.platform.find("symbian")>=0:
		print "please, look to the phone device ..."
		#output: ('host',{u'service':port})
                host,services = socket.bt_discover()
                print "Discovered: %s, %s"%(host, services)
		if service==None:
			return (host,services)

		# search the specific service name
		# and return port=0 if not found
		port=services.get(service,0)
		return (host,port)

        else:	# we assume Bluez 

		print "bluez devices discovering ..."
                try:
			if service!=None:
                        	device = bluetooth.find_service(name=service)[0]
				host=device["host"]
				port=device["port"]
				return (host,port)
			else:
                        	device = bluetooth.find_service()
				return device
                except:
                        raise Exception,"bluez bluetooth unavailable"

# connect(): you can specify or the service= or
# the host/port pair

    def connect(self,service=None,host=None,port=None):
        self.host=host
	self.port=port

     	# discovery?
	if service!=None:
		self.host,self.port=self.find_service(service)

	if self.host==None or self.port==None or self.port==0:
		trace("connect(): no host or no port")

	self.client = self.create_socket() 

	if self.client == None:
		self.log("impossible to create the socket")
		raise Exception,"create_socket() failed!"	

	target=(self.host,int(self.port))
        self.log("Connecting to %s, via %s port %s" %(self.host,self.mode,self.port))
        self.client.connect(target)
        sys.stdout.flush()
        self.log( "Connected." )
	return True 



#--------------------------
# MAIN
#--------------------------

# simple TCP python console

if __name__ == '__main__':
	print "ciao"	
