/*
 * ppdncrypt.c - encrypt a ppdd filesystem without kernel suppor
 *
 * Copyright 1999,2002 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "ppddmount.h"

static char *progname;

static int encrypt_all(int pgp, int convent, int fd, int ffd, 
				const char *fname, const char *function,
				struct ppdd_keys *keys, 
				unsigned char *pmd5, unsigned char *cmd5)
{
	unsigned char 	iblock[32768];
	unsigned char 	oblock[32768];
	int 		i, rlength, wlength, count, reply, oseek, iseek;
        unsigned long   size, dlength, mlength, block, tblocks;
        time_t          now, then;
	struct MD5Context pmd5C;
	struct MD5Context cmd5C;

	tblocks = getfilesize(fd);
	size = tblocks >> 11;
        printf("%s %s (%ldMb)\n", function, fname, size);

        then = time(NULL);

	MD5Init(&pmd5C);
	MD5Init(&cmd5C);

	iseek = oseek = 1024;
	if (convent) {
	    iseek = 0;
	    tblocks += 2;
	}
	
	if (pgp) oseek = 2048;

/* if conventional encryption we want to preserve the first 1024 bytes */

	lseek (fd, iseek, SEEK_SET);

/* for pgp we have already written the first 2048 bytes, for ppdd 1024 */

	lseek (ffd, oseek, SEEK_SET);

	count = 1024;
	dlength = 1024;
	block = 2;
	mlength = 0;

	while (1)
	{
		if ((tblocks - block) < 65) {
		    if (block == tblocks) {
			rlength = 0;
		    } else {
	            	rlength = read (fd, iblock, 512);
  		    }
                } else {
	            rlength = read (fd, iblock, sizeof(iblock));
		}
		if (rlength == 0) {
			MD5Final(pmd5, &pmd5C);
			MD5Final(cmd5, &cmd5C);
			for (i=0;i<16;i++) cmd5[i] ^= pmd5[i];
			mlength += dlength >> 20;
			return mlength;
		}
		if (rlength < 0) {
			PPDDERROR(180)
			return 0;
		}
		MD5Update(&pmd5C, iblock, rlength);
	        reply = transfer_bf(keys, PPDDENCRYPT, iblock,
					oblock, rlength >> 9, block);
		if (reply) {
			PPDDERROR(181)
			return 0;
		}
		MD5Update(&cmd5C, oblock, rlength);
		wlength = write (ffd, oblock, rlength);
		if (wlength != rlength) {
			PPDDERROR(182)
			return 0;
		}
		dlength += wlength;
		block += wlength >> 9;
		count--;
		if (count == 0) {
			count = 1024;
                        now = time(NULL);
                        if (now > (then + 59)) {
                                then = now;
				mlength += dlength >> 20;
                        	dlength &= 0x0fffff;
                                printf("%s: %ld of %ld Mb so far\n",
                                        fname, mlength, size);
                                fflush(stdout);
			}
		}
	}
	return 0;
}

static int pgp_encrypt(int size, void *iblock, void *oblock)
{
	int 		apipe[2];
	int 		bpipe[2];
	int 		cstatus, reply, max, rlen;
	pid_t 		pid;
 	fd_set 		rfds, wfds, *prfds, *pwfds;
 	struct timeval 	tv;

	if (pipe (apipe)) {
		PPDDERROR(183)
		return 0;
	}
	if (pipe (bpipe)) {
		PPDDERROR(184)
		return 0;
	}

	if ((pid = fork())) {
	    if (pid < 0) {
		PPDDERROR(185)
		return 0;
	    }
	    close (apipe[0]);
	    close (bpipe[1]);
	    max = apipe[1];
	    if (max < bpipe[0]) max = bpipe[0];
	    max++;
	    prfds = &rfds;
	    pwfds = &wfds;

	    for(;;) {
		tv.tv_usec = 0;
		tv.tv_sec = 1;
  		FD_ZERO(&rfds);
  		FD_ZERO(&wfds);
  		FD_SET(bpipe[0], &rfds);
		FD_SET(apipe[1], &wfds);
		select(max, prfds, pwfds, NULL, &tv);
  		if (pwfds && (FD_ISSET(apipe[1], &wfds))) {
		    if (size != write (apipe[1], iblock, size)) {
			PPDDERROR(186)
			return 0;
	    	    }
	    	    close (apipe[1]);
		    pwfds = NULL;
		}
  		if (prfds && (FD_ISSET(bpipe[0], &rfds))) {
	    	    rlen = read (bpipe[0], oblock+4, size<<1);
	    	    if (rlen < 0) {
			PPDDERROR(187)
		  	return 0;
	            }
		    memcpy (oblock,&rlen,sizeof(rlen));
	    	    close (bpipe[0]);
		    prfds = NULL;
		}
	    	if (waitpid (pid, &cstatus, WNOHANG)) {
	    	    reply = WEXITSTATUS(cstatus);
	    	    return (!reply);
		}
	    }
	} else {
	    close (apipe[1]);
	    close (bpipe[0]);
	    dup2(apipe[0],0);
	    close (apipe[0]);
	    dup2(bpipe[1],1);
	    close (bpipe[1]);
	    execlp("pgpe","pgpe","-r", "backup","-f","+batchmode=1",
			"+compress=0","+textmode=0",NULL);
	    PPDDERROR(188)
	    exit (1);
	}
	return 0;
}


static int ppdncrypt(int init, int rand, int pgp, int convent, int weak, int fd,
				const char *ifile, const char *ofile )
{
	struct crypt_control_block cblock;
	struct ppdd_keys 	   keys;
	Blowfish_Key		   bkey;
	unsigned char		   ukey[PPDDKEYSIZE];
	unsigned char		   pgpblock[2048];
	unsigned short		   version;
	unsigned long		   tblocks;
	time_t			   time_created;
	int                        dlength, ifd, ofd, ok;


	dlength = 0;

	if ((ifd = open (ifile, O_RDONLY)) < 0) {
	    	PPDDERROR(189)
		return 1;
	}

	check_access (ifile, ifd, 0);

	tblocks = getfilesize(ifd);
	if (!tblocks) {
	    	PPDDERROR(888)
		return 1;
	}
	if (tblocks & 7) {
	    fprintf(stderr, "Warning: %s is not a multiple of 4096 bytes\n\n",
                                                                       ifile);
	}

	if (ofile) {
		if ((ofd = open (ofile, O_WRONLY|O_TRUNC|O_CREAT, 0)) < 0) {
	    		PPDDERROR(190)
			return 1;
		}
		tblocks = getfilesize(ofd);
		if (tblocks & 7) {
		    fprintf(stderr,
		 "Warning: %s is not a multiple of 4096 bytes\n\n", ifile);
		}
	} else {
		if ((ofd = open (ifile, O_WRONLY)) < 0) {
	    		PPDDERROR(191)
			return 1;
		}
	}

	if ((fchown(ofd, 0, 0))) {
	  	PPDDERROR(192)
		return 1;
	}

	if ((fchmod(ofd, 0))) {
	  	PPDDERROR(193)
		return 1;
	}

	printf("Starting key generation - please wait\n\n");

	ok = random_fill_fast (&cblock, sizeof(cblock));
	if (!ok) {
		return 1;
	}

	ok = random_fill_fast (pgpblock, sizeof(pgpblock));
	if (!ok) {
		return 1;
	}

	lseek (ofd, 0, SEEK_SET);

	if (pgp) {
	    if (2048 != write(ofd, pgpblock, 2048)) {
	  	PPDDERROR(194)
	        ok = 0;
	    }
	    if (ok) {
	        setup_bf((struct ppdd_ukeys*)cblock.keys, &keys);
     		ok = dlength = encrypt_all (pgp, convent, ifd, ofd, 
					ifile, "Encrypting", &keys,
					cblock.pmd5, cblock.cmd5);
	    }
	    if (ok) {
		cblock.flags |= 3;
	        ok = pgp_encrypt(1024, &cblock, pgpblock);
	    }
	    if (ok) {
		printf("Finished: %d bytes encrypted\n", dlength);
	    }
	    if (ok) {
	        lseek (ofd, 0, SEEK_SET);
	        if (2048 != write(ofd, pgpblock, 2048)) {
	  	    PPDDERROR(195)
		    ok = 0;
	        }
	    }
	} else {
	    if (1024 != write(ofd, pgpblock, 1024)) {
	  	PPDDERROR(196)
	        ok = 0;
	    }
	    if (ok) {
                if (rand) {
		    printf("Writing random data.\n");
	            setup_bf((struct ppdd_ukeys*)cblock.keys, &keys);
       		    ok = dlength = encrypt_all (pgp, convent, ifd, ofd,
					ifile, "Randomising", &keys,
					cblock.pmd5, cblock.cmd5);
	            if (ok) {
	                ok = random_fill_fast (&cblock, sizeof(cblock));
                    }
                }
            }
	    if (ok) {
	        ok = newpass(fd, ifile, ukey, &cblock, 1);
	    }
	    if (ok) {
	        if (!weak) {
		    ok = random_fill_slow (cblock.keys, sizeof(cblock.keys));
	        }
	    }
	    if (ok) {
	        setup_bf((struct ppdd_ukeys*)cblock.keys, &keys);
                if (!init)
       		    ok = dlength = encrypt_all (pgp, convent, ifd, ofd,
					ifile, "Encrypting", &keys,
					cblock.pmd5, cblock.cmd5);
	    }
	    if (ok) {
                if (init) {
                    cblock.flags &= 0xfc;
                } else {
		    cblock.flags |= 3;
                }
	        Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, &cblock.flags, &cblock.flags,
								CB5LENGTH);
	        time_created = time(NULL);
	        memcpy(&cblock.time_created, &time_created, sizeof(time_t));
	        version = PP_VERSION;
	        memcpy(&cblock.file_version, &version, sizeof(unsigned short));
	        make_crc(&cblock,PPDD_CRC_MAKE);
	        Blowfish_ExpandUserKey(ukey, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, cblock.mkey, cblock.mkey, CBMLENGTH);
	    }
	    if (ok) {
	        if (write_control_block(ofd, &cblock)) {
	  	    PPDDERROR(197)
		    ok = 0;
	        }
	    }
	    if (ok) {
                if (init) {
		    printf("Initialized\n");
		} else {
		    printf("Finished: %d Mb encrypted\n", dlength);
		}
	    }
	}

	memset(pgpblock, 0, sizeof(pgpblock));
	memset(&cblock, 0, sizeof(cblock));
	memset(&keys, 0, sizeof(keys));
	memset(&ukey, 0, sizeof(ukey));
	memset(&bkey, 0, sizeof(bkey));

	if (ok) return 0;
	return (1);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-cw] [ -f file_descriptor] input-file|input-device output-file|output-device\n\
  %s -p[c] input-file|input-device output-file\n\
  %s -o[w] [ -f file_descriptor] file|device\n\
  %s -i[w][r] [ -f file_descriptor] file|device\n\n",progname,progname,progname,progname);
	fprintf(stderr, "      -c conventional encryption i.e the first\n");
	fprintf(stderr, "         1024 byte block is preserved\n");
	fprintf(stderr, "      -p use pgp to encrypt the data\n");
	fprintf(stderr, "      -w use weaker random key generation\n");
	fprintf(stderr, "      -o overwrite\n");
	fprintf(stderr, "      -i initialize\n");
        fprintf(stderr, "      -f get pass phrase from an open file descriptor\n");
	fprintf(stderr, "      -r fill file with random bytes\n\n");
	fprintf(stderr, "      Use this to encrypt a ppdd filesystem\n");
	fprintf(stderr, "      without having ppdd support in the kernel\n\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int fd = -1;
	int init = 0;
	int rand = 0;
	int pgp = 0;
	int weak = 0;
	int convent = 0;
	int overwrite = 0;

	progname = "ppdncrypt";
	if (ppdd_intro(progname)) usage ();

	while ((c = getopt(argc,argv,"cwopirf:")) != EOF) {
		switch (c) {
			case 'c':
                                convent = 1;
				break;
			case 'w':
                                weak = 1;
				break;
			case 'o':
                                overwrite = 1;
				break;
			case 'r':
                                rand = 1;
				break;
			case 'i':
                                init = 1;
				break;
			case 'p':
                                pgp = 1;
				break;
                       case 'f':
                                if (optarg[1]) usage();
                                if (optarg[0] > '9') usage();
                                if (optarg[0] < '0') usage();
                                fd = optarg[0] - '0';
                                break;
			default:
				usage();
		}
	}

	if (rand && !init) usage();

	if (convent) {
	  if (overwrite) usage();
	  if (pgp) usage();
	  if (init) usage();
        }

	if (pgp) {
	  if (weak) usage();
	  if (overwrite) usage();
	  if (init) usage();
	  if (fd != -1) usage();
        }

	if (init) {
	  if (overwrite) usage();
        }

	if (overwrite) {
	    if (argc != (optind + 1)) usage();
	    return (ppdncrypt(init, 0, pgp, convent, weak, fd, argv[argc-1], NULL));
	} else {
	    if (init) {
	        if (argc != (optind + 1)) usage();
	        return (ppdncrypt(init, rand, pgp, convent, weak, fd, argv[argc-1],
                                                                         NULL));
	    } else {
	        if (argc != (optind + 2)) usage();
	        return (ppdncrypt(init, 0, pgp, convent, weak, fd, argv[argc-2],
                                                                 argv[argc-1]));
	    }
	}
}

