/*
 * ppddpassw.c - change password of an encrypted file system on a ppdd
 *
 * Copyright 1998,9, 2002 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <getopt.h>
#include <errno.h>

#include "ppddmount.h"

static char *progname;

static int ppddpassw(const char *file, int function, int master)
{
	struct crypt_control_block cblock;
	int                        reply, ffd, ok;
	unsigned char              ukey[PPDDKEYSIZE];
	Blowfish_Key               bkey;

	ok = 1;

	if ((ffd = open (file, O_RDWR)) < 0) {
		PPDDERROR(302)
		return 1;
	}

	check_access (file, ffd, 0);

	ok = checkpass(ffd, NULL, NULL, NULL, &cblock, NULL, ukey);

	if ((ok) &&
	   (function == 0)) {
		memcpy(&cblock.mkey[4], ukey, PPDDKEYSIZE);
		ok = newpass(-1, NULL, ukey, &cblock, master);
	}
	if ((ok) &&
	   (function == 1)) {
	    printf("Destroy data encryption keys on %s?\n", file);
	    printf("Enter Y to confirm: ");
	    reply = getc(stdin);
	    if ((reply != 'Y') && (reply != 'y')) {
		printf("No data overwritten. Operation cancelled by user.\n");
		ok = 0;
	    } else {
	        printf("OK - started.\n");
		memset(&cblock, 0, sizeof(cblock));
		ok = random_fill_fast (&cblock, sizeof(cblock));
	    }
	}
	if ((ok) &&
	   (function == 4)) {
	    printf("Erase the working pass phrase on %s?\n", file);
	    printf("Enter Y to confirm: ");
	    reply = getc(stdin);
	    if ((reply != 'Y') && (reply != 'y')) {
		printf("No data overwritten. Operation cancelled by user.\n");
		ok = 0;
	    } else {
		ok = newpass(-1, NULL, ukey, &cblock, -1);
	    }
	}
	if ((ok) && 
	    (function != 2)) {
	        Blowfish_ExpandUserKey(ukey, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, cblock.mkey, cblock.mkey, CBMLENGTH);
	        write_control_block(ffd, &cblock);
	}
        sync();
	memset(ukey, 0, PPDDKEYSIZE);
	memset(bkey, 0, sizeof(bkey));
	memset(&cblock, 0, sizeof(cblock));
	close(ffd);
	if ((ok) &&
	   (function == 0)) {
		if (master) {
			printf("New master pass phrase is in effect.\n");
		} else {
			printf("New working pass phrase is in effect.\n");
		}
	        sync();
		return 0;
	}
	if ((ok) &&
	   (function == 1)) {
    		printf("Data encryption keys destroyed.\n");
	        sync();
		return 0;
	}
	if ((ok) &&
	   (function == 2)) {
    		printf("All data decrypted.\n");
	        sync();
		return 0;
	}
	if ((ok) &&
	   (function == 4)) {
    		printf("Working pass phrase erased.\n");
	        sync();
		return 0;
	}
	exit (1);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-e | -m | -z] file \n", progname);
	fprintf(stderr, "      Use this to change pass phrase,\n");
	fprintf(stderr, "      or to destroy the data encryption keys.\n");
	fprintf(stderr, "      or to erase the working pass phrase.\n");
	fprintf(stderr, "      -e to erase the working pass phrase.\n");
	fprintf(stderr, "      -m to change master pass phrase.\n");
	fprintf(stderr, "      -z to destroy data encryption keys.\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c,function,master;

	progname = "ppddpassw";
	ppdd_intro(progname);

	function = 0;
	master = 0;
	while ((c = getopt(argc,argv,"demz")) != EOF) {
		switch (c) {
			case 'e':
                                function += 4;
				break;
			case 'm':
                                master = 1;
				break;
			case 'z':
                                function += 1;
				break;
			default:
				usage();
		}
	}
	if (argc < 2) usage();
	if ((function != 0) &&
	    (function != 1) &&
	    (function != 4)) usage();
	ppddpassw(argv[argc-1],function,master);
	return 0;
}

