/*
 * ppdncrypt.c - encrypt a ppdd filesystem without kernel suppor
 *
 * Copyright 1999 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "ppddmount.h"

static char *progname;

static int encrypt_all(int pgp, int convent, int fd, int ffd, 
				struct ppdd_keys *keys, 
				unsigned char *pmd5, unsigned char *cmd5)
{
	unsigned char 	iblock[32768];
	unsigned char 	oblock[32768];
	int 		i, rlength, wlength, count, reply, oseek, iseek;
	long 		dlength, mlength, block;
	struct MD5Context pmd5C;
	struct MD5Context cmd5C;

	printf("Starting encryption\n");

	MD5Init(&pmd5C);
	MD5Init(&cmd5C);

	iseek = oseek = 1024;
	if (convent) iseek = 0;
	
	if (pgp) oseek = 2048;

/* if conventional encryption we want to preserve the first 1024 bytes */

	lseek (fd, iseek, SEEK_SET);

/* for pgp we have already written the first 2048 bytes, for ppdd 1024 */

	lseek (ffd, oseek, SEEK_SET);

	count = 1024;
	dlength = 1024;
	block = 2;
	mlength = 0;

	while (1)
	{
	        rlength = read (fd, iblock, sizeof(iblock));
		if (rlength == 0) {
			MD5Final(pmd5, &pmd5C);
			MD5Final(cmd5, &cmd5C);
			for (i=0;i<16;i++) cmd5[i] ^= pmd5[i];
			mlength += dlength >> 20;
			return mlength;
		}
		if (rlength < 0) {
			PPDDERROR(180)
			return 0;
		}
		MD5Update(&pmd5C, iblock, rlength);
	        reply = transfer_bf(keys, PPDDENCRYPT, oblock,
					iblock, rlength >> 9, block);
		if (reply) {
			PPDDERROR(181)
			return 0;
		}
		MD5Update(&cmd5C, oblock, rlength);
		wlength = write (ffd, oblock, rlength);
		if (wlength != rlength) {
			PPDDERROR(182)
			return 0;
		}
		dlength += wlength;
		block += wlength >> 9;
		count--;
		if (count == 0) {
			mlength += dlength >> 20;
                        dlength &= 0x0fffff;
			printf("%ld Mb encrypted so far\n", mlength);
			count = 1024;
		}
	}
	return 0;
}

static int pgp_encrypt(int size, void *iblock, void *oblock)
{
	int 		apipe[2];
	int 		bpipe[2];
	int 		cstatus, reply, max, rlen;
	pid_t 		pid;
 	fd_set 		rfds, wfds, *prfds, *pwfds;
 	struct timeval 	tv;

	if (pipe (apipe)) {
		PPDDERROR(183)
		return 0;
	}
	if (pipe (bpipe)) {
		PPDDERROR(184)
		return 0;
	}

	if ((pid = fork())) {
	    if (pid < 0) {
		PPDDERROR(185)
		return 0;
	    }
	    close (apipe[0]);
	    close (bpipe[1]);
	    max = apipe[1];
	    if (max < bpipe[0]) max = bpipe[0];
	    max++;
	    prfds = &rfds;
	    pwfds = &wfds;

	    for(;;) {
		tv.tv_usec = 0;
		tv.tv_sec = 1;
  		FD_ZERO(&rfds);
  		FD_ZERO(&wfds);
  		FD_SET(bpipe[0], &rfds);
		FD_SET(apipe[1], &wfds);
		select(max, prfds, pwfds, NULL, &tv);
  		if (pwfds && (FD_ISSET(apipe[1], &wfds))) {
		    if (size != write (apipe[1], iblock, size)) {
			PPDDERROR(186)
			return 0;
	    	    }
	    	    close (apipe[1]);
		    pwfds = NULL;
		}
  		if (prfds && (FD_ISSET(bpipe[0], &rfds))) {
	    	    rlen = read (bpipe[0], oblock+4, size<<1);
	    	    if (rlen < 0) {
			PPDDERROR(187)
		  	return 0;
	            }
		    memcpy (oblock,&rlen,sizeof(rlen));
	    	    close (bpipe[0]);
		    prfds = NULL;
		}
	    	if (waitpid (pid, &cstatus, WNOHANG)) {
	    	    reply = WEXITSTATUS(cstatus);
	    	    return (!reply);
		}
	    }
	} else {
	    close (apipe[1]);
	    close (bpipe[0]);
	    dup2(apipe[0],0);
	    close (apipe[0]);
	    dup2(bpipe[1],1);
	    close (bpipe[1]);
	    execlp("pgpe","pgpe","-r", "backup","-f","+batchmode=1",
			"+compress=0","+textmode=0",NULL);
	    PPDDERROR(188)
	    exit (1);
	}
	return 0;
}


static int ppdncrypt(int pgp, int convent, int weak,
				const char *ifile, const char *ofile )
{
	struct ppdd_info           info;
	struct crypt_control_block cblock;
	struct ppdd_keys 	   keys;
	Blowfish_Key		   bkey;
	unsigned char		   ukey[PPDDKEYSIZE];
	unsigned char		   pgpblock[2048];
	unsigned short		   version;
	time_t			   time_created;
	int                        dlength, ifd, ofd, ok;

	memset(&info, 0, sizeof(info));
	strncpy(info.name, ifile, PP_NAME_SIZE);
	info.name[PP_NAME_SIZE-1] = 0;

	if ((ifd = open (ifile, O_RDONLY)) < 0) {
	    	PPDDERROR(189)
		return 1;
	}

	check_access (ifile, ifd, 0);

	if (ofile) {
		if ((ofd = open (ofile, O_WRONLY|O_TRUNC|O_CREAT, 0)) < 0) {
	    		PPDDERROR(190)
			return 1;
		}
	} else {
		if ((ofd = open (ifile, O_WRONLY)) < 0) {
	    		PPDDERROR(191)
			return 1;
		}
	}

	if ((fchown(ofd, 0, 0))) {
	  	PPDDERROR(192)
		return 1;
	}

	if ((fchmod(ofd, 0))) {
	  	PPDDERROR(193)
		return 1;
	}

	printf("Starting key generation - please wait\n\n");

	ok = random_fill_fast (&cblock, sizeof(cblock));
	if (!ok) {
		return 1;
	}

	ok = random_fill_fast (pgpblock, sizeof(pgpblock));
	if (!ok) {
		return 1;
	}

	lseek (ofd, 0, SEEK_SET);

	if (pgp) {
	    if (2048 != write(ofd, pgpblock, 2048)) {
	  	PPDDERROR(194)
	        ok = 0;
	    }
	    if (ok) {
	        setup_bf((struct ppdd_ukeys*)cblock.keys, &keys);
     		ok = dlength = encrypt_all (pgp, convent, ifd, ofd, &keys,
					cblock.pmd5, cblock.cmd5);
	    }
	    if (ok) {
		cblock.flags |= 3;
	        ok = pgp_encrypt(1024, &cblock, pgpblock);
	    }
	    if (ok) {
		printf("Finished: %d bytes encrypted\n", dlength);
	    }
	    if (ok) {
	        lseek (ofd, 0, SEEK_SET);
	        if (2048 != write(ofd, pgpblock, 2048)) {
	  	    PPDDERROR(195)
		    ok = 0;
	        }
	    }
	} else {
	    if (1024 != write(ofd, pgpblock, 1024)) {
	  	PPDDERROR(196)
	        ok = 0;
	    }
	    if (ok) {
	        ok = newpass(ukey, &cblock, 1);
	    }
	    if (ok) {
	        if (!weak) {
		    ok = random_fill_slow (cblock.keys, sizeof(cblock.keys));
	        }
	    }
	    if (ok) {
	        setup_bf((struct ppdd_ukeys*)cblock.keys, &keys);
     		ok = dlength = encrypt_all (pgp, convent, ifd, ofd, &keys,
					cblock.pmd5, cblock.cmd5);
	    }
	    if (ok) {
		cblock.flags |= 3;
	        Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, &cblock.flags, &cblock.flags,
								CB5LENGTH);
	        time_created = time(NULL);
	        memcpy(&cblock.time_created, &time_created, sizeof(time_t));
	        version = PP_VERSION;
	        memcpy(&cblock.file_version, &version, sizeof(unsigned short));
	        make_crc(&cblock,PPDD_CRC_MAKE);
	        Blowfish_ExpandUserKey(ukey, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, cblock.mkey, cblock.mkey, CBMLENGTH);
	    }
	    if (ok) {
	        if (write_control_block(ofd, &cblock)) {
	  	    PPDDERROR(197)
		    ok = 0;
	        }
	    }
	    if (ok) {
		printf("Finished: %d Mb encrypted\n", dlength);
	    }
	}

	memset(pgpblock, 0, sizeof(pgpblock));
	memset(&cblock, 0, sizeof(cblock));
	memset(&info, 0, sizeof(info));
	memset(&keys, 0, sizeof(keys));
	memset(&ukey, 0, sizeof(ukey));
	memset(&bkey, 0, sizeof(bkey));

	if (ok) return 0;
	return (1);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-cw] input-file|input-device output-file|output-device\n\
  %s -p[c] input-file|input-device output-file\n\
  %s -o[w] file|device\n\n", progname, progname, progname);
	fprintf(stderr, "      -c conventional encryption i.e the first\n");
	fprintf(stderr, "         1024 byte block is preserved\n");
	fprintf(stderr, "      -p use pgp to encrypt the data\n");
	fprintf(stderr, "      -w use weaker random key generation\n");
	fprintf(stderr, "      -o overwrite\n\n");
	fprintf(stderr, "      Use this to encrypt a ppdd filesystem\n");
	fprintf(stderr, "      without having ppdd support in the kernel\n\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int pgp = 0;
	int weak = 0;
	int convent = 0;
	int overwrite = 0;

	while ((c = getopt(argc,argv,"cwop")) != EOF) {
		switch (c) {
			case 'c':
                                convent = 1;
				break;
			case 'w':
                                weak = 1;
				break;
			case 'o':
                                overwrite = 1;
				break;
			case 'p':
                                pgp = 1;
				break;
			default:
				usage();
		}
	}

	progname = "ppdncrypt";
	if (ppdd_intro(progname)) usage ();
	c = pgp|weak|convent|overwrite;
	if (convent && overwrite) usage();
	if (pgp && overwrite) usage();
	if (weak && pgp) usage();
	if (overwrite) {
	    if (argc !=3) usage();
	    return (ppdncrypt(pgp, convent, weak, argv[argc-1],NULL));
	} else {
	    if (argc != (c + 3)) usage();
	    return (ppdncrypt(pgp, convent, weak, argv[argc-2],argv[argc-1]));
	}
}

