/*
 * ppddsetup.c - setup and control ppdd devices
 *
 * Copyright 1998,9 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
/* #include <linux/fs.h> this is needed just for the #define below
                         and it is now complaining about missing
                         prototypes so I've put the #define in directly
*/ 
#define BLKGETSIZE _IO(0x12,96) /* return device size */


#include "ppddmount.h"

#include <sys/ioctl.h>
static char *progname;

static int flush_ppdd(const char *device)
{
	struct	ppdd_info	info;
	int			fd;
	
	if ((fd = open(device, O_RDONLY)) < 0) {
		PPDDERROR(410)
		return 1;
	}
	if (ioctl(fd, PPDD_GET_MD5_KEY, &info) < 0) {
	    PPDDERROR(411)
	    close(fd);
	    return 1;
	}
	memset(&info,0,sizeof(info));
	close(fd);
	sync();
	return 0;
}

static int show_ppdd(const char *device)
{
	struct	ppdd_info	info;
	int			fd;
	long			size;
	
	if ((fd = open(device, O_RDONLY)) < 0) {
		PPDDERROR(401)
		return 1;
	}
	if (ioctl(fd, PPDD_GET_STATUS, &info) < 0) {
		PPDDERROR(402)
		close(fd);
		return 1;
	}
	if (ioctl(fd, BLKGETSIZE, &size) < 0) {
		PPDDERROR(403)
		close(fd);
		return 1;
	}
	printf("%s:\tdevice:inode [%04lx]:%ld (%s)\n\t\t%ld Mb, file version %01d.%01d, driver version %01d.%01d\n",
	       device, info.device, info.inode, info.name, size/2048, 
		info.file_version/10, info.file_version%10,
		info.driver_version/10, info.driver_version%10);
	close(fd);
	return 0;
}

static int del_ppdd(const char *device, int check)
{
	struct crypt_control_block	cblock;
	struct ppdd_info		info;
	Blowfish_Key			bkey;
	int 				fd, ffd;

	sync();

	if ((fd = open(device, O_RDONLY)) < 0) {
		PPDDERROR(404)
		return 1;
	}

	if (check) {
	    if (ioctl(fd, PPDD_GET_STATUS, &info) < 0) {
		PPDDERROR(405)
		close(fd);
		return 1;
	    }
	    if ((ffd = open(info.name, O_RDWR)) > -1) {
		if (read(ffd,&cblock,1024) != 1024) {
		    PPDDERROR(406)
		    close(fd);
		    close(ffd);
		    return 1;
		}
		if (ioctl(fd, PPDD_GET_MD5_KEY, &info) < 0) {
		    PPDDERROR(407)
		    close(fd);
		    close(ffd);
		    return 1;
		}
		sync();
		memcpy(cblock.keys, &info.keys, PPDDKEYSIZE);
		memset(&info,0,sizeof(info));
        	Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
        	Blowfish_Decrypt_ecb(bkey, &cblock.flags,
						&cblock.flags, CB5LENGTH);
		if (cblock.flags & 0x08) {
			random_fill_fast(&cblock.x5,sizeof(cblock.x5));
			random_fill_fast(&cblock.x6,sizeof(cblock.x6));
			checkcsum(0, 1, ffd, &cblock);
                	sync();
		}
		memset(&cblock,0,sizeof(cblock));
		memset(&bkey,0,sizeof(bkey));
	        close(ffd);
	    }
	}
	if (ioctl(fd, PPDD_CLR_FD, 0) < 0) {
	    PPDDERROR(408)
	    return 1;
	}

	close(fd);
	sync();
	return 0;
}


static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s -a                                     # automatic (/etc/ppddtab)\n\
  %s       ppdd_device                      # show details\n\
  %s -f    ppdd_device                      # flush\n\
  %s -d[c] ppdd_device                      # disconnect\n\
  %s -p[c] ppdd_device file passphrase-file # setup from file \n\
  %s -s[c] ppdd_device file                 # setup\n\
  %s -t    ppdd_device | file               # setup from /etc/ppddtab\n\
               use -c to check the ciphertext checksum\n\n"
		,progname,progname,progname,progname,
			progname,progname,progname);
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	int autotab = 0;
	int delete = 0;
	int flush = 0;
	int setup = 0;
	int pfile = 0;
	int tfile = 0;
	int check = 0;
	int count = 0;
	int blocksize = 0;

	char ppdd_device[32];
	char file[PP_NAME_SIZE];
	char pass1[120];
	char pass2[120];

	progname = "ppddsetup";

	if (ppdd_intro(progname)) usage();

	while ((c = getopt(argc,argv,"acdfpstB:")) != EOF) {
		switch (c) {
			case 'B':
                                blocksize = strtol(optarg,NULL,10);
				break;
			case 'a':
				autotab = 1;
                                count++;
				break;
			case 'c':
				check = 1;
				break;
			case 'd':
				delete = 1;
                                count++;
				break;
			case 'f':
				flush = 1;
                                count++;
				break;
			case 'p':
				pfile = 1;
                                count++;
				break;
			case 's':
				setup = 1;
                                count++;
				break;
			case 't':
				tfile = 1;
                                count++;
				break;
			default:
				usage();
		}
	}
	if (count > 1) usage();
	if ((count == 0) && (argc !=2)) usage();
	if (autotab && (argc != 2)) usage();
	if (autotab && check) usage();
	if (tfile && (argc != 3)) usage();
	if (tfile && check) usage();
	if (delete && (argc != 3)) usage();
	if (flush && (argc != 3)) usage();
	if (pfile && (argc != 5)) usage();
        if (blocksize) {
	    if (!setup) usage();
	    if (argc != 5) usage();
        } else {
	    if (setup && (argc != 4)) usage();
        }
	if (autotab) {
	    find_ppdd_entry("", ppdd_device,file,pass1,pass2,&check);
	    exit(0);
	}

	if (tfile) {
	    if (find_ppdd_entry(argv[optind],
				ppdd_device,file,pass1,pass2,&check)) {
	    	PPDDERROR(409)
		exit(1);
	    }
	    if (pass1[0]) {
		exit (set_ppdd(ppdd_device,file,pass1,pass2,NULL,
                                                         check,blocksize));
	    } else {
		exit (set_ppdd(ppdd_device,file,NULL,NULL,NULL,
                                                         check,blocksize));
	    }
	}

	if (pfile) {
	    exit (set_ppdd(argv[optind],argv[optind+1],NULL,NULL,argv[optind+2],
                                                         check,blocksize));
	} 

	if (setup) {
	    exit (set_ppdd(argv[optind],argv[optind+1],NULL,NULL,NULL,
                                                         check,blocksize));
	} 

	if (flush) {
	    exit (flush_ppdd(argv[optind]));
	}

	if (delete) {
	    exit (del_ppdd(argv[optind],check));
	} else {
	    exit (show_ppdd(argv[optind]));
	}
	exit(1);
}

