/*
 * ppddpassw.c - change password of an encrypted file system on a ppdd
 *
 * Copyright 1998,9 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "ppddmount.h"

static char *progname;

static int decrypt_all(int fd, int ffd)
{
	unsigned char block[32768];
	int rlength, wlength, count;
	long int dlength;
	
	lseek (fd,0,SEEK_SET);
	lseek (ffd,0,SEEK_SET);

	count = 1024;
	dlength = 0;

	while (1)
	{
	        rlength = read (fd, block, sizeof(block));
		if (rlength < 0) return (0);
		if (rlength == 0) return (1);
		wlength = write (ffd, block, rlength);
		if (wlength != rlength) return (0);
		dlength += wlength;
		count--;
		if (count == 0) {
			printf("%ld bytes decrypted\n", dlength);
			count = 20480;
		}
	}

}

static int ppddpassw(const char *device, const char *file, 
	                                int function, int master)
{
	struct ppdd_info           info;
	struct crypt_control_block cblock;
	int                        reply, fd, ffd, ok;
	unsigned char              ukey[PPDDKEYSIZE];
	Blowfish_Key               bkey;

	ok = 1;

	memset(&info, 0, sizeof(info));
	strncpy(info.name, file, PP_NAME_SIZE);
	info.name[PP_NAME_SIZE-1] = 0;

	if ((ffd = open (file, O_RDWR)) < 0) {
		PPDDERROR(302)
		return 1;
	}

	check_access (file, ffd, 0);

	if ((fd = open (device, O_RDWR)) < 0) {
		PPDDERROR(303)
		return 1;
	}

	check_access (device, fd, 0);

	if (ioctl(fd, PPDD_SET_FD, ffd) < 0) {
		PPDDERROR(304)
		return 1;
	}

	if (ioctl(fd, PPDD_GET_STATUS, &info) < 0) {
		PPDDERROR(305)
		ioctl(fd, PPDD_CLR_FD, 0);
		return 1;
	}

	if (info.driver_version != PP_VERSION) {
		fprintf(stderr,"Warning: kernel driver is version %d \n\
       but I am version %d !!!\n\n", info.driver_version, PP_VERSION);
	}

	ok = checkpass(ffd, NULL, NULL, NULL, &info, &cblock, ukey);

	if ((ok) &&
	   (function == 0)) {
		memcpy(&cblock.mkey[4], ukey, PPDDKEYSIZE);
		ok = newpass(ukey, &cblock, master);
	}
	if ((ok) && 
	    (function == 2)) {
	    printf("Decrypting existing data on %s (%s)?\n",
                          device, file);
	    printf("Enter Y to confirm: ");
	    reply = getc(stdin);
	    if ((reply != 'Y') && (reply != 'y')) {
		printf("No data decrypted. Operation cancelled by user.\n");
		ok = 0;
	    } else {
	        printf("OK - started.\n");
		sync();

		ok = set_new_keys(fd, ffd, &info, &cblock);

		if (ok) {
		    ok = decrypt_all(fd,ffd);
        	    sync();
	   	    ioctl(fd, PPDD_CLR_FD, 0);
	        }
	    }
	}
	if ((ok) &&
	   (function == 1)) {
	    printf("Destroy data encryption keys on %s (%s)?\n",
                          device, file);
	    printf("Enter Y to confirm: ");
	    reply = getc(stdin);
	    if ((reply != 'Y') && (reply != 'y')) {
		printf("No data overwritten. Operation cancelled by user.\n");
		ok = 0;
	    } else {
	        printf("OK - started.\n");
		memset(&cblock, 0, sizeof(cblock));
		ok = random_fill_fast (&cblock, sizeof(cblock));
	    }
	}
	if ((ok) &&
	   (function == 4)) {
	    printf("Erase the working pass phrase on %s (%s)?\n",
                          device, file);
	    printf("Enter Y to confirm: ");
	    reply = getc(stdin);
	    if ((reply != 'Y') && (reply != 'y')) {
		printf("No data overwritten. Operation cancelled by user.\n");
		ok = 0;
	    } else {
		ok = newpass(ukey, &cblock, -1);
	    }
	}
	if ((ok) && 
	    (function != 2)) {
	        Blowfish_ExpandUserKey(ukey, PPDDKEYSIZE, bkey);
	        Blowfish_Encrypt_ecb(bkey, cblock.mkey, cblock.mkey, CBMLENGTH);
	        write_control_block(ffd, &cblock);
	}
        sync();
   	ioctl(fd, PPDD_CLR_FD, 0);
	memset(ukey, 0, PPDDKEYSIZE);
	memset(bkey, 0, sizeof(bkey));
	memset(&info.keys, 0, sizeof(info.keys));
	memset(&cblock, 0, sizeof(cblock));
	close(fd);
	close(ffd);
	if ((ok) &&
	   (function == 0)) {
		if (master) {
			printf("New master pass phrase is in effect.\n");
		} else {
			printf("New working pass phrase is in effect.\n");
		}
	        sync();
		return 0;
	}
	if ((ok) &&
	   (function == 1)) {
    		printf("Data encryption keys destroyed.\n");
	        sync();
		return 0;
	}
	if ((ok) &&
	   (function == 2)) {
    		printf("All data decrypted.\n");
	        sync();
		return 0;
	}
	if ((ok) &&
	   (function == 4)) {
    		printf("Working pass phrase erased.\n");
	        sync();
		return 0;
	}
	exit (1);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [-d | -e | -m | -z] ppdd_device file \n", progname);
	fprintf(stderr, "      Use this to change pass phrase,\n");
	fprintf(stderr, "      or to decrypt the whole crypto filesystem,\n");
	fprintf(stderr, "      or to destroy the data encryption keys.\n");
	fprintf(stderr, "      or to erase the working pass phrase.\n");
	fprintf(stderr, "      -d to decrypt all data.\n");
	fprintf(stderr, "      -e to erase the working pass phrase.\n");
	fprintf(stderr, "      -m to change master pass phrase.\n");
	fprintf(stderr, "      -z to destroy data encryption keys.\n");
	exit(1);
}

int main(int argc, char **argv)
{
	int c,function,master;

	progname = "ppddpassw";
	ppdd_intro(progname);

	function = 0;
	master = 0;
	while ((c = getopt(argc,argv,"demz")) != EOF) {
		switch (c) {
			case 'd':
                                function += 2;
				break;
			case 'e':
                                function += 4;
				break;
			case 'm':
                                master = 1;
				break;
			case 'z':
                                function += 1;
				break;
			default:
				usage();
		}
	}
	if (argc < 3) usage();
	if ((function != 0) &&
	    (function != 1) &&
	    (function != 2) &&
	    (function != 4)) usage();
	ppddpassw(argv[argc-2],argv[argc-1],function,master);
	return 0;
}

