/*
 * ppddinit.c - initialise an encrypted file system on a ppdd
 *
 * Copyright 1998,9 Allan Latham <alatham@flexsys-group.com>
 *
 * Use permitted under terms of GNU Public Licence only.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include "ppddmount.h"

static char *progname;


static int encrypt_all(int fd, int ffd)
{
	unsigned char 	block[1024];
	int 		rlength, wlength, count;
	long 		dlength, mlength;


        lseek (ffd, 0, SEEK_SET);
        lseek (fd, 0, SEEK_SET);

	count = 32768;
	dlength = 0;
	mlength = 0;

	while (1)
	{
	        rlength = read (ffd, block, sizeof(block));
		if (rlength < 0) {
			memset(block,0,sizeof(block));
			return (0);
		}
		if (rlength == 0) {
		       	sync();
			mlength += dlength >> 20;
			printf("written %ld Mb. Finished.\n", mlength);
			memset(block,0,sizeof(block));
			return (1);
		}
		wlength = write (fd, block, rlength);
		if (wlength != rlength) return (0);
		count--;
		dlength += rlength;
		if (count == 0) {
			mlength += dlength >> 20;
			dlength &= 0x0fffff;
			printf("written %ld Mb so far\n", mlength);
			count = 32768;
		}
	}
}


static int ppddinit(const char *device, const char *file, 
	            const char *backup, int function, 
			long int bytes, int weak, int strong, int blocksize)
{
	struct ppdd_info           info;
	struct crypt_control_block cblock, copy_cb;
	int	                   reply, fd, ffd, bfd, i;
	long			   ok;
	char	                   message[80];
	char	                   block0[1024];
	unsigned short             version;
	time_t	                   time_created;
	Blowfish_Key               bkey;
	unsigned char              ukey[PPDDKEYSIZE];
	char			   filename[PP_NAME_SIZE];

/* check that cblock really is 1024 bytes */

	if (sizeof(cblock) != 1024) {
		sprintf(message,"cblock is %d bytes long", sizeof(cblock));
		PPDDEMESS(102,message)
		return 0;
	}
 
	if (file[0] == '/') {
		strcpy (filename, file);
	} else {
		if (!getcwd (filename, PP_NAME_SIZE-1)) {
			PPDDERROR(122)
			return 1;
		}
		strcat (filename, "/");
		strcat (filename, file);
	}

/* initial values */

	ok = 1;

	memset(&info, 0, sizeof(info));
	strncpy(info.name, filename, PP_NAME_SIZE);
	info.name[PP_NAME_SIZE-1] = 0;

/* open the host file */

	if ((ffd = open (filename, O_RDWR)) < 0) {
		PPDDERROR(104)
		return 1;
	}

	check_access (filename, ffd, 0);

/* open the device i.e. /dev/ppddx */

	if ((fd = open (device, O_RDWR)) < 0) {
		PPDDERROR(105)
		return 1;
	}

	check_access (device, fd, 0);

/* connect the file to the device */

	if (ioctl(fd, PPDD_SET_FD, ffd) < 0) {
		PPDDERROR(106)
		return 1;
	}

	if (blocksize) {
            if (ioctl(fd, PPDD_SET_BSIZE, blocksize) < 0) {
		PPDDERROR(123)
		ioctl(fd, PPDD_CLR_FD, 0);
		return 1;
	    }
	}

/* get the basic data about the device */

	if (ioctl(fd, PPDD_GET_STATUS, &info) < 0) {
		PPDDERROR(107)
		ioctl(fd, PPDD_CLR_FD, 0);
		return 1;
	}

/* version number check - our version vs the kernel driver */

	if (info.driver_version != PP_VERSION) {
		fprintf(stderr,"Warning: kernel driver is version %d \n\
       but I am version %d !!!\n\n", info.driver_version, PP_VERSION);
	}

/* fill the cblock with random bytes */

	printf("Starting key generation - please wait\n");

	ok = random_fill_fast (&cblock, sizeof(cblock));
	if (!ok) {
		ioctl(fd, PPDD_CLR_FD, 0);
		return 1;
	}

/* ask pass.c to get us a new pass phrase from the user and to
 * derive a key from it
 */

	ok = newpass(ukey, &cblock, 1);

/* if there is no -w switch use the real random device to generate keys */

	if ((weak == 0) && (ok)) {
	    ok = random_fill_slow (cblock.keys, (PPDDNKEYS + 1) * PPDDKEYSIZE);
	    if (!ok) {
		ioctl(fd, PPDD_CLR_FD, 0);
		return 1;
	    }
	}

	if (strong && ok) {
	        printf("\nYou asked for stronger keys so we need more randomness.\n");
		ok = random_fill_slow (cblock.keys, sizeof(cblock.keys));
		if (!ok) {
			ioctl(fd, PPDD_CLR_FD, 0);
			return 1;
		}
	}

/* tell the user what we are going to do and ask him to confirm */

	if (ok) { 
	    if (function == 2) {
		printf("Encrypt existing data on %s (%s)?\n",
                          device, filename);
	    } else {
		printf("Destroy existing data on %s (%s)?\n",
                          device, filename);
	    }
	    printf("Enter Y to confirm: ");
	    reply = getc(stdin);
	    if ((reply != 'Y') && (reply != 'y')) {
		printf("No data overwritten. Operation cancelled by user.\n");
		ioctl(fd, PPDD_CLR_FD, 0);
		ok = 0;
	    }
	    else {
	        printf("OK - started.\n");
	    }
	}

/* keys are now in the cblock - either from "random" or "urandom" */
/* so we now write the control block directly */

	if (ok) { 
	    time_created = time(NULL);
	    memcpy(&cblock.time_created, &time_created, sizeof(time_t));
	    version = PP_VERSION;
	    memcpy(&cblock.file_version, &version, sizeof(unsigned short));
	    make_crc(&cblock,PPDD_CRC_MAKE);
	    cblock.flags &= 0xfc;
	    Blowfish_ExpandUserKey(cblock.keys, PPDDKEYSIZE, bkey);
	    Blowfish_Encrypt_ecb(bkey, &cblock.flags, &cblock.flags, CB5LENGTH);
	    memcpy(&copy_cb, &cblock, 1024); 
	    Blowfish_ExpandUserKey(ukey, PPDDKEYSIZE, bkey);
	    Blowfish_Encrypt_ecb(bkey, cblock.mkey, copy_cb.mkey, CBMLENGTH);
	    write_control_block(ffd, &copy_cb);
	}
        sync();
	if ((ok) && (backup != NULL)) {
	    if ((bfd = open(backup, O_WRONLY|O_CREAT|O_TRUNC, 0)) < 0) {
		PPDDERROR(108)
	  	ok = 0;
	    }
            if (ok) { 
	        if ((fchown(bfd, 0, 0))) {
		    PPDDERROR(109)
	  	    ok = 0;
	        }
	    }
            if (ok) { 
	        if ((fchmod(bfd, 0))) {
		    PPDDERROR(119)
	  	    ok = 0;
	        }
	    }
            if (ok) { 
	        lseek (ffd, 0, SEEK_SET);
	        i = read (ffd, block0, 1024);
	        if (i < 0) {
		    PPDDERROR(120)
	  	    ok = 0;
		}
	    }
            if (ok) { 
	        lseek (bfd, 0, SEEK_SET);
	        i = write (bfd, block0, 1024);
	        if (i != 1024) {
		    PPDDERROR(121)
	  	    ok = 0;
		}
	    }
            memset(&block0, 0, sizeof(block0));
	    close(bfd);
	}
	if ((ok) && (function == 1)) {
		printf("Starting to write random data.\n");
		ok = random_fill_fast (&cblock, sizeof(cblock));
		if (ok) {
 	            ok = set_new_keys(fd, ffd, &info, &cblock);
		    if (ok) { 
			ok = encrypt_all(fd,ffd);
        		sync();
                    }
		}
	}
	if ((ok) && (function == 2)) {
 		ok = set_new_keys(fd, ffd, &info, &cblock);
		if (ok) { 
	                printf("Starting to encrypt data.\n");
			ok = encrypt_all(fd,ffd);
        		sync();
		}
	}
   	ioctl(fd, PPDD_CLR_FD, 0);
	memset(bkey, 0, sizeof(bkey));
	memset(ukey, 0, PPDDKEYSIZE);
	memset(&info.keys, 0, sizeof(info.keys));
	memset(&cblock, 0, sizeof(cblock));
       	sync();
	close(fd);
	close(ffd);
	if (ok) { 
	    	printf("Crypto file system initialised OK.\n");
		return 0;
	}
    	printf("Crypto file system initialisation failed.\n");
	exit (1);
}

static int usage(void)
{
	fprintf(stderr, "usage:\n\
  %s [ -r bytes | -x] [-b file] [-w] ppdd_device file|real_device \n", progname);
	fprintf(stderr, "       Use it as the very first step to create an ecrypted filesystem,\n");
	fprintf(stderr, "       -b create a backup of the encryption keys\n");
	fprintf(stderr, "       -r fills the device with \"bytes\" bytes of random data.\n");
	fprintf(stderr, "             \"bytes\" = zero means fill to overflowing\n");
	fprintf(stderr, "                       otherwise \"bytes\" must be > 2047.\n");
	fprintf(stderr, "       -w use weaker keys e.g. for test purposes\n");
	fprintf(stderr, "       -s use stronger keys i.e. more random input\n");
	fprintf(stderr, "       -x encrypts any existing data on the device\n");
	exit(1);
}

int main(int argc, char **argv)
{
	char *backup;
	int off,c,function,weak,strong;
	long int bytes;
	int blocksize = 0;

	progname = "ppddinit";
	ppdd_intro(progname);

	off = function = bytes = weak = strong = 0;
	backup = NULL;

	while ((c = getopt(argc,argv,"swxr:B:b:")) != EOF) {
		switch (c) {
			case 'B':
				blocksize = strtol(optarg,NULL,10);
				break;
			case 'b':
				backup = optarg;
				break;
			case 'r':
				bytes = strtol(optarg,NULL,10);
                                function += 1;
				break;
			case 's':
                                strong = 1;
				break;
			case 'w':
                                weak = 1;
				break;
			case 'x':
                                function += 2;
				break;
			default:
				usage();
		}
	}
	if (argc < 3) usage();
	if (strong && weak) usage();
	if ((function != 0) &&
	    (function != 1) &&
	    (function != 2)) usage();
	if ((bytes !=0) && (bytes < 2048)) usage();

	return (ppddinit(argv[argc-2], argv[argc-1], backup,
                     function, bytes, weak, strong, blocksize));
}

