/*
 *  linux/drivers/block/ppdd.c
 * 
 *  Copyright 1997,8,9 by Allan Latham <alatham@flexsys-group.com>  
 *
 *  Redistribution of this file is permitted under the GNU Public License.
 *
 *  Derived in part from the works of other authors who made those works
 *  available for use under GNU Public Licence.
 * 
 *  From "loop.c" original and modifications:
 *  Copyright 1993 by Theodore Ts'o  
 *                    Mitch Dsouza 28th May 1994
 *                    Andries Brouwer, 1 Feb 1996
 *                    <Vincent.Renardias@waw.com> Mar 20, 1997
 */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/major.h>
#include <linux/malloc.h>
#include <linux/ppdd.h>		

#define MAJOR_NR PPDD_MAJOR

#define DEVICE_NAME "ppdd"
#define DEVICE_REQUEST do_request
#define DEVICE_NR(device) (MINOR(device))
#define DEVICE_ON(device)
#define DEVICE_OFF(device)
#define DEVICE_NO_RANDOM
#define MAX_DISK_SIZE 1024*1024*1024
#define TIMEOUT_VALUE (6 * HZ)

#include <linux/blk.h>

#define MAX_PPDD 8

static struct ppdd_device ppdd_dev[MAX_PPDD];
static int ppdd_sizes[MAX_PPDD];
static int ppdd_blksizes[MAX_PPDD];


static void figure_ppdd_size(struct ppdd_device *lo)
{
	int	size;
	kdev_t lodev;

	if (S_ISREG(lo->inode->i_mode))
		size = (lo->inode->i_size) / BLOCK_SIZE;
	else {
		lodev = lo->device;
		if (blk_size[MAJOR(lodev)])
			size = blk_size[MAJOR(lodev)][MINOR(lodev)];
		else
			size = MAX_DISK_SIZE;
	}

	ppdd_sizes[lo->number] = size;
	ppdd_blksizes[lo->number] = BLOCK_SIZE;
}


static void do_request(void)
{
	int	real_block, block, offset, len, blksize, size, reply, ok;
	int 	sector, ppdd_minor, crcmd, cmd;
	char	*dest_addr;
	struct ppdd_device *lo;
	struct buffer_head *bh;
	struct request *current_request;

repeat:
	INIT_REQUEST;
	current_request = CURRENT;
	CURRENT = current_request->next;
	crcmd = current_request->cmd;
	ppdd_minor = MINOR(current_request->rq_dev);
	dest_addr = current_request->buffer;
	sector = current_request->sector;
	len = current_request->current_nr_sectors << 9;

	PPDD_SPIN_UNLOCK	

/* from here on the thread is not protected by spinlocks */

	ok = 0;

	if (ppdd_minor >= MAX_PPDD) {
		goto next_request;
	}

	lo = &ppdd_dev[ppdd_minor];

	if (!lo->inode)
		goto next_request;

	cmd = PPDDDECRYPT;
	if (crcmd == WRITE) {
		cmd = PPDDENCRYPT;
		if (lo->flags & PP_FLAGS_READ_ONLY)
			goto next_request;
	} else if (crcmd != READ) {
		printk("ppdd: %s error 001, unknown command (%d).",
			       kdevname(lo->device), crcmd);
		goto next_request;
	}

	blksize = ppdd_blksizes[ppdd_minor];
	if ((!(lo->flags & PP_FLAGS_USER_BSIZE)) && (blksize != BLOCK_SIZE)) {
	    printk("ppdd: %s error 005, something changed my blocksize to %d\n",
			       kdevname(lo->device), blksize);
	    ppdd_blksizes[ppdd_minor] = BLOCK_SIZE;
            goto next_request;
	}

	blksize = lo->blocksize;

	block = sector / (blksize >> 9);
	offset = (sector % (blksize >> 9)) << 9;

/* This loop gets repeated if the host blocksize is less than the request.
   This is the case only if the host is a file on a dos fs (512 bytes).
   I am not happy that this loop is completely safe
*/ 

	while (len > 0) {
		real_block = block;
		if (lo->flags & PP_FLAGS_DO_BMAP) {
			real_block = bmap(lo->inode, block);
			if (!real_block) {
			    printk("ppdd: %s error 002, block %d not found.\n",
					kdevname(lo->device), block);
			    goto next_request;
			}
		}

		bh = getblk(lo->device, real_block, blksize);
		if (!bh) {
			printk("ppdd: %s error 003, getblk(-, %d, %d) NULL.\n",
			       kdevname(lo->device), real_block, blksize);
			goto next_request;
		}
		if (!buffer_uptodate(bh) && ((crcmd == READ)
						|| offset || (len < blksize))) {
			bh->b_count++;
			ll_rw_block(READ, 1, &bh);
			wait_on_buffer(bh);
			bh->b_count--;
			if (!buffer_uptodate(bh)) {
				brelse(bh);
				goto next_request;
			}
		}
		size = blksize - offset;
		if (size > len)
			size = len;

	        reply = transfer_bf(lo->keys, cmd, (bh->b_data + offset), 
				dest_addr, (size >> 9),
				((block * (blksize >> 9)) + (offset >> 9)));

		if (reply) {
			printk("ppdd: %s error 004, transfer error block %d\n",
			       kdevname(lo->device), block);
			brelse(bh);
			goto next_request;
		}

		if ((crcmd == WRITE) 
		 && ((block * (blksize >> 9)) > 1)) {
			mark_buffer_uptodate(bh, 1);
			mark_buffer_dirty(bh, 1);
			bh->b_count++;
			ll_rw_block(WRITE, 1, &bh);
			bh->b_count--;
		}

		brelse(bh);
		dest_addr += size;
		len -= size;
		offset = 0;
		block++;
	}
	ok = 1;

/* before changing the request chain the thread locks io_request */

next_request:

	PPDD_SPIN_LOCK	

	current_request->next=CURRENT;
	CURRENT=current_request;
	end_request(ok);
	goto repeat;
}

static int ppdd_set_bsize(struct ppdd_device *lo, kdev_t dev, unsigned int arg)
{
	if (!lo->inode)
		return -ENXIO;

	if ((arg != 0) && (arg != 512) && (arg != 1024) && (arg != 2048) && (arg != 4096)) {
		printk("ppdd: %s error 011, block size %d not valid\n",
			       kdevname(lo->device), arg);
		return -EINVAL;
        }
	if (!(S_ISBLK(lo->inode->i_mode))) {
		printk("ppdd: %s error 011, Change blocksize only allowed on real devices\n",
			       kdevname(lo->device));
		return -EINVAL;
        }
        if (arg) { 
	        lo->blocksize = arg;
	        lo->flags |= PP_FLAGS_USER_BSIZE;
        } else {
	        lo->blocksize = BLOCK_SIZE;
	        lo->flags &= ~PP_FLAGS_USER_BSIZE;
        }
	ppdd_blksizes[lo->number] = lo->blocksize;
	return 0;
}

static int ppdd_set_fd(struct ppdd_device *lo, kdev_t dev, unsigned int arg)
{
	struct file	*file;
	struct inode	*inode;
	int		error;

	if (sizeof(unsigned long) != 4) {
		printk("ppdd: %s error 012, unsigned long is %d bytes, I expected 4\n",
			       kdevname(lo->device), sizeof(unsigned long));
		return -EINVAL;
        }

	if (arg >= NR_OPEN || !(file = current->files->fd[arg]))
		return -EBADF;
	if (lo->inode)
		return -EBUSY;
	PPDD_GET_INODE (inode,file)
	if (!inode) {
		return -EFAULT;
	}
	if (S_ISBLK(inode->i_mode)) {
		error = blkdev_open(inode, file);
		if (error) {
		    return error;
		}
		lo->device = inode->i_rdev;
		lo->blocksize = BLOCK_SIZE;
		lo->flags = 0;
	} else {
		if (S_ISREG(inode->i_mode)) {
		    lo->device = inode->i_dev;
		    lo->flags = PP_FLAGS_DO_BMAP;
		    if (blksize_size[MAJOR(lo->device)]) {
	    		lo->blocksize = 
			    blksize_size[MAJOR(lo->device)][MINOR(lo->device)];
	    		if (!lo->blocksize) {
	      		    lo->blocksize = BLOCK_SIZE;
			}
		    }
		} else {
		    return -EINVAL;
		}
	}

	if ((lo->blocksize & 511) != 0) {
		printk("ppdd: %s error 011, block size %d is not a multiple of 512\n",
			       kdevname(lo->device), lo->blocksize);
		return -EINVAL;
        }

	if (IS_RDONLY (inode) || is_read_only(lo->device)) {
		lo->flags |= PP_FLAGS_READ_ONLY;
		set_device_ro(dev, 1);
	} else {
		invalidate_inode_pages (inode);
		set_device_ro(dev, 0);
	}

	lo->keys = kmalloc(sizeof(struct ppdd_keys), GFP_KERNEL);
	if (lo->keys == NULL) {
		printk("ppdd: %s error 013, out of memory\n",
			       kdevname(lo->device));
		return -EINVAL;
	}
	lo->file_version = 0;
	lo->inode = inode;
	lo->inode->i_count++;
	figure_ppdd_size(lo);
	MOD_INC_USE_COUNT;
	invalidate_buffers(dev);
	return 0;
}


static int ppdd_clr_fd(struct ppdd_device *lo, kdev_t dev)
{
	if (!lo->inode)
		return -ENXIO;
	if (lo->refcnt > 1)	/* we needed one fd for the ioctl */
		return -EBUSY;
	if (!IS_RDONLY(lo->inode) && !is_read_only(lo->device)) 
		invalidate_inode_pages (lo->inode);
	if (S_ISBLK(lo->inode->i_mode))
		blkdev_release (lo->inode);
	iput(lo->inode);
	lo->device = 0;
	lo->inode = NULL;
	memset((char *)&(lo->keys->key), 0, sizeof(struct ppdd_keys));
	memset(lo->name, 0, PP_NAME_SIZE);
	ppdd_sizes[lo->number] = 0;
	kfree(lo->keys);
	lo->keys = NULL;
	invalidate_buffers(dev);
	MOD_DEC_USE_COUNT;
	return 0;
}


static int ppdd_set_status(struct ppdd_device *lo, kdev_t dev,
						struct ppdd_info *arg)
{
	struct 	ppdd_info info;
	int 	reply;

	if (!lo->inode)
		return -ENXIO;
	if (!arg)
		return -EINVAL;
	PPDD_COPY_FROM_USER (reply, info, arg)
	if (reply) return -EFAULT; 
	memcpy(lo->md5_bkey, &info.keys.md5_bkey, PPDDKEYSIZE);
	setup_bf(&info.keys, lo->keys);
	lo->file_version = info.file_version;
	strncpy(lo->name, info.name, PP_NAME_SIZE);
	memset(&info, 0, sizeof(info));
	figure_ppdd_size(lo);
	invalidate_buffers(dev);
	return 0;
}


static int ppdd_get_md5_key(struct ppdd_device *lo, kdev_t dev, 
						struct ppdd_info *arg)
{
	struct ppdd_info	info;
	int 			reply;
	
	if (!lo->inode)
		return -ENXIO;
	if (!arg)
		return -EINVAL;
	memset(&info, 0, sizeof(info));
	memcpy(&info.keys.md5_bkey, lo->md5_bkey, PPDDKEYSIZE);
	PPDD_COPY_TO_USER (reply, arg, info)
	if (reply) return -EFAULT;
	invalidate_inode_pages (lo->inode);
	invalidate_buffers(dev);
	return 0;
}


static int ppdd_get_status(struct ppdd_device *lo, struct ppdd_info *arg)
{
	struct ppdd_info	info;
	int 			reply;
	
	if (!lo->inode)
		return -ENXIO;
	if (!arg)
		return -EINVAL;
	memset(&info, 0, sizeof(info));
	info.number = lo->number;
	info.blocksize = lo->blocksize;
	info.device = kdev_t_to_nr(lo->inode->i_dev);
	info.inode = lo->inode->i_ino;
	info.rdevice = kdev_t_to_nr(lo->device);
	info.flags = lo->flags;
	info.driver_version = PP_VERSION;
	info.file_version = lo->file_version;
	strncpy(info.name, lo->name, PP_NAME_SIZE);
	PPDD_COPY_TO_USER (reply, arg, info)
	if (reply) return -EFAULT;
	return 0;
}


static int ppdd_swap_file_inode(struct ppdd_device *lo, kdev_t dev,
						  unsigned int arg)
{
	struct file	*file;
	struct inode	*inode;
	int		reply;


	if (!lo->inode) {
		printk("ppdd: %s warn 01\n", kdevname(lo->device));
		return -EINVAL;
	}

	reply = 0;

	if (arg >= NR_OPEN || !(file = current->files->fd[arg]))
		return -EBADF;

	PPDD_GET_INODE (inode,file);

	if (!inode) {
		printk("ppdd: %s warn 02\n", kdevname(lo->device));
		return -EINVAL;
	}

	if ((S_ISBLK(inode->i_mode)) 
	 && (lo->device = inode->i_rdev)) {
		reply = blkdev_open (inode, file);
		if (reply) {
			printk("ppdd: %s warn 03\n", kdevname(lo->device));
			return reply;	
		}
		invalidate_inode_pages (lo->inode);
		blkdev_release (lo->inode);
		iput(lo->inode);
		lo->device = inode->i_rdev;
		lo->inode = inode;
		lo->inode->i_count++;
	} else {
		printk("ppdd: %s warn 04\n", kdevname(lo->device));
		return -EINVAL;
	}
	return 0;
}


static int ppdd_ioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	struct ppdd_device *lo;
	int    dev, reply;

	if (!inode)
		return -EINVAL;
	if (MAJOR(inode->i_rdev) != MAJOR_NR) {
		return -ENODEV;
	}
	dev = MINOR(inode->i_rdev);
	if (dev >= MAX_PPDD)
		return -ENODEV;
	lo = &ppdd_dev[dev];
	switch (cmd) {
	case PPDD_SWAP_FILE_INODE:
		return ppdd_swap_file_inode(lo, inode->i_rdev, arg);
	case PPDD_SET_BSIZE:
		return ppdd_set_bsize(lo, inode->i_rdev, arg);
	case PPDD_SET_FD:
		return ppdd_set_fd(lo, inode->i_rdev, arg);
	case PPDD_CLR_FD:
		return ppdd_clr_fd(lo, inode->i_rdev);
	case PPDD_SET_STATUS:
		return ppdd_set_status(lo, inode->i_rdev, 
					(struct ppdd_info *) arg);
	case PPDD_GET_MD5_KEY:
		return ppdd_get_md5_key(lo, inode->i_rdev, 
					(struct ppdd_info *) arg);
	case PPDD_GET_STATUS:
		return ppdd_get_status(lo, (struct ppdd_info *) arg);
	case BLKGETSIZE:   /* Return device size */
		if (!lo->inode)
			return -ENXIO;
		if (!arg)  return -EINVAL;
		PPDD_PUT_USER_LONG (reply, ppdd_sizes[lo->number]<<1, arg)
		if (reply) return -EINVAL;
		return 0;
	default:
                return -EINVAL;
	}
	return 0;
}


static int ppdd_open(struct inode *inode, struct file *file)
{
	struct ppdd_device *lo;
	int	dev;

	if (!inode)
		return -EINVAL;
	if (MAJOR(inode->i_rdev) != MAJOR_NR) {
		return -ENODEV;
	}
	dev = MINOR(inode->i_rdev);
	if (dev >= MAX_PPDD)
		return -ENODEV;
	lo = &ppdd_dev[dev];
	lo->refcnt++;
	MOD_INC_USE_COUNT;
	return 0;
}


static int ppdd_release_int(struct inode *inode, struct file *file)
{
	struct ppdd_device *lo;
	int	dev;

	if (!inode)
		return -EINVAL;
	if (MAJOR(inode->i_rdev) != MAJOR_NR)
		return -ENODEV;
	dev = MINOR(inode->i_rdev);
	if (dev >= MAX_PPDD)
		return -ENODEV;
	fsync_dev(inode->i_rdev);
	lo = &ppdd_dev[dev];
	if (lo->refcnt > 0) {
		lo->refcnt--;
		MOD_DEC_USE_COUNT;
	}
	return 0;
}


#if LINUX_VERSION_CODE < 0x020100

static void ppdd_release_void(struct inode *inode, struct file *file)
{
	ppdd_release_int(inode, file);
}

#endif

static struct file_operations fops = {
	NULL,			/* lseek - default */
	block_read,		/* read - general block-dev read */
	block_write,		/* write - general block-dev write */
	NULL,			/* readdir - bad */
	NULL,			/* poll */
	ppdd_ioctl,		/* ioctl */
	NULL,			/* mmap */
	ppdd_open,		/* open */
#if LINUX_VERSION_CODE >= 0x020200
	NULL,			/* flush */
	ppdd_release_int	/* release */
#else
	ppdd_release_void	/* release */
#endif
};

/*
 * And now the modules code and kernel interface.
 */
#ifdef MODULE
#define ppdd_init init_module
#endif

int ppdd_init(void) {
	int	i;
	char    title[10];

	sprintf(title,"ppdd %d.%d", PP_VERSION/10, PP_VERSION%10);
	if (register_blkdev(MAJOR_NR, DEVICE_NAME, &fops)) {
		printk("ppdd: Error 006, Unable to get major number %d for ppdd device.\n",
		       MAJOR_NR);
		return -EIO;
	}
#ifndef MODULE
	printk("ppdd (c) 1998,9 Allan Latham <alatham@flexsys-group.com>\n");
	printk("     (c) 1993-98 Various other contributors\n");
	printk("     NO WARRANTY or LIABILITY is associated with this product.\n");
	printk("     See GNU Public Licence for full details.\n");
#endif
	printk("%s: registered device at major %d\n", title, MAJOR_NR);

	blk_dev[MAJOR_NR].request_fn = DEVICE_REQUEST;
	for (i=0; i < MAX_PPDD; i++) {
		memset(&ppdd_dev[i], 0, sizeof(struct ppdd_device));
		ppdd_dev[i].number = i;
	}
	memset(&ppdd_sizes, 0, sizeof(ppdd_sizes));
	memset(&ppdd_blksizes, 0, sizeof(ppdd_blksizes));
	blk_size[MAJOR_NR] = ppdd_sizes;
	blksize_size[MAJOR_NR] = ppdd_blksizes;

	return 0;
}

#ifdef MODULE
void
cleanup_module( void ) {
  if (unregister_blkdev(MAJOR_NR, DEVICE_NAME) != 0)
	printk("ppdd: Error 007, cleanup_module failed.\n");
}
#endif
