//---------------------------------------------------------------------------
#include <vcl.h>
#include <controls.hpp>
#pragma hdrstop

#include <stdio.h>
#include "registry.hpp"

#include "faxnt.h"

extern AnsiString   MyRegistryRoot;

WCHAR szLocalMonitor[]  = L"mGetty Fax Monitor";
WCHAR szPortDesc[]      = L"Fax port";
DWORD PortInfo1Strings[]={offsetof(PORT_INFO_1, pName),
                          (DWORD)-1};

DWORD PortInfo2Strings[]={offsetof(PORT_INFO_2, pPortName),
                          offsetof(PORT_INFO_2, pMonitorName),
                          offsetof(PORT_INFO_2, pDescription),
                          (DWORD)-1};


//============================================================================
//The main diffrence between the Windows 9x and Windows NT is that NT monitor
//is called with all the string being passed as UNISTRING or WIDE strings where
//Windows 9x is called using null terminated strings.
//
//All the function is implemented in commonfax.c. The Windows 9x calls these
//functions as per normal, but in the case of Windows NT we first have to
//convert the Wide strings to null terminated string before calling the common
//functions.
//============================================================================                           

//--------------------------------------------------------[ PackStrings ]---
// DESC: Utility function used to pack strings in the correct format for the
//       EnumPorts function
// ARGS: pSource       - Source string
//       pDest         - Dest string
//       DestOffsets   - Index into source
//       pEnd          - Index to the end of the last string
// RETURN:
//       The new index to the end of last string
//--------------------------------------------------------------------------
static LPBYTE PackStrings(LPWSTR *pSource,
                          LPBYTE pDest,
                          DWORD *DestOffsets,
                          LPBYTE pEnd)
{

   while ((int)*DestOffsets != -1)
   {
      if (*pSource)
      {
        pEnd-=wcslen(*pSource)*sizeof(WCHAR) + sizeof(WCHAR);
        *(LPWSTR *)(pDest+*DestOffsets)=wcscpy((LPWSTR)pEnd, *pSource);
      }//if
      else
       *(LPWSTR *)(pDest+*DestOffsets)=0;

      pSource++;
      DestOffsets++;
   }//while

   return pEnd;
}//PackStrings

//---------------------------------------------------------[ AssignPort ]---
// DESC: Utility function to insert a port into the port list structure.
// ARGS:
//       pPortName - The port name to insert
//       Level     - The level of pPortInfo (passed to EnumPorts)
//       pPortInfo - The Port info struct.  (passed to EnumPorts)
//       pEnd      - Index to the end of the last string
// RETURN:
//       The new index to the end of last string
//--------------------------------------------------------------------------
static LPBYTE AssignPort(LPWSTR pPortName,
                  DWORD   Level,
                  LPBYTE  pPortInfo,
                  LPBYTE   pEnd)
{
    LPWSTR         *SourceStrings,  *pSourceStrings;
    PPORT_INFO_2    pPort2 = (PPORT_INFO_2)pPortInfo;

    DWORD          *pOffsets;
    DWORD           Count;

    switch (Level)
    {
      case 1:
          pOffsets = PortInfo1Strings;
          break;

      case 2:
          pOffsets = PortInfo2Strings;
          break;

      default:
          return NULL;
    }//switch

    for ( Count = 0 ; (int)pOffsets[Count] != -1 ; ++Count ) {}


    SourceStrings = pSourceStrings = (LPWSTR *)GlobalAlloc(GMEM_FIXED, Count * sizeof(LPWSTR));
    if ( !SourceStrings )
    {
      OutputDebugString("LPRFAX: Failed to alloc port source strings.");
      return NULL;
    }//if
    ZeroMemory( pSourceStrings, Count * sizeof(LPWSTR) );

    switch (Level)
    {
      case 1:
          *pSourceStrings  =pPortName;
          break;

      case 2:
          *pSourceStrings++ = pPortName;
          *pSourceStrings++ = szLocalMonitor;
          *pSourceStrings   = szPortDesc;

          // How do i findout other types ???
          pPort2->fPortType = PORT_TYPE_WRITE;//PORT_TYPE_NET_ATTACHED;

          // Reserved
          pPort2->Reserved = 0;
          break;

      default:
          //DBGMSG(DBG_ERROR,("CopyIniPortToPort: invalid level %d", Level));
          return NULL;
    }//switch

    pEnd = PackStrings(SourceStrings, pPortInfo, pOffsets, pEnd);
    GlobalFree(SourceStrings);

    return pEnd;
}//AssignPort

//------------------------------------------------------[ WinNtEnumPorts ]---
// DESC:  Builds a list of all port managed by this port monitor.
//        see DDK help on EnumPorts
// ARGS:
//       pName      - Pointer to server name
//       Level      - Specifies type of port info structure
//       pPorts     - Pointer to buffer to receive array of port info. structures
//       cbBuf      - Specifies size, in bytes, of buffer
//       pcbNeeded  - Pointer to number of bytes stored into buffer (or required buffer size)
//       pcReturned - Pointer to number of PORT_INFO_*. structures stored into buffer
// RETURN:
//    TRUE if OK else return FALSE
//--------------------------------------------------------------------------
BOOL WINAPI WinNtEnumPorts(LPWSTR   pName,
                            DWORD   Level,
                            LPBYTE  pPorts,
                            DWORD   cbBuf,
                            LPDWORD pcbNeeded,
                            LPDWORD pcReturned)
{
  WCHAR        pPortName[256];
  LPBYTE       pEnd;
  int          i;
  TStringList *PortList;

  PortList = new TStringList();

  if (!GetPortList(PortList))
  {
    delete PortList;
    return NULL;
  }//if

  *pcbNeeded = 0;
  for (i = 0; i < PortList->Count; i++)
  {
    memset(pPortName, 0, sizeof(pPortName));
    MultiByteToWideChar( CP_ACP, 0, PortList->Strings[i].c_str(),
                                     PortList->Strings[i].Length(), pPortName,
                                     PortList->Strings[i].Length());

    switch (Level)
    {
      case 1:
        *pcbNeeded += sizeof(PORT_INFO_1) + (wcslen(pPortName) *sizeof(WCHAR)) + sizeof(WCHAR);
        break;

      case 2:
        *pcbNeeded += sizeof(PORT_INFO_2) + (sizeof(WCHAR) *
                      (wcslen(pPortName) + 1 +
                       wcslen(szLocalMonitor) + 1 +
                       wcslen(szPortDesc) + 1));

        break;

      default:
        SetLastError(ERROR_INVALID_LEVEL);
        delete PortList;
        return FALSE;
    }//switch
  }//for

  if (cbBuf < *pcbNeeded)
  {
    SetLastError(ERROR_INSUFFICIENT_BUFFER);
    delete PortList;
    return FALSE;
  }//if

  pEnd = pPorts+cbBuf;
  *pcReturned = 0;
  for (i = 0; i < PortList->Count; i++)
  {
    memset(pPortName, 0, sizeof(pPortName));
    MultiByteToWideChar( CP_ACP, 0, PortList->Strings[i].c_str(),
                                     PortList->Strings[i].Length(), pPortName,
                                     PortList->Strings[i].Length());


    pEnd = AssignPort(pPortName, Level, pPorts, pEnd);

    if (Level == 1)
      pPorts += sizeof(PORT_INFO_1);
    else
      pPorts += sizeof(PORT_INFO_2);
    (*pcReturned)++;
  }//for

  delete PortList;
  return TRUE;
}//EnumPorts

//-------------------------------------------------------[ WinNtOpenPort ]---
// DESC: Opens the specified port
//       see DDK help on OpenPort
// ARGS:
//    pName   - Points to a port name.
//    pHandle - Points to the memory location into which to put the handle to the opened port.
// RETURN:
//   Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL WINAPI WinNtOpenPort(LPWSTR  pName, PHANDLE pHandle)
{
   unsigned long ulLen;
   LPSTR         pNameA;
   bool          bResult;

   ulLen =  WideCharToMultiByte( GetACP(), 0, pName, -1, NULL, 0, NULL, NULL );
   pNameA = (LPSTR)GlobalAlloc(GPTR, ulLen);
   if (pNameA != NULL)
   {
     WideCharToMultiByte( GetACP(), 0, pName, ulLen, pNameA, ulLen, NULL, NULL );

     bResult = CommonOpenPort(pNameA, pHandle);

     GlobalFree((HGLOBAL)pNameA);
   }//if
   else
   {
     bResult = false;
   }//else

   return bResult;
}//WinNtOpenPort

//----------------------------------------------------[ WinNtStartDocPort ]---
// DESC: Performs the tasks it takes to start a print job on the specified port.
//       see DDK help on StartDocPort
// ARGS:
//   hPort        - The handle of the port to which the print job is being sent.
//   pPrinterName - Points to the zero-terminated string that is the name of the
//                  printer to which the job is being sent.
//   JobId        - Identifies the job being sent to the spooler.
//   Level        - The job level.
//   pDocInfo     - Points to the document information.
// RETURN:
//   Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL WINAPI WinNtStartDocPort(HANDLE  hPort,
                               LPWSTR  pPrinterName,
                               DWORD   JobId,
                               DWORD   Level,
                               LPBYTE  pDocInfo)
{
  bool            bResult;
  unsigned long   ulLen;
  LPSTR           pPrinterNameA;
  LPSTR           pDocName;

  while(1)
  {
    ulLen =  WideCharToMultiByte( GetACP(), 0, (LPWSTR)((PDOC_INFO_1)pDocInfo)->pDocName, -1, NULL, 0, NULL, NULL );
    pDocName = (LPSTR)GlobalAlloc(GPTR, ulLen);
    if (pDocName == NULL)
    {
      bResult = false;
      break;
    }//if
    WideCharToMultiByte( GetACP(), 0, (LPWSTR)((PDOC_INFO_1)pDocInfo)->pDocName, ulLen, pDocName, ulLen, NULL, NULL );

    ulLen =  WideCharToMultiByte( GetACP(), 0, pPrinterName, -1, NULL, 0, NULL, NULL );
    pPrinterNameA = (LPSTR)GlobalAlloc(GPTR, ulLen);
    if (pPrinterNameA == NULL)
    {
      bResult = false;
      break;
    }//if
    WideCharToMultiByte( GetACP(), 0, pPrinterName, ulLen, pPrinterNameA, ulLen, NULL, NULL );

    bResult = CommonStartDocPort(hPort, pPrinterNameA, JobId, pDocName);
    break;
  }//while

  if (pDocName != NULL)
    GlobalFree(pDocName);

  if (pPrinterNameA != NULL)
    GlobalFree(pPrinterNameA);

  return bResult;

}//WinNtStartDocPort

//------------------------------------------------------[ WinNtWritePort ]---
// DESC: Writes data to the printer.
//       see DDK help on WritePort
// ARGS:
//   hPort      - The handle of the port to which to write the data.
//   pBuffer    - Points to the buffer that contains the data to be written.
//   cbBuf      - The size of the buffer, in bytes, pointed to by the pBuffer parameter.
//   pcbWritten - Points to a memory location in which to put the number of bytes written
// RETURN:
//   Returns TRUE if the operation was successful.
//---------------------------------------------------------------------------
BOOL WINAPI WinNtWritePort(HANDLE  hPort,
                            LPBYTE  pBuffer,
                            DWORD   cbBuf,
                            LPDWORD pcbWritten)
{
  return CommonWritePort(hPort, pBuffer, cbBuf, pcbWritten);
}//WinNtWritePort

//-------------------------------------------------------[ WinNtReadPort ]---
// DESC: Reads data from the printer.
//       see DDK help on ReadPort
// ARGS:
//     hPort   - Handle of the port to read.
//     pBuffer - Points to the buffer into which to read the data.
//     cbBuf   - Specifies the size, in bytes, of the buffer pointed to by the
//               pBuffer parameter.
//     pcbRead - The number of bytes read is put into the memory location
//               pointed to by this parameter.
// RETURN:
//     Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL WINAPI WinNtReadPort(HANDLE hPort,
                          LPBYTE pBuffer,
                          DWORD  cbBuffer,
                          LPDWORD pcbRead)
{
  return CommonReadPort(hPort, pBuffer, cbBuffer, pcbRead);
}//WinNtReadport

//-----------------------------------------------------[ WinNtEndDocPort ]---
// DESC: Performs end-of-print-job tasks on the specified port.
//       see DDK help on EndDocPort
// ARGS:
//   hPort - Handle of the port the document was printed on.
// RETURN:
//   Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL WINAPI WinNtEndDocPort(HANDLE   hPort)
{
  return CommonEndDocPort(hPort);
}//WinNtEndDocPort

//------------------------------------------------------[ WinNtClosePort ]---
// DESC: Closes a previously-opened port when no printers are connected to the port. 
//       see DDK help on ClosePort
// ARGS:
//   hPort  The handle of a port previously opened by a call to the
//          OpenPort function.
// RETURN:
//   Returns TRUE if the operation is successful.
//---------------------------------------------------------------------------
BOOL WINAPI WinNtClosePort(HANDLE hPort)
{
  return CommonClosePort(hPort);
}//WinNtClosePort

//-------------------------------------------------------------[ AddPort ]---
// DESC: A port management function that adds a port to the list of currently
//       supported ports in the spooler environment.
//       see DDK help on ClosePort
// ARGS:
//    pName        - Points to a zero-terminated string that specifies the name
//                   of the server to which the port is connected. If this string
//                   is NULL, the port is local.
//    hWnd         - A handle to the parent window of the dialog box used by the
//                   AddPort function.
//    pMonitorName - Points to a zero-terminated string that specifies the monitor
//                   associated with the port.
// RETURN:
//   Returns TRUE if the operation is successful.
//---------------------------------------------------------------------------
BOOL WINAPI WinNtAddPort(LPWSTR   pName,
                          HWND     hWnd,
                          LPWSTR   pMonitorName)
{

  //CommonAddPort do not use pName and pMonitorName - pass NULL's
  return CommonAddPort(NULL, hWnd, NULL);
}//WinNtAddPort

//--------------------------------------------------[ WinNtConfigurePort ]---
// DESC: A port management function that configures the specified port.
//       see DDK help on ConfigurePort
// ARGS:
//    pName     - Points to a zero-terminated string that names the server on
//                which the port named in the pPortName parameter exists.
//    hWnd      - Handle to the parent window of the port-configuration dialog.
//    pPortName - Points to a zero-terminated string that names the port to be
//                configured.
// RETURN:
//    Returns TRUE if the operation is successful.
//---------------------------------------------------------------------------
BOOL WINAPI WinNtConfigurePort(LPWSTR   pName,
                                HWND     hWnd,
                                LPWSTR   pPortName)
{
  bool           bResult;
  unsigned long  ulLen;
  LPSTR          pPortNameA;
  LPSTR          pNameA;

  while(1)
  {
     ulLen =  WideCharToMultiByte( GetACP(), 0, pName, -1, NULL, 0, NULL, NULL );
     pNameA = (LPSTR)GlobalAlloc(GPTR, ulLen);
     if (pNameA == NULL)
     {
       bResult = false;
       break;
     }//if

     memset(pNameA, 0 , ulLen);
     WideCharToMultiByte( GetACP(), 0, pName, ulLen, pNameA, ulLen, NULL, NULL );


     ulLen =  WideCharToMultiByte( GetACP(), 0, pPortName, -1, NULL, 0, NULL, NULL );
     pPortNameA = (LPSTR)GlobalAlloc(GPTR, ulLen);
     if (pPortNameA == NULL)
     {
       bResult = false;
       break;
     }//if

     memset(pPortNameA, 0 , ulLen);
     WideCharToMultiByte( GetACP(), 0, pPortName, ulLen, pPortNameA, ulLen, NULL, NULL );


     bResult = CommonConfigurePort(pNameA, hWnd, pPortNameA);
     break;
  }//while

  if (pNameA != NULL)
     GlobalFree(pNameA);

  if (pPortNameA != NULL)
     GlobalFree(pPortNameA);
  return bResult;
}//WinNtConfigurePort

//-----------------------------------------------------[ WinNtDeletePort ]---
// DESC: A port management function that frees the port monitor of any
//       responsibility for managing the port or monitoring print jobs on the
//       port.
//       see DDK help on DeletePort
// ARGS:
//   pName     - Points to a zero-terminated string that names the machine on
//               which the port specified in the pPortName parameter exists.
//   hWnd      - Handle to the parent window of the port deletion dialog box.
//   pPortName - Points to a zero-terminated string that names the port to be
//               deleted.
// RETURN:
//    Returns TRUE if the operation is successful.
//--------------------------------------------------------------------------
BOOL WINAPI WinNtDeletePort(LPWSTR  pName,
                             HWND    hWnd,
                             LPWSTR  pPortName)
{
  bool           bResult;
  unsigned long  ulLen;
  LPSTR          pPortNameA;
  LPSTR          pNameA;

  while(1)
  {
     ulLen =  WideCharToMultiByte( GetACP(), 0, pName, -1, NULL, 0, NULL, NULL );
     pNameA = (LPSTR)GlobalAlloc(GPTR, ulLen);
     if (pNameA == NULL)
     {
       bResult = false;
       break;
     }//if

     memset(pNameA, 0 , ulLen);
     WideCharToMultiByte( GetACP(), 0, pName, ulLen, pNameA, ulLen, NULL, NULL );


     ulLen =  WideCharToMultiByte( GetACP(), 0, pPortName, -1, NULL, 0, NULL, NULL );
     pPortNameA = (LPSTR)GlobalAlloc(GPTR, ulLen);
     if (pPortNameA == NULL)
     {
       bResult = false;
       break;
     }//if

     memset(pPortNameA, 0 , ulLen);
     WideCharToMultiByte( GetACP(), 0, pPortName, ulLen, pPortNameA, ulLen, NULL, NULL );


     bResult = CommonDeletePort(pNameA, hWnd, pPortNameA);
     break;
  }//while

  if (pNameA != NULL)
     GlobalFree(pNameA);

  if (pPortNameA != NULL)
     GlobalFree(pPortNameA);
  return bResult;
}//WinNtDeletePort

//-----------------------------------------------------------[ MonitorEx ]---
// The Monitor structure to be passed by InitializePrintMonitor
//---------------------------------------------------------------------------
MONITOREX MonitorEx =
{
  sizeof(MONITOR),
  {
      WinNtEnumPorts,
      WinNtOpenPort,
      NULL,
      WinNtStartDocPort,
      WinNtWritePort,
      WinNtReadPort,
      WinNtEndDocPort,
      WinNtClosePort,
      WinNtAddPort,
      NULL,
      WinNtConfigurePort,
      WinNtDeletePort,
      NULL,
      NULL
  }
};

//----------------------------------------------[ InitializePrintMonitor ]---
//DESC: Initializes the print monitor.
//      see DDK help on InitializePrintMonitor
//ARGS:
//  pRegistryRoot - Points to the null-terminated string that identifies the
//                  registry root at which the print monitor can store internal
//                  information. The spooler will clean up this registry path
//                  when the print monitor is deleted.
//RETURN:
//   Upon success, InitializePrintMonitor returns an initialized MONITOREX
//   structure. If it fails, this function logs an error code and returns null.
//---------------------------------------------------------------------------
LPMONITOREX WINAPI _export InitializePrintMonitor(LPWSTR pRegistryRoot)
{
   unsigned long ulLen;
   LPSTR         lpstr;

   OutputDebugString("In InitializePrintMonitor");
   ulLen =  WideCharToMultiByte( GetACP(), 0, pRegistryRoot, -1, NULL, 0, NULL, NULL );
   lpstr = (LPSTR)GlobalAlloc(GPTR, ulLen);
   if (lpstr != NULL)
   {
     WideCharToMultiByte( GetACP(), 0, pRegistryRoot, ulLen, lpstr, ulLen, NULL, NULL );
     MyRegistryRoot = lpstr;
     GlobalFree((HGLOBAL)lpstr);
   }//if

   return &MonitorEx;
}//InitializePrintMonitor
//---------------------------------------------------------------------------
//
//                             T H E  E N D
//
//---------------------------------------------------------------------------
