//---------------------------------------------------------------------------
#include <vcl.h>
#include <controls.hpp>
#include "stdlib.h"
#pragma hdrstop

#include <stdio.h>
#include <winsock.h>
#include <winspool.h>
#include <shellapi.h>

#include "dialog.h"
#include "addport.h"
#include "registry.hpp"
#include "coverpg.h"

#include "commonfax.h"

AnsiString       MyRegistryRoot;  // Global to store the registry path passed
CRITICAL_SECTION SpoolerSection;  // Critical section handle


//----------------------------------------------------------[ Sendcmd ]---
// DESC: Utility function used to send a command to the LPD server
// ARGS: Socket   - Opened Socket handle
//       pcBuffer - Pointer to the data to be send
//       usSize   - The size of the data buffer
// RETURN:
//     Return TRUE if successful
//--------------------------------------------------------------------------
bool Sendcmd(SOCKET Socket, char *pcBuffer, int usSize)
{
  char cReceiveBuf[10];

  //Send the command
  if (send(Socket, pcBuffer, usSize, 0) == SOCKET_ERROR)
  {
    return false;
  }//if

  //Wait for ack
  if (recv(Socket, cReceiveBuf, 1, 0) == SOCKET_ERROR)
  {
    return false;
  }//if

  return (cReceiveBuf[0] == 0);
}//Sendcmd

//--------------------------------------------------------[ GetPortList ]---
// DESC: Utility function used to get a list of all the port managed by the
//       monitor
// ARGS: List   - The list in which the ports is to be returned
// RETURN:
//     Return TRUE if successful
//--------------------------------------------------------------------------
bool GetPortList(TStrings *List)
{
  TRegistry   *MyRegistry;

  if (MyRegistryRoot == "")
  {
    OutputDebugString("The printer driver was not initialized");
    return false;
  }//if

//  EnterCriticalSection(&SpoolerSection);

  MyRegistry = new TRegistry();
  MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
  try
  {
    if (MyRegistry->OpenKey(MyRegistryRoot+"\\Ports", true))
    {
       MyRegistry->GetKeyNames(List);
       MyRegistry->CloseKey();
    }//if
  }
  catch(...)
  {
     delete MyRegistry;
     LeaveCriticalSection(&SpoolerSection);
     return false;
  }//catch

//  LeaveCriticalSection(&SpoolerSection);
  delete MyRegistry;
  return true;
}//GetPortList

//------------------------------------------------------[ CommonPreview ]---
// DESC: Utility function used to preview a ps file
// ARGS: FilePath - The file to preview
// RETURN:
//     none
//--------------------------------------------------------------------------
void CommonPreview(const char *FilePath)
{
      char                Result[255] = "/";
      STARTUPINFO         Startup;
      PROCESS_INFORMATION ProcessInfo;
      HINSTANCE           h;
      AnsiString          Command;

      memset(&Startup, 0, sizeof(Startup));
      Startup.cb = sizeof(Startup);


      h = FindExecutable(FilePath, (const char *)&Result, (char *)&Result);
      if ((int)h > 32) {
         Command = AnsiString(Result) + " " + FilePath;
         CreateProcess(NULL, Command.c_str(),
                       NULL, NULL, FALSE,0,NULL, NULL,
                       &Startup,
                       &ProcessInfo);

         WaitForSingleObject(ProcessInfo.hProcess,  INFINITE);
      }//if

}//CommonPreview

//-----------------------------------------------------[ CommonPortExist ]---
// DESC: Utility function used to find out if a port already exists.
// ARGS: pPortName - The portname in question
// RETURN:
//     Return TRUE if exists
//--------------------------------------------------------------------------
bool CommonPortExist(char *pPortName)
{
  PORT_INFO_2   *pPort2,*ppPort2;
  unsigned long  i;
  unsigned long  ulNeeded;
  unsigned long  ulReturned;
  bool           bResult=false;

  OutputDebugString("Find out how much memory we need");
  //Find out how much memory we need
  EnumPorts(NULL, 2, NULL, 0, &ulNeeded, &ulReturned);
  if (GetLastError() != ERROR_INSUFFICIENT_BUFFER)
  {
    OutputDebugString("No printers is defined");
    return true;
  }
  OutputDebugString("Allocate the required buffer and call EnumPorts again");
  //Allocate the required buffer and call EnumPorts again
  pPort2 = (PORT_INFO_2 *)GlobalAlloc(GMEM_FIXED, ulNeeded);
  if (pPort2 != NULL && EnumPorts(NULL, 2, (LPBYTE)pPort2, ulNeeded, &ulNeeded, &ulReturned))
  {
    ppPort2 = pPort2;
    for ( i = 0; i < ulReturned; i++)
    {
      #ifdef DEBUG
      AnsiString Tmp = AnsiString("Checking ") + AnsiString(ppPort2->pPortName);
      OutputDebugString(Tmp.c_str());
      #endif
      if (AnsiString(ppPort2->pPortName) == AnsiString(pPortName))
      {
        bResult = true;
        break;
      }//if
      ppPort2 ++;
    }//for
  }//if

  GlobalFree(pPort2);
  return bResult;
}//CommonPortExist

//-----------------------------------------------------[ CommonOpenPort ]---
// DESC: Opens the specified port
//       see DDK help on OpenPort
// ARGS:
//    pName   - Points to a port name.
//    pHandle - Points to the memory location into which to put the handle to
//              the opened port.
// RETURN:
//   Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL CommonOpenPort(LPSTR  pName, PHANDLE pHandle)
{
   PortInfo      *pPortInfo;
   TRegistry     *MyRegistry;
   bool           bResult=true;

   pPortInfo = new PortInfo;
   pPortInfo->Comment = new TStringList;
   *pHandle = (PHANDLE)pPortInfo;

   pPortInfo->State = PS_OPEN;

   EnterCriticalSection(&SpoolerSection);

   // Retrieve the port information from registry
   MyRegistry = new TRegistry();
   MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
   pPortInfo->PortName = pName;
   try
   {
     AnsiString Msg;
     Msg = "RegRoot="+MyRegistryRoot+" pName="+pName;
     OutputDebugString(Msg.c_str());

     if (MyRegistry->OpenKey(MyRegistryRoot+"\\Ports\\"+pName, false)==true)
     {
        pPortInfo->Server   = MyRegistry->ReadString("Server");
		pPortInfo->Printer  = MyRegistry->ReadString("Printer")  ;
        pPortInfo->Port     = MyRegistry->ReadInteger("Port");
        pPortInfo->usSeqNo  = (unsigned short)MyRegistry->ReadInteger("SeqNo");
        MyRegistry->CloseKey();
     }//if
     else
     {
       MessageBox(NULL, "Error reading registry.","LPRFAX Error", MB_OK+MB_ICONERROR);
       bResult = false;
     }//else
   }//try
   catch(...)
   {
     bResult = false;
   }//catch

   LeaveCriticalSection(&SpoolerSection);
   delete MyRegistry;
   return bResult;
}//CommonOpenPort

//--------------------------------------------------[ CommonStartDocPort ]---
// DESC: Performs the tasks it takes to start a print job on the specified port.
//       see DDK help on OpenPort
// ARGS:
//   hPort        - The handle of the port to which the print job is being sent.
//   pPrinterName - Points to the zero-terminated string that is the name of the
//                  printer to which the job is being sent.
//   JobId        - Identifies the job being sent to the spooler.
//   Level        - The job level.
//   pDocInfo     - Points to the document information.
// RETURN:
//   Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL CommonStartDocPort(HANDLE  hPort,
                                LPSTR   pPrinterName,
                                DWORD   JobId,
                                LPSTR   pDocName)
{
  char            ControlFile[256];
  char            ControlLine[256];
  SOCKADDR_IN     sin;
  char            cLocalHost[128];
  short           i;
  struct hostent *pHost;
  PortInfo       *pPortInfo;
  TfrmDialog     *frmDialog;
  bool            bResult=true;
  TRegistry       *MyRegistry;
  AnsiString      Host;
  AnsiString      User;

  pPortInfo = (PortInfo *)hPort;

  if (pPortInfo->State != PS_OPEN)
    return FALSE;

  frmDialog = new TfrmDialog(NULL);
  EnterCriticalSection(&SpoolerSection);
  while(1)
  {
    //Open the printer
    if (!OpenPrinter(pPrinterName, &pPortInfo->hPrinter, NULL))
    {
	  bResult = false;
      break;
    }//if

	//Create the dialog as a child window of the current active window
//	frmDialog->ParentWindow =  GetForegroundWindow();
	frmDialog->ParentWindow =  NULL;

    //Read the user information from the registry
    MyRegistry = new TRegistry();
    MyRegistry->RootKey = HKEY_CURRENT_USER;
    try
    {
      if (MyRegistry->OpenKey("Software\\mgetty\\", true))
      {
        frmDialog->edtSenderName->Text = MyRegistry->ReadString("Sender");
        frmDialog->edtCompany->Text = MyRegistry->ReadString("Company");
        frmDialog->edtVoiceTel->Text = MyRegistry->ReadString("VoiceTel");
        frmDialog->edtFaxTel->Text = MyRegistry->ReadString("FaxTel");
        frmDialog->edtEmail->Text =  MyRegistry->ReadString("email");
        frmDialog->edtUserid->Text =  MyRegistry->ReadString("Userid");
        frmDialog->cbxNotify->Checked =  MyRegistry->ReadInteger("Notify");
        frmDialog->cbxPreview->Checked =  MyRegistry->ReadInteger("Preview");
        MyRegistry->CloseKey();
      }//if

      MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
      if (MyRegistry->OpenKey(MyRegistryRoot, false))
      {
         frmDialog->CoverPagelbx->Directory = MyRegistry->ReadString("Cover");
         MyRegistry->CloseKey();
      }//if
    }//try
    catch(...)
    {
    }//catch
	delete MyRegistry;

    //Let the user fill in all the information we need
    if (frmDialog->ShowModal() == 1)
    {

      //Save the user information in the registry
      MyRegistry = new TRegistry();
      MyRegistry->RootKey = HKEY_CURRENT_USER;
      try
      {
        if (MyRegistry->OpenKey("Software\\mgetty\\", true))
        {
          MyRegistry->WriteString("Sender",frmDialog->edtSenderName->Text);
          MyRegistry->WriteString("Company",frmDialog->edtCompany->Text);
          MyRegistry->WriteString("VoiceTel",frmDialog->edtVoiceTel->Text);
          MyRegistry->WriteString("FaxTel",frmDialog->edtFaxTel->Text);
          MyRegistry->WriteString("email",frmDialog->edtEmail->Text);
          MyRegistry->WriteString("Userid",frmDialog->edtUserid->Text);
          MyRegistry->WriteInteger("Notify",frmDialog->cbxNotify->Checked);
          MyRegistry->WriteInteger("Preview",frmDialog->cbxPreview->Checked);

          MyRegistry->CloseKey();
        }//if
      }//try
      catch(...)
      {
      }//catch
      delete MyRegistry;

      if (frmDialog->CoverPage->Checked)
      {
        pPortInfo->CoverPage = frmDialog->CoverPagelbx->FileName;
        pPortInfo->Topic = frmDialog->edtTopic->Text;
        pPortInfo->Comment->Text = frmDialog->MsgMemo->Lines->Text;
      }//if
	  else
        pPortInfo->CoverPage = "";

      pPortInfo->Preview = frmDialog->cbxPreview->Checked;
      pPortInfo->Dest = frmDialog->edtName->Text;
      pPortInfo->DestFax = frmDialog->FaxNo->Text;
      OutputDebugString(pPortInfo->CoverPage.c_str());

      //Create the socket
      pPortInfo->JobId  = JobId;
      pPortInfo->Socket = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
      sin.sin_family    = AF_INET;

      //Get the IP Address of the local machine
      memset(cLocalHost, 0 , sizeof(cLocalHost));
	  gethostname((char *)&cLocalHost, sizeof(cLocalHost));

	  // See if the host is specified in "dot address" form
	  sin.sin_addr.s_addr = inet_addr(cLocalHost);
	  if (sin.sin_addr.s_addr == INADDR_NONE)
	  {
		//See if we know where this host is
		pHost = gethostbyname(cLocalHost);
		if (pHost != NULL)
		{
		  memcpy( (char *)&sin.sin_addr, pHost->h_addr, pHost->h_length);
        }//if
        else
        {
          bResult = false;
          break;
        }//else
	  }//if
      else
      {
        bResult = false;
        break;
      }//else

	  //Make sure we as per RFC1179 bind to a port 721 - 731
	  for (i = 721; i < 731; i ++)
	  {
		sin.sin_port = htons(i);
		if (bind(pPortInfo->Socket, (sockaddr *)&sin, sizeof(sin)) == 0)
		{
		  break;
		}//if
	  }//for

	  if (i == 731)
	  {
		OutputDebugString("LPRFAX: No ports!");
		bResult = false;
		break;
	  }//if

	  #ifdef DEBUG
	  AnsiString Msg;
	  Msg = AnsiString("Connected from ") +  inet_ntoa(sin.sin_addr) +":"+i;
	  OutputDebugString(Msg.c_str());
	  OutputDebugString(pPortInfo->Server.c_str());
	  #endif

	  //Convert the IP Address of the FAX Server
	  sin.sin_addr.s_addr = inet_addr(pPortInfo->Server.c_str());
	  if (sin.sin_addr.s_addr == INADDR_NONE)
	  {
		// See if the host is specified in "dot address" form
		pHost = gethostbyname(pPortInfo->Server.c_str());
		if (pHost == NULL)
		{
		  OutputDebugString("LPRFAX: unknown server");
		  bResult = false;
		  break;
		}//if
		else
		{
		  memcpy( (char *)&sin.sin_addr, pHost->h_addr, pHost->h_length);
		}//if
	  }//if

	  #ifdef DEBUG
	  Msg = Msg + " to " +  inet_ntoa(sin.sin_addr) +":"+(int)pPortInfo->Port;
	  OutputDebugString(Msg.c_str());
	  #endif

	  // Connect to the socket to the PRINTER well-known port.
	  sin.sin_family = AF_INET;
	  sin.sin_port = htons((unsigned short)pPortInfo->Port );
	  if (connect (pPortInfo->Socket, (LPSOCKADDR) &sin, sizeof (sin)))
	  {
		OutputDebugString("Connect error");
		bResult = false;
		break;
	  }//if

	  //Tell the LPD server that we want to print - access verification
	  sprintf(ControlFile,"\2%s\n",pPortInfo->Printer.c_str());
	  if (!Sendcmd(pPortInfo->Socket, ControlFile, strlen(ControlFile)))
	  {
		OutputDebugString("LPRFAX: Access denied");
		bResult = false;
		break;
	  }//if

	  //Build the Controlfile message
	  sprintf(ControlFile,"dfA%03.3d%s",pPortInfo->usSeqNo, cLocalHost);
	  pPortInfo->Name = ControlFile;

      memset(ControlFile, 0, sizeof(ControlLine));

      User = frmDialog->edtEmail->Text.SubString(1,frmDialog->edtEmail->Text.Pos("@")-1);
      Host = frmDialog->edtEmail->Text.SubString(frmDialog->edtEmail->Text.Pos("@")+1,frmDialog->edtEmail->Text.Length());
      if (frmDialog->cbxNotify->Checked)
      {
        sprintf(ControlFile,"H%s\nP%s\nJ%s\nM%s\nN%s\nf%s\nU%s\n\0",
                          Host.c_str(),
                          frmDialog->edtUserid->Text.c_str(),
                          pPortInfo->DestFax.c_str(),
                          User.c_str(),
                          pDocName,
                          pPortInfo->Name.c_str(),
                          pPortInfo->Name.c_str());
      }//if
      else
      {
        sprintf(ControlFile,"H%s\nP%s\nJ%s\nN%s\nf%s\nU%s\n\0",
                          cLocalHost,
                          frmDialog->edtUserid->Text.c_str(),
                          pPortInfo->DestFax.c_str(),
                          pDocName,
                          pPortInfo->Name.c_str(),
                          pPortInfo->Name.c_str());
      }//else

      sprintf(ControlLine,"\2%d cfA%03.3d%s\n",(int)strlen(ControlFile), pPortInfo->usSeqNo, cLocalHost);
      if (Sendcmd(pPortInfo->Socket, ControlLine, strlen(ControlLine)) == false)
      {
        OutputDebugString("LPRFAX: Send error 2");
        bResult = false;
        break;
      }//if

      if (Sendcmd(pPortInfo->Socket, ControlFile, strlen(ControlFile)+1) == false)
      {
        OutputDebugString("LPRFAX: Send error 3");
        bResult = false;
        break;
      }//if

      //Get a temp file to store the printed document in before we send it to
      //the LPD server
      if (GetTempPath(MAX_PATH, pPortInfo->TempFilePath) > MAX_PATH)
      {
        OutputDebugString("LPRFAX: Error GetTempPath");
        bResult = false;
        break;
      }//if

      if (GetTempFileName(pPortInfo->TempFilePath, "SPL", 0, pPortInfo->TempFilePath) == 0)
      {
        OutputDebugString("LPRFAX: Error GetTempFileName");
        bResult = false;
        break;
      }//if

      pPortInfo->File = CreateFile(pPortInfo->TempFilePath,
                                   GENERIC_READ | GENERIC_WRITE,
                                      FILE_SHARE_READ,
                                  NULL,
                                  OPEN_EXISTING,
                                  FILE_ATTRIBUTE_TEMPORARY  |
                                  FILE_ATTRIBUTE_NORMAL |
                                  FILE_FLAG_SEQUENTIAL_SCAN,
                                  NULL);

      if (pPortInfo->File == INVALID_HANDLE_VALUE)
      {
        OutputDebugString("LPRFAX: Error opening spooler file");
        bResult = false;
        break;
      }//if

      pPortInfo->usSeqNo++;
      if (pPortInfo->usSeqNo > 999)
        pPortInfo->usSeqNo = 1;
    }//if
    else
    {
      // The user pressed cancel - We will cancel the job
      SetJob(pPortInfo->hPrinter, JobId, 0, NULL, JOB_CONTROL_CANCEL);
      CloseHandle(pPortInfo->hPrinter);
      bResult = true;
      pPortInfo->State = PS_OPEN;
    }//else

    break;
  }//while

  if (bResult == false)
  {
    pPortInfo->State = PS_OPEN;

    if (pPortInfo->Socket != NULL)
      closesocket(pPortInfo->Socket);

    if (pPortInfo->File != NULL)
      CloseHandle(pPortInfo->File);

    if (pPortInfo->hPrinter != NULL)
      CloseHandle(pPortInfo->hPrinter);
  }//if
  else
    pPortInfo->State = PS_PRINT;

  LeaveCriticalSection(&SpoolerSection);
  delete frmDialog;
  return bResult;
}//CommonStartDocPort

//-----------------------------------------------------[ CommonWritePort ]---
// DESC: Writes data to the printer.
//       see DDK help on WritePort
// ARGS:
//   hPort      - The handle of the port to which to write the data.
//   pBuffer    - Points to the buffer that contains the data to be written.
//   cbBuf      - The size of the buffer, in bytes, pointed to by the pBuffer parameter.
//   pcbWritten - Points to a memory location in which to put the number of bytes written
// RETURN:
//   Returns TRUE if the operation was successful.
//---------------------------------------------------------------------------
BOOL CommonWritePort(HANDLE  hPort,
							 LPBYTE  pBuffer,
							 DWORD   cbBuf,
							 LPDWORD pcbWritten)
{
  PortInfo *pPortInfo = (PortInfo *)hPort;

  if (pPortInfo == NULL &&  pPortInfo->State != PS_PRINT)
    return FALSE;

  //Write the data to the temp file
  if (WriteFile(pPortInfo->File, (char *)pBuffer, cbBuf, pcbWritten, NULL) == 0)
  {
    #ifdef DEBUG
    AnsiString Msg;
    Msg = AnsiString("LPRFAX: Error writting to temp file ") + (int) GetLastError();
    OutputDebugString(Msg.c_str());
    #endif
    return FALSE;
  }//if

  return TRUE;
}//CommonWritePort

//------------------------------------------------------[ CommonReadPort ]---
// DESC: Reads data from the printer. This function is not called as far
//       as I could see, maybe there is same special condition this is called.
//       see DDK help on ReadPort
// ARGS:
//     hPort   - Handle of the port to read.
//     pBuffer - Points to the buffer into which to read the data.
//     cbBuf   - Specifies the size, in bytes, of the buffer pointed to by the
//               pBuffer parameter.
//     pcbRead - The number of bytes read is put into the memory location
//               pointed to by this parameter.
// RETURN:
//     Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL CommonReadPort(HANDLE hPort,
                            LPBYTE pBuffer,
                            DWORD  cbBuffer,
                            LPDWORD pcbRead)
{
  PortInfo *pPortInfo = (PortInfo *)hPort;

  if (pPortInfo == NULL &&  pPortInfo->State != PS_PRINT)
    return FALSE;

  return FALSE;
}//CommonReadport

//----------------------------------------------------[ CommonEndDocPort ]---
// DESC: Performs end-of-print-job tasks on the specified port. See RFC1179 for
//       more information on how to communicate with a LPD server and see DDK
//       help on EndDocPort.
// ARGS:
//   hPort - Handle of the port the document was printed on.
// RETURN:
//   Returns TRUE if the operation is successful
//---------------------------------------------------------------------------
BOOL CommonEndDocPort(HANDLE  hPort)
{
  char           cBuffer[512];
  unsigned long  ulBytesRead;
  bool           bResult = true;
  PortInfo       *pPortInfo = (PortInfo *)hPort;
  char           FilePath[MAX_PATH]; 
  CoverPageCl    *Input;
  CoverPageCl    *Cover;
  TRegistry      *MyRegistry;
  THandleStream  *InputStream;
  TFileStream    *CoverStream;
  TStream        *OutputStream;  
  long           Page;
  AnsiString     Value;
  AnsiString     Token;

  if (pPortInfo == NULL)
    return FALSE;

  //Make sure that the print job is in the right state
  if (pPortInfo->State == PS_PRINT)
  {
    //Rewind the temp file and start to xmit it to the LPD server
    memset(cBuffer, 0 , sizeof(cBuffer));
    SetEndOfFile(pPortInfo->File);

    if (pPortInfo->CoverPage != "")
    {
      OutputDebugString("LPRFAX: Inserting Cover page");
      //Get a temp file to concatenate the coverpage and printed document in
      GetTempPath(MAX_PATH, FilePath);
	  GetTempFileName(FilePath, "SPL", 0, FilePath);
      strcat(FilePath,".ps");
      OutputDebugString(FilePath);

      InputStream = new THandleStream((int)pPortInfo->File);
      OutputStream = new TFileStream(FilePath, fmCreate);
      Input  = new CoverPageCl(InputStream);

      CoverStream =  new TFileStream(pPortInfo->CoverPage,fmOpenRead);
      Cover = new CoverPageCl(CoverStream);

      OutputDebugString("LPRFAX: Adding replacement values");
      //Add all the replacement variables.
      MyRegistry = new TRegistry();
      MyRegistry->RootKey = HKEY_CURRENT_USER;
      try
      {
        if (MyRegistry->OpenKey("Software\\mgetty\\", true))
        {
          Cover->Replace("(%%Sender)", "("+MyRegistry->ReadString("Sender")+")");
          Cover->Replace("(%%Company)", "("+MyRegistry->ReadString("Company")+")");
          Cover->Replace("(%%VoiceTel)", "("+MyRegistry->ReadString("VoiceTel")+")");
          Cover->Replace("(%%FaxTel)", "("+MyRegistry->ReadString("FaxTel")+")");
          Cover->Replace("(%%email)", "("+MyRegistry->ReadString("email")+")");
          MyRegistry->CloseKey();
        }//if

        MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
        if (MyRegistry->OpenKey(MyRegistryRoot, false))
        {
           frmDialog->CoverPagelbx->Directory = MyRegistry->ReadString("Cover");
           MyRegistry->CloseKey();
        }//if
	  }//try
      catch(...)
      {
      }//catch
      delete MyRegistry;

      OutputDebugString("LPRFAX: Calcualting pages");
      Cover->Replace("(%%Pages)", AnsiString("(")+(Cover->Pages()+Input->Pages())+")");
      Cover->Replace("(%%Dest)", "("+pPortInfo->Dest+")");
      Cover->Replace("(%%DestFax)", "("+pPortInfo->DestFax+")");
      Cover->Replace("(%%Topic)", "("+pPortInfo->Topic+")");

      OutputDebugString("LPRFAX: Comments");
      try
      {
        Cover->Replace("(%%Comment1)", "("+pPortInfo->Comment->Strings[0]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment1)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment2)", "("+pPortInfo->Comment->Strings[1]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment2)", "( )");
      }//catch

      try
	  {
        Cover->Replace("(%%Comment3)", "("+pPortInfo->Comment->Strings[2]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment3)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment4)", "("+pPortInfo->Comment->Strings[3]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment4)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment5)", "("+pPortInfo->Comment->Strings[4]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment5)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment6)", "("+pPortInfo->Comment->Strings[5]+")");
      }//try
      catch(...)
      {
		Cover->Replace("(%%Comment6)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment7)", "("+pPortInfo->Comment->Strings[6]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment7)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment8)", "("+pPortInfo->Comment->Strings[7]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment8)", "( )");
      }//catch

      try
      {
        Cover->Replace("(%%Comment9)", "("+pPortInfo->Comment->Strings[8]+")");
      }//try
      catch(...)
      {
        Cover->Replace("(%%Comment9)", "( )");
      }//catch

      OutputDebugString("LPRFAX: Processing docs");
      Input->NextToken();
	  Page = 1;
      do
      {
        Value = Input->Value.SubString(Input->Value.Pos("(atend)"), 7);
        if (Input->Token == "%%Pages:" && Value != "(atend)")
        {
          Input->Value = Page-1;
        }//if

        if (Input->Token == "%%Page:")
        {
          Input->Value = AnsiString(" ") + Page + " " + Page;
          Page++;
        }//if

        Token = Input->Token;

        Input->WriteToken(OutputStream);
        Input->NextToken();

        if (Token == "%%BeginProlog")
        {
          Page += Cover->InsertProcSet(OutputStream);
        }//if

        if (Token == "%%EndSetup")
        {
          Page += Cover->InsertCover(OutputStream);
          continue;
        }//if
      } while (Input->Token != "");

      if (pPortInfo->Preview)
         CommonPreview(FilePath);
    }//if
    else
    {
      OutputStream = new THandleStream((int)pPortInfo->File);
/*      if (pPortInfo->TempFilePath)
         ShellExecute(NULL,"open",
                      FilePath,
                      NULL, NULL,SW_SHOW );
*/
    }//else      

    sprintf(cBuffer,"\3%d %s\n",OutputStream->Size,pPortInfo->Name.c_str());
    if (Sendcmd(pPortInfo->Socket, cBuffer, strlen(cBuffer)))
    {
      OutputDebugString("LPRFAX: Sending doc");

      OutputStream->Position = 0;
      while (OutputStream->Position < OutputStream->Size)
      {
        ulBytesRead = min((int)sizeof(cBuffer),OutputStream->Size -OutputStream->Position);
        OutputStream->ReadBuffer(cBuffer, ulBytesRead);

        if (send(pPortInfo->Socket, cBuffer, ulBytesRead, 0)  == SOCKET_ERROR)
        {
          OutputDebugString("LPRFAX: Error writting to socket");
          bResult = false;
          break;
        }//if
      }//while

      if (Sendcmd(pPortInfo->Socket, "\0", 1))
        OutputDebugString("LPRFAX: doc send");
      else
        OutputDebugString("LPRFAX: Error sending doc");

      if (bResult)
        SetJob(pPortInfo->hPrinter, pPortInfo->JobId, 0, NULL, JOB_CONTROL_SENT_TO_PRINTER);

      if (pPortInfo->CoverPage != "")
      {
        delete Input;
        delete Cover;
        delete InputStream;
        delete CoverStream;
      }//if
      delete OutputStream;
      DeleteFile(FilePath);

    }//if
    else
    {
      OutputDebugString("LPRFAX: Error sending");
      bResult = false;
    }//else
  }//if

  closesocket(pPortInfo->Socket);
  CloseHandle(pPortInfo->File);
  DeleteFile(pPortInfo->TempFilePath);
  CloseHandle(pPortInfo->hPrinter);
  pPortInfo->State = PS_OPEN;

  return bResult;
}//CommonEndDocPort

//-----------------------------------------------------[ CommonClosePort ]---
// DESC: Closes a previously-opened port when no printers are connected to the port.
//       see DDK help on ClosePort
// ARGS:
//   hPort  The handle of a port previously opened by a call to the
//          OpenPort function.
// RETURN:
//   Returns TRUE if the operation is successful.
//---------------------------------------------------------------------------
BOOL CommonClosePort(HANDLE hPort)
{
   PortInfo   *pPortInfo = (PortInfo *)hPort;
   TRegistry  *MyRegistry;

   //Store the sequence number before closing the port
   MyRegistry = new TRegistry();
   MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
   try
   {
     if (MyRegistry->OpenKey(MyRegistryRoot+"\\Ports\\"+pPortInfo->PortName, true))
     {
       MyRegistry->WriteInteger("SeqNo",pPortInfo->usSeqNo);
       MyRegistry->CloseKey();
     }//if
   }//try
   catch(...)
   {
   }//catch
   delete MyRegistry;

   OutputDebugString("LPRFAX: In ClosePort");
   delete pPortInfo->Comment;
   delete pPortInfo;

   return TRUE;
}//CommonClosePort

//-------------------------------------------------------[ CommonAddPort ]---
// DESC: A port management function that adds a port to the list of currently
//       supported ports in the spooler environment. 
//       see DDK help on ClosePort
// ARGS:
//    pName        - Points to a zero-terminated string that specifies the name
//                   of the server to which the port is connected. If this string
//                   is NULL, the port is local.
//    hWnd         - A handle to the parent window of the dialog box used by the
//                   AddPort function.
//    pMonitorName - Points to a zero-terminated string that specifies the monitor
//                   associated with the port.
// RETURN:
//   Returns TRUE if the operation is successful.
//---------------------------------------------------------------------------
BOOL CommonAddPort(LPSTR   pName,
                           HWND    hWnd,
                           LPSTR   pMonitorName)
{
  TfrmAddPort *frmAddPort;
  AnsiString   PortName;
  TRegistry   *MyRegistry;
  bool         bResult = true;

  OutputDebugString("LPRFAX: In AddPort");
  MyRegistry = new TRegistry();
  frmAddPort = new TfrmAddPort(NULL);
  frmAddPort->ParentWindow = NULL;

  EnterCriticalSection(&SpoolerSection);
  while(1)
  {
    //Call the add dialog to get the new port's information. The dialog will
    //verify that the port does not exists.
    if (frmAddPort->ShowModal() == mrOk)
    {
       OutputDebugString("LPRFAX: ModalResult == mrOk ");
       if (frmAddPort->edtServerName->Text != "" && frmAddPort->edtFaxName->Text != "")
       {
         MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
         try
         {
           PortName = frmAddPort->edtServerName->Text + ":" +frmAddPort->edtFaxName->Text;

           //Now add the new entry
           if (MyRegistry->OpenKey(MyRegistryRoot+"\\Ports\\"+PortName, true))
           {
             MyRegistry->WriteString("Server", frmAddPort->edtServerName->Text);
             MyRegistry->WriteString("Printer", frmAddPort->edtFaxName->Text)  ;
             MyRegistry->WriteInteger("Port", frmAddPort->Port->Value);
             MyRegistry->WriteInteger("SeqNo",1);
             MyRegistry->CloseKey();
           }//if
         }//try
         catch(...)
         {
           bResult = false;
           break;
         }//catch
       }//if
       else
         bResult = false;
    }//if
    break;
  }//while
  delete  MyRegistry;
  delete  frmAddPort;

  LeaveCriticalSection(&SpoolerSection);
  return bResult;
}//CommonAddPort

//-------------------------------------------------[ CommonConfigurePort ]---
// DESC: A port management function that configures the specified port.
//       see DDK help on ConfigurePort
// ARGS:
//    pName     - Points to a zero-terminated string that names the server on
//                which the port named in the pPortName parameter exists.
//    hWnd      - Handle to the parent window of the port-configuration dialog.
//    pPortName - Points to a zero-terminated string that names the port to be
//                configured.
// RETURN:
//    Returns TRUE if the operation is successful.
//---------------------------------------------------------------------------
BOOL CommonConfigurePort(LPSTR  pName,
                                 HWND   hWnd,
                                 LPSTR  pPortName)
{
  TfrmAddPort   *frmAddPort;
  AnsiString     PortName;
  TRegistry     *MyRegistry;
  unsigned long  SeqNo;
  bool           bResult = true;

  OutputDebugString("LPRFAX: In ConfigurePortw");

  PortName = pPortName;

  OutputDebugString(PortName.c_str());

  MyRegistry = new TRegistry();
  frmAddPort = new TfrmAddPort(NULL);
  frmAddPort->ParentWindow = NULL;

  EnterCriticalSection(&SpoolerSection);
  while(1)
  {
    MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
    try
    {
       //Test to see if the port name already exists
      if (MyRegistry->OpenKey(MyRegistryRoot+"\\Ports\\"+PortName, false))
      {
		 frmAddPort->edtServerName->Text = MyRegistry->ReadString("Server");
		 frmAddPort->edtFaxName->Text = MyRegistry->ReadString("Printer");
		 frmAddPort->Port->Value = MyRegistry->ReadInteger("Port");
		 SeqNo = MyRegistry->ReadInteger("SeqNo");
		 MyRegistry->CloseKey();
	  }//if

	  //Display the add/change dialog
	  if (frmAddPort->ShowModal() == mrOk)
	  {
		  //Delete the key and create a new one
		  MyRegistry->DeleteKey(MyRegistryRoot+"\\Ports\\"+PortName);

		  PortName = frmAddPort->edtServerName->Text + ":" + frmAddPort->edtFaxName->Text;
		  if (MyRegistry->OpenKey(MyRegistryRoot+"\\Ports\\"+PortName, true))
		  {
			MyRegistry->WriteString("Server", frmAddPort->edtServerName->Text);
			MyRegistry->WriteString("Printer", frmAddPort->edtFaxName->Text);
            MyRegistry->WriteInteger("Port", frmAddPort->Port->Value);
            MyRegistry->WriteInteger("SeqNo",SeqNo);
            MyRegistry->CloseKey();
          }//if
          else
            bResult = false;
      }//if
    }//try
    catch(...)
    {
      bResult = false;
    }//catch
    break;
  }//while

  delete  MyRegistry;
  delete  frmAddPort;

  LeaveCriticalSection(&SpoolerSection);
  return bResult;
}//CommonConfigurePort

//----------------------------------------------------[ CommonDeletePort ]---
// DESC: A port management function that frees the port monitor of any
//       responsibility for managing the port or monitoring print jobs on the
//       port.
//       see DDK help on DeletePort
// ARGS:
//   pName     - Points to a zero-terminated string that names the machine on
//               which the port specified in the pPortName parameter exists.
//   hWnd      - Handle to the parent window of the port deletion dialog box.
//   pPortName - Points to a zero-terminated string that names the port to be
//               deleted.
// RETURN:
//    Returns TRUE if the operation is successful.
//--------------------------------------------------------------------------
BOOL CommonDeletePort(LPSTR   pName,
                              HWND    hWnd,
                              LPSTR   pPortName)
{
  AnsiString     PortName;
  AnsiString     Msg;
  TRegistry     *MyRegistry;
  bool           bResult = true;

  PortName = pPortName;

  EnterCriticalSection(&SpoolerSection);
  MyRegistry = new TRegistry();
  Msg = "Delete port " + PortName + " ?";
  //Make sure that the user knew what his was doing
  if (MessageBox(hWnd, Msg.c_str(), "Confimation",MB_YESNO+MB_ICONQUESTION) == IDYES)
  {
    MyRegistry->RootKey = HKEY_LOCAL_MACHINE;
    try
    {
      MyRegistry->DeleteKey(MyRegistryRoot+"\\Ports\\"+PortName);
    }//try
    catch(...)
    {
      bResult = false;
    }//catch
  }//if

  delete MyRegistry;
  LeaveCriticalSection(&SpoolerSection);

  return bResult;
}//CommonDeletePort
//---------------------------------------------------------------------------
//
//                             T H E  E N D
//
//---------------------------------------------------------------------------
