/* JavaCVS - The Hungry Java CVS Client/Server.
 * Copyright (C) 1997-1998 The Hungry Programmers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.hungry.javacvs.util;

/**
   This class corresponds to the CVSROOT variable.  It has access methods to
   get all the useful information out of the CVSROOT.
*/
public class CVSRoot
{
  /** corresponds to CVSROOTs with :pserver: */
  public final static int PSERVER_METHOD = 0;
  /** corresponds to CVSROOTs with :jserver: */
  public final static int JSERVER_METHOD = 1;
  /** corresponds to CVSROOTs with :kserver: */
  public final static int KSERVER_METHOD = 2;
  /** corresponds to CVSROOTs without any of the above patterns. */
  public final static int SERVER_METHOD = 3;

    /** Construct a CVSRoot corresponding to the given CVSROOT spec.
	@param root the CVSROOT string
	@throws CVSBadCVSRootError the CVSROOT was invalid.
    */
  public CVSRoot(String root) throws CVSBadCVSRootError
    {
      parseFromString(root);
    }

  /** Returns the access method */
  public int getMethod()
    {
      return m_method;
    }

  /** Returns the server-side root directory for */
  public String getRootDir()
    {
      return m_rootdir;
    }

  /** Returns the server hostname */
  public String getServer()
    {
      return m_server;
    }

  /** Returns the user's login */
  public String getLogin()
  {
    return m_login;
  }

  /** Converts from the string of type "pserver" to one of the *_METHOD constants */
  public static int stringToMethod(String method)
    {
      if (method.equals("pserver")) return PSERVER_METHOD;
      else if (method.equals("jserver")) return JSERVER_METHOD;
      else if (method.equals("kserver")) return KSERVER_METHOD;
      else if (method.equals("server")) return SERVER_METHOD;
      else return -1; /* XXX should throw something */
    }

  /** Converts from a *_METHOD constant to its corresponding string. */
  public static String methodToString(int method)
    {
      switch (method)
	{
	case PSERVER_METHOD: return "pserver";
	case JSERVER_METHOD: return "jserver";
	case KSERVER_METHOD: return "kserver";
	case SERVER_METHOD: return "server";
	default: return null;
	}
    }

  /** Converts the CVSRoot to the form from which it was parsed. */
  public String toString()
    {
      return ":" + methodToString(m_method) + ":" + m_login + "@" + m_server + ":" + m_rootdir;
    }

  /** 
      Populate this CVSRoot from the passed in string.
      @param root the CVSROOT string
  */
  private void parseFromString(String root) throws CVSBadCVSRootError
  {
    String method = null;
    String rootdir = null;
    String user = null;
    String host = null;

    if (root == null)
      {
	throw new CVSBadCVSRootError("null cvsroot property");
      }

    if (root.charAt(0) == ':')
      {
	int second_colon_index;

	if ((second_colon_index = root.indexOf(":", 1)) == -1)
	  {
	    throw new CVSBadCVSRootError("It's just bad");
	  }
	else
	  {
	    method = root.substring(1, second_colon_index);

	    root = root.substring(second_colon_index + 1);

	    m_method = stringToMethod(method);
	    /* Now we have an access method -- see if it's valid */
	    if (m_method != PSERVER_METHOD
		&& m_method != JSERVER_METHOD)
	      {
		throw new CVSBadCVSRootError("This release only supports the pserver and jserver access methods.");
	      }
	  }
      }
    else
      {
	throw new CVSBadCVSRootError("The cvsroot property must begin with ':{p,j}server:'");
      }

    int at_index;
    int third_colon_index;

    if ((at_index = root.indexOf('@')) != -1)
      {
	if (at_index != 0)
	  {
	    user = root.substring(0, at_index);
	    
	    root = root.substring(at_index + 1);
	  }
      }

    if ((third_colon_index = root.indexOf(':')) != -1)
      {
	if (third_colon_index != 0)
	  {
	    host = root.substring(0, third_colon_index);

	    root = root.substring(third_colon_index + 1);
	  }
      }

    if (root.length() != 0)
      rootdir = root;

    /* Do various sanity checks */
    if (host == null)
      {
	throw (new CVSBadCVSRootError("Missing hostname in CVSROOT"));
      }

    if (rootdir == null)
      {
	throw (new CVSBadCVSRootError("Missing directory in CVSROOT"));
      }

    if (user == null)
      user = System.getProperty("user.name");

    /* if we've made it here, things must be ok. */
    m_login = user;
    m_server = host;
    m_rootdir = rootdir;
  }

  /** The access method */
  private int m_method;
  /** The root directory */
  private String m_rootdir;
  /** The server hostname */
  private String m_server;
  /** The user's login */
  private String m_login;
}
