/* JavaCVS - The Hungry Java CVS Client/Server.
 * Copyright (C) 1997-1998 The Hungry Programmers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.hungry.javacvs.client.util;

import com.hungry.javacvs.util.*;

import java.util.*;
import java.io.*;

public class CVSProject
{
  private String m_name;
  private Hashtable m_entriesDictionary;

  public CVSProject(String project_name)
    {
      m_entriesDictionary = new Hashtable();

      m_name = project_name;
    }

  public CVSProject(String project_name, String archived_data)
    {
      try {
	m_entriesDictionary = parseArchivedData(archived_data);
      }
      catch (Exception e)
	{
	  System.err.println("Invalid project specification for project " + project_name);
	  m_entriesDictionary = new Hashtable();
	}

      m_name = project_name;
    }

  public void setName(String name)
    {
      m_name = name;
    }

  public String getName()
    {
      return m_name;
    }

  public Hashtable parseArchivedData(String archived_data) throws CVSInvalidProjectException
    {
      Hashtable table = new Hashtable();
      int equals, semicolon;

      archived_data = archived_data.trim();

      if (archived_data.charAt(0) != '{')
	throw new CVSInvalidProjectException("Archived data was not enclosed in braces");
      
      archived_data = archived_data.substring(1).trim();

      equals = archived_data.indexOf('=');

      while (equals != -1)
	{
	  String line, entry, tag;
	  semicolon = archived_data.indexOf(";");
	  
	  if (equals > semicolon)
	    throw new CVSInvalidProjectException("Syntax error"); /* XXX */

	  line = archived_data.substring(0, semicolon).trim();
	  equals = line.indexOf('=');

	  entry = line.substring(0, equals).trim();
	  tag = line.substring(equals + 1).trim();

	  CVSDebug.debug("Entry: " + entry + ", Tag: " + tag);
	  table.put(entry, tag);

	  archived_data = archived_data.substring(semicolon + 1);

	  equals = archived_data.indexOf('=');
	}

      return table;
    }

  public Enumeration getModuleEnumeration()
    {
      return m_entriesDictionary.keys();
    }

  public void addModule(String module_name)
    {
      addModuleWithTag(module_name, "HEAD");
    }

  public void addModuleWithTag(String module_name, String tag)
    {
      m_entriesDictionary.put(module_name, tag);
    }

  public String tagForModule(String module)
    {
      return m_entriesDictionary.get(module).toString();
    }

  public String archiveString()
    {
      String str;
      String entry;
      Enumeration keys;

      keys = m_entriesDictionary.keys();

      str = "{\n";

      while (keys.hasMoreElements() && (entry = (String)keys.nextElement()) != null)
	{
	  str = str + "    " + entry + " = " + (String)m_entriesDictionary.get(entry) + ";\n";
	}

      str += "}";

      return str;
    }

  private static Hashtable m_projectDictionary = new Hashtable();

  public static void readFromStream(InputStream s)
    {
      String new_project_name;
      String new_archived_data;
      PushbackInputStream is = new PushbackInputStream(s);
      byte b[] = new byte[1000];
      int i;

      try {
	while (is.available() != 0)
	  {
	    i = 0;
	    while ((b[i++] = (byte)is.read()) != '{') {}

	    is.unread(b[i - 1]);
	    i -= 2; // get rid of the space at the end of the name
	    new_project_name = new String(b, 0, i);
	    
	    CVSDebug.debug("project name: **** " + new_project_name);

	    i = 0;
	    while ((b[i++] = (byte)is.read()) != '}') {}

	    new_archived_data = new String(b, 0, i+1);

	    CVSDebug.debug("archived data: **** " + new_archived_data);
	    
	    addProject(new CVSProject(new_project_name, new_archived_data));

	    is.read(); // read in the \n
	  }
      }
      catch (Exception e)
	{
	}
    }

  public static void writeToStream(OutputStream f)
    {
      Enumeration p_enum = getProjectNameEnumeration();
      String project_name;
      PrintWriter writer = new PrintWriter(f);

      while (p_enum.hasMoreElements() 
	     && (project_name = (String)p_enum.nextElement()) != null)
	{
	  String bar = projectForName(project_name).archiveString();

	  writer.println(project_name + " " + bar);
	}
    }

  public static void addProject(CVSProject project)
    {
      m_projectDictionary.put(project.getName(), project);
    }

  public static Enumeration getProjectNameEnumeration()
    {
      return m_projectDictionary.keys();
    }

  public static CVSProject projectForName(String name)
    {
      return (CVSProject)m_projectDictionary.get(name);
    }

  public static void dumpProject(CVSProject project)
    {
      Enumeration m_enum = project.getModuleEnumeration();
      String module_name;

      CVSDebug.debug("Project:  '" + project.getName() + "'");
      CVSDebug.debug("---------");

      while (m_enum.hasMoreElements() 
	     && (module_name = (String)m_enum.nextElement()) != null)
	{
	  CVSDebug.debug(module_name +
			     " (tag '" +
			     project.tagForModule(module_name).toString() +
			     "')");
	}

      CVSDebug.debug("");
    }

  public static void dumpProjects()
    {
      Enumeration p_enum = getProjectNameEnumeration();
      String project_name;

      while (p_enum.hasMoreElements() 
	     && (project_name = p_enum.nextElement().toString()) != null)
	{
	  dumpProject(projectForName(project_name));
	}
    }

  public static void main(String args[])
    {
      CVSDebug.debug("Reading from System.in");
      CVSProject.readFromStream(System.in);
      CVSDebug.debug("Archiving to System.out");
      CVSProject.writeToStream(System.out);

      CVSProject.dumpProjects();
    }
}
