/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *
 * JavaCVS - The Hungry Java CVS Client/Server.
 * Copyright (C) 1997-1998 The Hungry Programmers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.hungry.javacvs.client;

import com.hungry.javacvs.client.i18n.*;
import com.hungry.javacvs.util.*;
import com.hungry.javacvs.client.util.*;
import com.hungry.javacvs.client.requests.*;
import com.hungry.javacvs.client.handlers.*;

import javax.swing.event.*; /* for EventListenerList */

import java.io.*;
import java.net.*;
import java.util.*;

public class CVSClient extends Thread implements CVSHandlerListener
{
  public CVSClient() {
    super("CVS Client Thread");

    /* set up the default handlers */
    CVSResponseHandler.setupHandlers();
    
    m_busy = false;
    
    m_pass = new CVSPass();

    m_queue = new CVSRequestQueue();
  }
  
  /**
   * lets the UI do whatever they want (including nothing) to display
   * status messages to the user.
   */
  private void fireStatusEvent(String msg) {
    fireClientEvent(new CVSClientStatusEvent(this, msg));
  }

  private void fireErrorEvent(String err) {
    fireClientEvent(new CVSClientErrorEvent(this, err));
  }

  private void fireMsgEvent(String msg) {
    fireClientEvent(new CVSClientMsgEvent(this, msg));
  }

  private void fireBusyChangeEvent(boolean busy) {
    fireClientEvent(new CVSClientBusyChangeEvent(this, busy));
  }

  private void fireConnectionClosedEvent() {
    fireClientEvent(new CVSClientConnectionClosedEvent(this));
  }

  private void fireRequestCompleteEvent() {
    fireClientEvent(new CVSClientRequestCompleteEvent(this));
  }

  public void fireClientEvent(CVSClientEvent e) {
    // Guaranteed to return a non-null array
    Object[] listeners = listenerList.getListenerList();
    
    for (int i = listeners.length-2; i>=0; i-=2) {
      if (listeners[i] == CVSClientListener.class) {
        ((CVSClientListener)listeners[i+1]).clientEvent(e);
      }
    }
  }

  /* managing listeners */
  public void addCVSClientListener(CVSClientListener l) {
    listenerList.add(CVSClientListener.class, l);
  }

  public void removeCVSClientListener(CVSClientListener l) {
    listenerList.remove(CVSClientListener.class, l);
  }
  
  /**
   * Sets the busy flag of the cvs client.
   */
  private synchronized void setBusy(boolean flag) {
    m_busy = flag;

    fireBusyChangeEvent(m_busy);
  }
  
  /**
   * Gets the busy flag of this cvs client.
   */
  public synchronized boolean getBusy() {
    return m_busy;
  }

  public synchronized void setDir(String dir) {
    /* if we're already there, don't re-read stuff. */
    if (m_dir != null 
        && (m_dir.equals(dir + File.separator)
            || m_dir.equals(dir)))
      return;
    
    /* read in <dir>/CVS/Entries, <dir>/CVS/Repository, and <dir>/CVS/Root,
       if they're available. */
    m_dir = dir;
    
    if (!m_dir.endsWith(File.separator))
      m_dir += File.separator;
    
    File cvs_repository_file = new File (m_dir + "CVS", "Repository");
    File cvs_root_file = new File (m_dir + "CVS", "Root");
    File cvs_entries_file = new File (m_dir + "CVS", "Entries");
    File cvs_ignore_file = new File (m_dir, ".cvsignore"); /* XXX platform specific? */
    
    /* if there's a CVS/Root file, read it in and override our
       currently set root with it. */
    try {
	  CVSDebug.debug("Trying to read " + cvs_root_file.toString());
	  
	  if (cvs_root_file.exists() && cvs_root_file.canRead()) {
	      FileInputStream istream = new FileInputStream(cvs_root_file);
	      byte buf[] = new byte[1000];
	      int length = istream.read(buf, 0, 1000);
          
	      m_root = new CVSRoot((new String(buf, 0, length)).trim());
          
	      CVSDebug.debug("Root file has been read.");
      }
	}
    catch (Exception e) {
	  System.out.println("Exception reading CVS/Root");
	  e.printStackTrace(System.out);
	}
    
    m_repository = m_root.getRootDir();
    
    /* if there's a CVS/Repository file, read it in and override our
       current idea of the repository (taken from the CVSRoot
       object, above. */
    try {
	  CVSDebug.debug("Trying to read " + 
                     cvs_repository_file.toString());
      
	  if (cvs_repository_file.exists() && cvs_repository_file.canRead()) {
        /* set the repository from the contents of the file. */
        FileInputStream istream =
          new FileInputStream(cvs_repository_file);
        byte buf[] = new byte[1000];
        int length = istream.read(buf, 0, 1000);
        
        m_repository = (new String(buf, 0, length)).trim();
        
        CVSDebug.debug("Repository file has been read.");
      }
	}
    catch (Exception e) {
	  System.out.println("Exception reading CVS/Repository");
	  e.printStackTrace(System.out);
	}
    
    /* we always create a CVSIgnore, so we ignore the default patterns. */
    m_ignore = new CVSIgnore();
    try {
	  CVSDebug.debug("Trying to read " + 
                     cvs_ignore_file.toString());
	  
	  if (cvs_ignore_file.exists() && cvs_ignore_file.canRead()) {
        m_ignore.readFromFile(cvs_ignore_file);
        
        CVSDebug.debug(".cvsignore file has been read.");
      }
	}
    catch (Exception e) {
	  System.out.println("Exception reading .cvsignore");
	  e.printStackTrace(System.out);
	}
    
    m_entries = new CVSEntries();
    m_entries.readFromFile(cvs_entries_file);
  }

  public synchronized String getDir() {
    return m_dir;
  }
  
  public synchronized void setRoot(String root) throws CVSBadCVSRootError {
    m_root = new CVSRoot(root);
  }
  
  public synchronized void setRoot(CVSRoot root) {
    m_root = root;
  }
  
  public synchronized CVSRoot getRoot() {
    return m_root;
  }
  
  public synchronized String getRepository() {
    return m_repository;
  }
  
  public synchronized CVSEntries getEntries() {
    return m_entries;
  }
  
  public synchronized CVSIgnore getIgnore() {
    return m_ignore;
  }

  public synchronized CVSPass getPass() {
    return m_pass;
  }

  /**
   * Opens a connection to the CVS server retrieved from our CVSRoot
   * object. Also, all authentication is driven from here.
   *
   * @throws IOException there was some generic IOException writing to or reading from the connection.
   * @throws UnknownHostException the CVS server name couldn't be resolved.
   * @throws CVSOperationComplete ??
   * @throws CVSAuthenticationError the user failed to authenticate with the cvs server.
   */
  private void initiateNewConnection()
    throws IOException, UnknownHostException,
           CVSOperationComplete, CVSAuthenticationError {
    CVSResponseHandler handler;
    
    m_conn = CVSClientConnection.createForRoot(m_root);
    
    fireStatusEvent(getResourceString(CVSStrings.OPENING_CONNECTION));
    
    m_conn.connect();
    
    fireStatusEvent(getResourceString(CVSStrings.AUTHENTICATING) +
                    " for " + m_root.toString());

    /* XXX this part here needs to be revisited in light of the
       jserver stuff. */
    String password = m_pass.getPasswordForRepository(m_root.toString());
    
    if (password == null) {
	  m_conn.close();
	  m_conn = null;
      fireErrorEvent(getResourceString(CVSStrings.INVALID_PASSWORD));
	  return;
	}
    else {
	  CVSDebug.debug("authenticating with password '" + password + "'");
      
	  m_conn.authenticate(password);
	}
    
    fireStatusEvent(getResourceString(CVSStrings.DETERMINING_VALID_REQUESTS));
    CVSValidRequestsRequest req_req = new CVSValidRequestsRequest();
    
    req_req.setConnection(m_conn);
    
    handler = req_req.makeRequest();
    
    if (handler != null) {
      handler.addCVSHandlerListener(this);
      handler.handleResponse();
    }
    
    if (CVSDebug.isDebuggingOn())
      CVSRequest.dumpValidRequests();
    
    /*
    ** tell the server what sort of responses we can handle.
    */
    fireStatusEvent(getResourceString(CVSStrings.SENDING_VALID_RESPONSES));
    CVSValidResponseRequest resp_req = new CVSValidResponseRequest();
    Enumeration enum = CVSResponseHandler.handlerNameEnumeration();
    
    resp_req.setConnection(m_conn);
    
    while (enum.hasMoreElements()) {
	  String response = (String)enum.nextElement();
      
	  resp_req.addArgument(response);
	}
    
    handler = resp_req.makeRequest();
    if (handler != null)
      handler.addCVSHandlerListener(this);
    
    fireStatusEvent(getResourceString(CVSStrings.SETTING_ROOT_AND_REPOSITORY));
    
    /*
    ** send the root request and handle it now 
    */
    CVSRootRequest root_req = new CVSRootRequest();
    root_req.addArgument(m_root.getRootDir());
    
    root_req.setConnection(m_conn);
    
    handler = root_req.makeRequest();
    if (handler != null) {
      handler.addCVSHandlerListener(this);
      handler.handleResponse();
    }
  }
  
  public void run() {
    CVSRequest req;
    CVSResponseHandler handler;

    while (true) {
	  req = m_queue.get();

	  if (req != null) {
        setBusy(true);
        
        try {
          if (m_conn == null) {
            initiateNewConnection();
            if (m_conn == null) {
              setBusy(false);
              fireStatusEvent(getResourceString(CVSStrings.DONE));
              continue;
            }
		  }
          
          req.setConnection(m_conn);
          
          fireStatusEvent(getResourceString(CVSStrings.SENDING_REQUEST));
          
          handler = req.makeRequest();
          
          if (handler != null) {
		    fireStatusEvent(getResourceString(CVSStrings.READING_RESPONSE));
		    
            handler.addCVSHandlerListener(this);
		    handler.handleResponse();
		    
		    if (handler.connectionShouldClose()) {
              m_conn.close();
              m_conn = null;

              fireConnectionClosedEvent();
            }
		  }
          
          fireStatusEvent(getResourceString(CVSStrings.DONE));
          
          setBusy(false);
          
          fireRequestCompleteEvent();
        }
        catch (Error e) {
		  e.printStackTrace(System.out);
          fireErrorEvent(e.toString());
		  fireStatusEvent(getResourceString(CVSStrings.DONE));
		  
		  setBusy(false);
		}
        catch (Exception e) {
		  e.printStackTrace(System.out);
          fireErrorEvent(e.toString());
          
		  fireStatusEvent(getResourceString(CVSStrings.DONE));
		  
		  setBusy(false);
		}
      }
	}
  }
  
  private String getResourceString(String key) {
    return CVSStrings.strings().getString(key);
  }

  /* for the CVSHandlerListener interface */
  public void handlerEvent(CVSHandlerEvent e) {
    int type = e.getType();

    switch (type) {
    case CVSHandlerEvent.ADD_ENTRY:
      m_entries.addEntriesLine(((CVSHandlerAddEntryEvent)e).getEntry());
      try {
        m_entries.saveToFile();
      }
      catch (IOException exc) {
        System.out.println("foo");
      }
      break;
    case CVSHandlerEvent.ERROR:
      fireErrorEvent(((CVSHandlerErrorEvent)e).getError());
      break;
    case CVSHandlerEvent.MSG:
      fireMsgEvent(((CVSHandlerMsgEvent)e).getMsg());
      break;
    case CVSHandlerEvent.UPDATE_FILE:
      CVSEntriesLine entry = ((CVSHandlerUpdateFileEvent)e).getEntry();
      
      File old_file, new_file;
      File backup_file;
      
      old_file = new File(entry.getName());
      backup_file = new File(".#" + entry.getName() + ".bak");
      
      if (old_file.exists())
        old_file.renameTo(backup_file);
      
      new_file = new File(".#jcvs-" + entry.getName() + "." + entry.getVersion());
      
      new_file.renameTo(old_file);
      
      backup_file.delete();
      break;
    case CVSHandlerEvent.INVALIDATE_REQS:
      CVSRequest.invalidateAllRequests();
      break;
    case CVSHandlerEvent.ENABLE_REQ:
      CVSRequest.enableRequest(((CVSHandlerEnableReqEvent)e).getReq());
      break;
    }
  }

  public CVSRequestQueue getQueue() {
    return m_queue;
  }

  private CVSRequestQueue m_queue;
  private boolean m_busy;
  private CVSRoot m_root;
  private CVSPass m_pass;
  private CVSEntries m_entries;
  private CVSIgnore m_ignore;
  private String m_repository;
  private String m_dir;
  private CVSClientConnection m_conn;
  /** our list of listeners. */
  protected EventListenerList listenerList = new EventListenerList();
}
