/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *
 * JavaCVS - The Hungry Java CVS Client/Server.
 * Copyright (C) 1997-1999 The Hungry Programmers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.hungry.javacvs.server.util;

import com.hungry.javacvs.server.requests.*;
import com.hungry.javacvs.server.util.*;
import com.hungry.javacvs.util.*;

import java.net.*;
import java.io.*;

public class CVSHandlerThread extends Thread
{
  public CVSHandlerThread(ThreadGroup threadgroup,
                          CVSConnectionQueue queue) {
    super (threadgroup, "CVS Handler");

    setState("created");

    m_queue = queue;
    m_state = new CVSHandlerThreadState();
  }

  public synchronized String toString() {
    return "CVS Handler [ " + m_stateName + " ]";
  }

  public synchronized void setState(String state) {
    m_stateName = state;
  }

  public void run() {
    while (true) {
      setState("waiting for connection");

      m_connection = m_queue.get();

      setState("authenticating user");

      int check_user = CVSConnection.AUTHENTICATION_FAILED;

      try {
        check_user = m_connection.checkUser();
        
        if (check_user == CVSConnection.VERIFICATION_FAILED
            || check_user == CVSConnection.AUTHENTICATION_FAILED) {
          CVSDebug.debug("Authentication failed...");
          shutDown();
        }
      }
      catch (Exception e) {
        System.err.println("authentication failed with exception: " + e);
        
        shutDown();
      }
      
      if (check_user == CVSConnection.VERIFICATION_SUCCESSFUL) {
        CVSDebug.debug("Verification successful...");
      }
      else {
        CVSDebug.debug("Authentication successful...");
      }

      try {
        handleRequests();
      }
      catch (IOException e) {
        System.err.println("io exception handling requests: " + e);

        shutDown();
      }
    }
  }

  public void handleRequests() throws IOException {
    String line;
    int space_index;

    setState("getting request");

    while (m_connection.isConnected()) {
      line = m_connection.readLine();

      if (line == null)
        return;

      CVSDebug.debug("handleRequest( " + line.trim() + ")");

      space_index = line.indexOf(' ');
      
      if (space_index == -1) space_index = line.indexOf('\n');

      if (space_index != -1)
        {
          String request_name = line.substring(0, space_index);
          CVSRequestHandler handler =
            CVSRequestHandler.getHandlerInstanceByName(request_name,
                                                       m_connection,
                                                       m_state);
          
          if (handler != null) {
            handler.setThreadState(m_state);
            
            // so the handler can read it.
            m_connection.unreadString(line.substring(space_index + 1));
            
            try {
              setState("handling request " + line.trim());
              handler.handleRequest();
            }
            catch (Exception e) {
              System.err.println("exception handling request: " + e);
              return;
            }
          }
        }
    }
  }
  
  public void shutDown() {
    try {
      m_connection.close();
    }
    catch (Exception e) {
      System.err.println("exception shutting down connection: " + e);
	}
  }

  public CVSHandlerThreadState getState() {
    return m_state;
  }
  
  public CVSConnection getConnection() {
    return m_connection;
  }

  private CVSHandlerThreadState m_state;
  private CVSServerConnection m_connection;
  private CVSConnectionQueue m_queue;
  private String m_stateName;
}
