/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *
 * JavaCVS - The Hungry Java CVS Client/Server.
 * Copyright (C) 1997-1998 The Hungry Programmers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package com.hungry.javacvs.client.ui;

import com.hungry.javacvs.util.*;
import com.hungry.javacvs.client.*;
import com.hungry.javacvs.client.util.*;
import com.hungry.javacvs.client.requests.*;
import com.hungry.javacvs.client.handlers.*;

import javax.swing.event.*; /* for EventListenerList */

import java.util.*;
import java.text.*;
import java.io.*;

/**
  * The CVSUI class serves as the superclass to all UI implementations
  * of the cvs client code.  It abstracts away the idea of requests
  * (which can make the UI code extremely large) and provides the
  * command abstraction.  The UI says "perform this command" and the
  * CVSUI class maps that command to a series of requests.  
  */
public class CVSUI implements CVSClientListener, CVSPassListener
{
  public static final int CMD_ADD = 0;
  public static final int CMD_ADMIN = 1;
  public static final int CMD_ANNOTATE = 2;
  public static final int CMD_CHECKOUT = 3;
  public static final int CMD_COMMIT = 4;
  public static final int CMD_DIFF = 5;
  public static final int CMD_EDIT = 6;
  public static final int CMD_EDITORS = 7;
  public static final int CMD_HISTORY = 8;
  public static final int CMD_IMPORT = 9;
  public static final int CMD_EXPORT = 10;
  public static final int CMD_INIT = 11;
  public static final int CMD_LOG = 12;
  public static final int CMD_LOGIN = 13;
  public static final int CMD_RDIFF = 14;
  public static final int CMD_RELEASE = 15;
  public static final int CMD_REMOVE = 16;
  public static final int CMD_STATUS = 17;
  public static final int CMD_TAG = 18;
  public static final int CMD_UNEDIT = 19;
  public static final int CMD_RTAG = 20;
  public static final int CMD_UPDATE = 21;
  public static final int CMD_WATCH = 22;
  public static final int CMD_WATCHERS = 23;

  /**
    * Creates a CVSUI object and starts a CVSClient thread
    */
  public CVSUI() {
    m_client = new CVSClient();
    
    m_client.addCVSClientListener(this);
    //    m_client.getPass().addCVSPassListener(this);

    System.out.println("starting client");
    m_client.start();
  }

  public void clientEvent(CVSClientEvent e) {
    int type = e.getType();

    switch(type) {
    case CVSClientEvent.BUSY_CHANGE:
      fireBusyStateChange(((CVSClientBusyChangeEvent)e).getBusy());
      break;
    case CVSClientEvent.CONNECTION_CLOSED:
      fireConnectionClosed();
      break;
    case CVSClientEvent.ERROR_EVENT:
      fireRequestError(((CVSClientErrorEvent)e).getError());
      break;
    case CVSClientEvent.MSG_EVENT:
      fireMessageText(((CVSClientMsgEvent)e).getMsg());
      break;
    case CVSClientEvent.STATUS_EVENT:
      fireStatusChange(((CVSClientStatusEvent)e).getStatus());
      break;
    }
  }

  public void passQuery(CVSPassQueryEvent e) {
	firePasswordPrompt();
  }

  /** 
    * Used internally to get the appropriate request (Lost, Modified, Unchanged) corresponding to a given file.
    * @param f the File we're dealing with.
    * @param entry the CVSEntriesLine that corresponds to this file (or null if there is no entry line for it).  */
  private CVSRequest getFileRequest(File f, CVSEntriesLine entry) {
    CVSRequest new_request = null;
    long entries_line_date = 0;
    
    try {
      entries_line_date = (DateFormat.getDateTimeInstance(DateFormat.LONG,
                                                          DateFormat.MEDIUM)
                           .parse(entry.getDate())).getTime();
    }
    catch (Exception e) {
	  e.printStackTrace();
	  return null;
	}
    
    if (entry == null) {
	  CVSQuestionableRequest questionable_req = new CVSQuestionableRequest();
      
	  questionable_req.addArgument(f.getName());
      
	  new_request = questionable_req;
	}
    else if (entry == null || !f.exists()) {
	  CVSLostRequest lost_req = new CVSLostRequest();
      
	  lost_req.addArgument(f.getName());
      
	  new_request = lost_req;
	}
    else if (f.lastModified() != entries_line_date)	{
	  CVSModifiedRequest modified_req = new CVSModifiedRequest();
      
	  modified_req.addArgument(f.getName());
	  modified_req.addArgument(new CVSMode(0777)); /* XXX where the hell do we get this? */
	  modified_req.addArgument(f);
	  
	  new_request = modified_req;
	}
    else {
	  CVSUnchangedRequest unchanged_req = new CVSUnchangedRequest();
      
	  unchanged_req.addArgument(f.getName());
	  
	  new_request = unchanged_req;
	}

    return new_request;
  }

  private void performArgRequest(CVSArgRequest req,
                                 Vector command_params,
                                 Vector files)  {
    Enumeration file_enumeration = files.elements();
    Enumeration param_enumeration = command_params.elements();
    String file;
    String param;
    CVSDirectoryRequest dir_req = new CVSDirectoryRequest();
    CVSUseUnchangedRequest useunchanged_req = new CVSUseUnchangedRequest();
    CVSEntriesLine entry;
    String starting_dir;
    boolean need_additional_dir_req = false;
    CVSRequestQueue queue = m_client.getQueue();

    /* XXX this next line shouldn't happen here. */
    m_client.setDir(System.getProperty("user.dir"));
    
    starting_dir = m_client.getDir();
    
    dir_req.addArgument(".");
    dir_req.addArgument(m_client.getRepository());
    
    queue.add(dir_req);
    queue.add(useunchanged_req);
    
    /* add in the command parameters */
    while (param_enumeration.hasMoreElements() 
           && (param = (String)param_enumeration.nextElement()) != null) {
	  req.addArgument(param);
	}
    
    while (file_enumeration.hasMoreElements() 
           && (file = (String)file_enumeration.nextElement()) != null) {
	  File f = new File(file);
	  
	  if (f.getParent() == null)
	    f = new File(".", file);
	  
	  CVSDebug.debug("Making request [ " + req.getClass().getName() + " ] for " + 
                     f.getPath() + f.getName());
	  
	  if (f.isDirectory())
	    break; // XX what do we do here?
	  
	  m_client.setDir(f.getParent());
	  
	  if (!starting_dir.equals(m_client.getDir())) {
        need_additional_dir_req = true;
        dir_req = new CVSDirectoryRequest();
        dir_req.addArgument(f.getParent());
        dir_req.addArgument(m_client.getRepository());
        
        queue.add(dir_req);
      }
	  
	  entry = m_client.getEntries().getEntryForFile(f.getName());
	  
      if (entry != null) {
	    CVSRequest file_request;
	    
        String entry_line = ("/" + entry.getName() +
                             "/" + entry.getVersion() + "///");
        CVSEntryRequest entry_req = new CVSEntryRequest();
        
        entry_req.addArgument(entry_line);
        queue.add(entry_req);
	    
	    file_request = getFileRequest(f, entry);
	    if (file_request != null)
	      queue.add(file_request);
	  }
	  
	  req.addArgument(file);
	}
      
    if (need_additional_dir_req) {
	  m_client.setDir(System.getProperty("user.dir"));
	  
	  dir_req = new CVSDirectoryRequest();
	  
	  dir_req.addArgument(".");
	  dir_req.addArgument(m_client.getRepository());
	  queue.add(dir_req);
	}
    
    queue.add(req);
  }
  
  private void performHistory(Vector command_params,
                              Vector files) {
    CVSHistoryRequest req = new CVSHistoryRequest();
    CVSRequestQueue queue = m_client.getQueue();
    
    queue.add(req);
  }

  private void addDotDotRequest(Vector files) {
    /* look through the list of files for ".."'s, and set the Max-dotdot */
    int max_dotdot = 0;
    Enumeration file_enumeration = files.elements();
    String file;
    CVSRequestQueue queue = m_client.getQueue();
    
    while(file_enumeration.hasMoreElements()
          && (file = (String)file_enumeration.nextElement()) != null) {
	  if (file.startsWith("../"))
	    {
	      int dotdot = 0;
	      while (file.startsWith("../")) {
            dotdot ++;
            file = file.substring(3 /* "../".length() */);
          }
	      if (dotdot > max_dotdot)
            max_dotdot = dotdot;
	    }
	}
    
    if (max_dotdot > 0) {
	  CVSMaxdotdotRequest req = new CVSMaxdotdotRequest();
	  
	  req.addArgument(new Integer(max_dotdot).toString());
	  
	  queue.add(req);
	}
  }
  
  /**
   * The main entry point used by subclasses of CVSUI.
   *
   * @param cvs_params The parameters to the client.  These normally appear on the left of the command in the TTY client.
   * @param cvs_command One of the constants CMD_*.
   * @param command_params The parameters to the command.  These normally appear on the right of the command in the TTY client.
   * @param files an array of java.lang.Strings representing the filenames being operated on.  
   */
  public void performCommandOnFiles(Vector cvs_params,
                                    int cvs_command,
                                    Vector command_params,
                                    Vector files) {
    addDotDotRequest(files);
    
    /* now switch on the command number and actually do stuff. */
    switch(cvs_command) {
    case CMD_COMMIT:
	  performArgRequest(new CVSCommitRequest(), command_params, files);
	  break;
	case CMD_DIFF:
	  performArgRequest(new CVSDiffRequest(), command_params, files);
	  break;
	case CMD_UPDATE:
	  performArgRequest(new CVSUpdateRequest(), command_params, files);
	  break;
	case CMD_ADD:
	  performArgRequest(new CVSAddRequest(), command_params, files);
	  break;
	case CMD_STATUS:
	  performArgRequest(new CVSStatusRequest(), command_params, files);
	  break;
	case CMD_HISTORY:
	  performHistory(command_params, files);
	  break;
	case CMD_LOG:
	  performArgRequest(new CVSLogRequest(), command_params, files);
	  break;
	case CMD_REMOVE:
	  performArgRequest(new CVSRemoveRequest(), command_params, files);
	  break;
	default:
	  CVSDebug.debug("unhandled case in switch statement.\n");
	  System.exit(0);
	}
  }

  public CVSClient getClient() {
    return m_client;
  }
  
  public void fireBusyStateChange(boolean busy) {
    fireUIChange(new CVSUIBusyStateEvent(this, busy));
  }

  public void fireMessageText(String text_message) {
    fireUIChange(new CVSUIMessageEvent(this, text_message));
  }

  public void fireStatusChange(String status_message) {
    fireUIChange(new CVSUIStatusEvent(this, status_message));
  }

  public void firePasswordPrompt() {
    fireUIChange(new CVSUIPasswordEvent(this));
  }

  public void fireRequestError(String error_message) {
    fireUIChange(new CVSUIRequestErrorEvent(this, error_message));
  }

  public void fireConnectionClosed() {
    fireUIChange(new CVSUIConnectionClosedEvent(this));
  }
  
  public void fireUIChange(CVSUIEvent e) {
    // Guaranteed to return a non-null array
    Object[] listeners = listenerList.getListenerList();
    
    for (int i = listeners.length-2; i>=0; i-=2) {
      if (listeners[i] == CVSUIListener.class) {
        ((CVSUIListener)listeners[i+1]).uiChanged(e);
      }
    }
  }
  
  /* managing listeners */
  public void addCVSUIListener(CVSUIListener l) {
    listenerList.add(CVSUIListener.class, l);
  }
  
  public void removeCVSUIListener(CVSUIListener l) {
    listenerList.remove(CVSUIListener.class, l);
  }

  /** Our client thread object. */
  protected CVSClient m_client;

  /** our list of listeners. */
  protected EventListenerList listenerList = new EventListenerList();
}
