/* -*- Mode: C; c-file-style: "gnu" -*-
   japharSecureEnv.cpp -- liveconnect glue.
   Created: Chris Toshok <toshok@hungry.com>, 5-Apr-1999
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1997, 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "config.h"

#include "jniint.h"
#include "japharSecureEnv.h"
#include "japharJVMPlugin.h"

#include <assert.h>

extern PRLogModuleInfo *japharOJILm;

japharSecureEnv::japharSecureEnv(japharJVMPlugin *plugin, JNIEnv *proxyJNI)
{
  JavaVM *vm;

  PR_LOG (japharOJILm, PR_LOG_DEBUG,
	  ("in japharSecureEnv::japharSecureEnv(this = %p, proxyEnv = %p)\n", this, proxyJNI));

  JavaVMAttachArgs attach_args;
  
  attach_args.version = JNI_VERSION_1_2;
  attach_args.name = "java-thread";
  attach_args.group = NULL;
  
  env = NULL;

  vm = plugin->GetRunningVM();

  /* if the thread already has an env, get ahold of it */
  if (vm->GetEnv((void**)&env, JNI_VERSION_1_2) < 0)
    {
      /* if that failed, try to attach to the current thread */
      if (plugin->GetRunningVM()->AttachCurrentThread(&env, &attach_args) < 0)
	assert(0);
    }
}

japharSecureEnv::~japharSecureEnv()
{
}

nsresult
japharSecureEnv::Create(japharJVMPlugin* plugin,
			JNIEnv* proxyEnv,
			const nsIID& aIID, void* *aInstancePtr)
{
  japharSecureEnv* secureEnv = new japharSecureEnv(plugin, proxyEnv);
  if (secureEnv == NULL)
    return NS_ERROR_OUT_OF_MEMORY;
  nsresult result = secureEnv->QueryInterface(aIID, aInstancePtr);
  if (NS_FAILED(result)) {
    delete secureEnv;
    return result;
  }
  return NS_OK;
}


/////////////////////////////////////////////////
// nsISupports
/////////////////////////////////////////////////

NS_IMPL_ADDREF(japharSecureEnv)
NS_IMPL_RELEASE(japharSecureEnv)

nsresult
japharSecureEnv::QueryInterface(REFNSIID aIID,
				void** aInstancePtr)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("japharSecureEnv::QueryInterface(this=%p)\n", this));
  if (NULL == aInstancePtr) {
    return NS_ERROR_NULL_POINTER;
  }

  *aInstancePtr = NULL;

  static NS_DEFINE_IID(kISupportsIID, NS_ISUPPORTS_IID);
  static NS_DEFINE_IID(kISecureEnvIID, NS_ISECUREENV_IID);

  if (aIID.Equals(kISecureEnvIID)) {
    *aInstancePtr = (void*) ((nsISecureEnv*)this);
    AddRef();
    PR_LOG (japharOJILm, PR_LOG_DEBUG, ("  + returning nsISecureEnv\n"));
    return NS_OK;
  }
  if (aIID.Equals(kISupportsIID)) {
    *aInstancePtr = (void*) ((nsISupports*)this);
    AddRef();
    PR_LOG (japharOJILm, PR_LOG_DEBUG, ("  + returning nsISupports\n"));
    return NS_OK;
  }

  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("  + returning NS_NOINTERFACE\n"));
  return NS_NOINTERFACE;
}

/////////////////////////////////////////////////
// nsISecureEnv
/////////////////////////////////////////////////

/**
 * Create new Java object in LiveConnect.
 *
 * @param clazz      -- Java Class object.
 * @param methodID   -- Method id
 * @param args       -- arguments for invoking the constructor.
 * @param result     -- return new Java object.
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::NewObject(/*[in]*/  jclass clazz, 
			   /*[in]*/  jmethodID methodID,
			   /*[in]*/  jvalue *args, 
			   /*[out]*/ jobject* result,
			   /*[in]*/  nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::NewObject\n"));
  jobject obj;

  obj = env->NewObjectA(clazz, methodID, args);

  *result = obj;

  return NS_OK;
}

   
/**
 * Invoke method on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param obj        -- Java object.
 * @param methodID   -- method id
 * @param args       -- arguments for invoking the constructor.
 * @param result     -- return result of invocation.
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::CallMethod(/*[in]*/  jni_type type,
			    /*[in]*/  jobject obj, 
			    /*[in]*/  jmethodID methodID,
			    /*[in]*/  jvalue *args, 
			    /*[out]*/ jvalue* result,
			    /*[in]*/  nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::CallMethod(%s,%s)\n", methodID->clazz->class_name, methodID->name));
  switch (type)
    {
    case jobject_type:
      result->l = env->CallObjectMethodA(obj, methodID, args);
      break;
    case jboolean_type:
      result->z = env->CallBooleanMethodA(obj, methodID, args);
      break;
    case jbyte_type:
      result->b = env->CallByteMethodA(obj, methodID, args);
      break;
    case jchar_type:
      result->c = env->CallCharMethodA(obj, methodID, args);
      break;
    case jshort_type:
      result->s = env->CallShortMethodA(obj, methodID, args);
      break;
    case jint_type:
      result->i = env->CallIntMethodA(obj, methodID, args);
      break;
    case jlong_type:
      result->j = env->CallLongMethodA(obj, methodID, args);
      break;
    case jfloat_type:
      result->f = env->CallFloatMethodA(obj, methodID, args);
      break;
    case jdouble_type:
      result->d = env->CallDoubleMethodA(obj, methodID, args);
      break;
    case jvoid_type:
      env->CallVoidMethodA(obj, methodID, args);
      break;
    }

  return NS_OK;
}


/**
 * Invoke non-virtual method on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param obj        -- Java object.
 * @param clazz      -- Class object
 * @param methodID   -- method id
 * @param args       -- arguments for invoking the constructor.
 * @param ctx        -- security context 
 * @param result     -- return result of invocation.
 */
nsresult
japharSecureEnv::CallNonvirtualMethod(/*[in]*/  jni_type type,
				      /*[in]*/  jobject obj, 
				      /*[in]*/  jclass clazz,
				      /*[in]*/  jmethodID methodID,
				      /*[in]*/  jvalue *args, 
				      /*[out]*/ jvalue* result,
				      /*[in]*/  nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::CallNonvirtualMethod(%s,%s)\n", methodID->clazz->class_name, methodID->name));
  switch (type)
    {
    case jobject_type:
      result->l = env->CallNonvirtualObjectMethodA(obj, methodID, args);
      break;
    case jboolean_type:
      result->z = env->CallNonvirtualBooleanMethodA(obj, methodID, args);
      break;
    case jbyte_type:
      result->b = env->CallNonvirtualByteMethodA(obj, methodID, args);
      break;
    case jchar_type:
      result->c = env->CallNonvirtualCharMethodA(obj, methodID, args);
      break;
    case jshort_type:
      result->s = env->CallNonvirtualShortMethodA(obj, methodID, args);
      break;
    case jint_type:
      result->i = env->CallNonvirtualIntMethodA(obj, methodID, args);
      break;
    case jlong_type:
      result->j = env->CallNonvirtualLongMethodA(obj, methodID, args);
      break;
    case jfloat_type:
      result->f = env->CallNonvirtualFloatMethodA(obj, methodID, args);
      break;
    case jdouble_type:
      result->d = env->CallNonvirtualDoubleMethodA(obj, methodID, args);
      break;
    case jvoid_type:
      env->CallNonvirtualVoidMethodA(obj, methodID, args);
      break;
    }

  return NS_OK;
}


/**
 * Get a field on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param obj        -- Java object.
 * @param fieldID    -- field id
 * @param result     -- return field value
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::GetField(/*[in]*/  jni_type type,
			  /*[in]*/  jobject obj, 
			  /*[in]*/  jfieldID fieldID,
			  /*[out]*/ jvalue* result,
			  /*[in]*/  nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetField(%s,%s)\n", fieldID->clazz->class_name, fieldID->name));
  switch (type)
    {
    case jobject_type:
      result->l = env->GetObjectField(obj, fieldID);
      break;
    case jboolean_type:
      result->z = env->GetBooleanField(obj, fieldID);
      break;
    case jbyte_type:
      result->b = env->GetByteField(obj, fieldID);
      break;
    case jchar_type:
      result->c = env->GetCharField(obj, fieldID);
      break;
    case jshort_type:
      result->s = env->GetShortField(obj, fieldID);
      break;
    case jint_type:
      result->i = env->GetIntField(obj, fieldID);
      break;
    case jlong_type:
      result->j = env->GetLongField(obj, fieldID);
      break;
    case jfloat_type:
      result->f = env->GetFloatField(obj, fieldID);
      break;
    case jdouble_type:
      result->d = env->GetDoubleField(obj, fieldID);
      break;
    }

  return NS_OK;
}


/**
 * Set a field on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param obj        -- Java object.
 * @param fieldID    -- field id
 * @param val        -- field value to set
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::SetField(/*[in]*/ jni_type type,
			  /*[in]*/ jobject obj, 
			  /*[in]*/ jfieldID fieldID,
			  /*[in]*/ jvalue val,
			  /*[in]*/ nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::SetField(%s,%s)\n", fieldID->clazz->class_name, fieldID->name));
  switch (type)
    {
    case jobject_type:
      env->SetObjectField(obj, fieldID, val.l);
      break;
    case jboolean_type:
      env->SetBooleanField(obj, fieldID, val.z);
      break;
    case jbyte_type:
      env->SetByteField(obj, fieldID, val.b);
      break;
    case jchar_type:
      env->SetCharField(obj, fieldID, val.c);
      break;
    case jshort_type:
      env->SetShortField(obj, fieldID, val.s);
      break;
    case jint_type:
      env->SetIntField(obj, fieldID, val.i);
      break;
    case jlong_type:
      env->SetLongField(obj, fieldID, val.j);
      break;
    case jfloat_type:
      env->SetFloatField(obj, fieldID, val.f);
      break;
    case jdouble_type:
      env->SetDoubleField(obj, fieldID, val.d);
      break;
    }

  return NS_OK;
}


/**
 * Invoke static method on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param clazz      -- Class object.
 * @param methodID   -- method id
 * @param args       -- arguments for invoking the constructor.
 * @param result     -- return result of invocation.
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::CallStaticMethod(/*[in]*/  jni_type type,
                                /*[in]*/  jclass clazz,
                                /*[in]*/  jmethodID methodID,
                                /*[in]*/  jvalue *args, 
                                /*[out]*/ jvalue* result,
                                /*[in]*/  nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::CallStaticMethod(%s,%s)\n", methodID->clazz->class_name, methodID->name));
  switch (type)
    {
    case jobject_type:
      result->l = env->CallStaticObjectMethodA(clazz, methodID, args);
      break;
    case jboolean_type:
      result->z = env->CallStaticBooleanMethodA(clazz, methodID, args);
      break;
    case jbyte_type:
      result->b = env->CallStaticByteMethodA(clazz, methodID, args);
      break;
    case jchar_type:
      result->c = env->CallStaticCharMethodA(clazz, methodID, args);
      break;
    case jshort_type:
      result->s = env->CallStaticShortMethodA(clazz, methodID, args);
      break;
    case jint_type:
      result->i = env->CallStaticIntMethodA(clazz, methodID, args);
      break;
    case jlong_type:
      result->j = env->CallStaticLongMethodA(clazz, methodID, args);
      break;
    case jfloat_type:
      result->f = env->CallStaticFloatMethodA(clazz, methodID, args);
      break;
    case jdouble_type:
      result->d = env->CallStaticDoubleMethodA(clazz, methodID, args);
      break;
    case jvoid_type:
      env->CallStaticVoidMethodA(clazz, methodID, args);
      break;
    }

  return NS_OK;
}


/**
 * Get a static field on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param clazz      -- Class object.
 * @param fieldID    -- field id
 * @param result     -- return field value
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::GetStaticField(/*[in]*/  jni_type type,
				/*[in]*/  jclass clazz, 
				/*[in]*/  jfieldID fieldID, 
				/*[out]*/ jvalue* result,
				/*[in]*/  nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStaticField(%s,%s)\n", fieldID->clazz->class_name, fieldID->name));
  switch (type)
    {
    case jobject_type:
      result->l = env->GetStaticObjectField(clazz, fieldID);
      break;
    case jboolean_type:
      result->z = env->GetStaticBooleanField(clazz, fieldID);
      break;
    case jbyte_type:
      result->b = env->GetStaticByteField(clazz, fieldID);
      break;
    case jchar_type:
      result->c = env->GetStaticCharField(clazz, fieldID);
      break;
    case jshort_type:
      result->s = env->GetStaticShortField(clazz, fieldID);
      break;
    case jint_type:
      result->i = env->GetStaticIntField(clazz, fieldID);
      break;
    case jlong_type:
      result->j = env->GetStaticLongField(clazz, fieldID);
      break;
    case jfloat_type:
      result->f = env->GetStaticFloatField(clazz, fieldID);
      break;
    case jdouble_type:
      result->d = env->GetStaticDoubleField(clazz, fieldID);
      break;
    }

  return NS_OK;
}



/**
 * Set a static field on Java object in LiveConnect.
 *
 * @param type       -- Return type
 * @param clazz      -- Class object.
 * @param fieldID    -- field id
 * @param val        -- field value to set
 * @param ctx        -- security context 
 */
nsresult
japharSecureEnv::SetStaticField(/*[in]*/ jni_type type,
				/*[in]*/ jclass clazz, 
				/*[in]*/ jfieldID fieldID,
				/*[in]*/ jvalue val,
				/*[in]*/ nsISecurityContext* ctx)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::SetStaticField(%s,%s)\n", fieldID->clazz->class_name, fieldID->name));
  switch (type)
    {
    case jobject_type:
      env->SetStaticObjectField(clazz, fieldID, val.l);
      break;
    case jboolean_type:
      env->SetStaticBooleanField(clazz, fieldID, val.z);
      break;
    case jbyte_type:
      env->SetStaticByteField(clazz, fieldID, val.b);
      break;
    case jchar_type:
      env->SetStaticCharField(clazz, fieldID, val.c);
      break;
    case jshort_type:
      env->SetStaticShortField(clazz, fieldID, val.s);
      break;
    case jint_type:
      env->SetStaticIntField(clazz, fieldID, val.i);
      break;
    case jlong_type:
      env->SetStaticLongField(clazz, fieldID, val.j);
      break;
    case jfloat_type:
      env->SetStaticFloatField(clazz, fieldID, val.f);
      break;
    case jdouble_type:
      env->SetStaticDoubleField(clazz, fieldID, val.d);
      break;
    }

  return NS_OK;
}

nsresult
japharSecureEnv::GetVersion(/*[out]*/ jint* version)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetVersion\n"));
  *version = env->GetVersion();

  return NS_OK;
}


nsresult
japharSecureEnv::DefineClass(/*[in]*/  const char* name,
			     /*[in]*/  jobject loader,
			     /*[in]*/  const jbyte *buf,
			     /*[in]*/  jsize len,
			     /*[out]*/ jclass* clazz)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::DefineClass\n"));

  *clazz = env->DefineClass(loader, buf, len);

  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+                 result = %p\n", *clazz));

  return NS_OK;
}


nsresult
japharSecureEnv::FindClass(/*[in]*/  const char* name, 
			   /*[out]*/ jclass* clazz)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::FindClass(%s)\n", name));

  *clazz = env->FindClass(name);

  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+                 result = %p\n", *clazz));
  return NS_OK;
}


nsresult
japharSecureEnv::GetSuperclass(/*[in]*/  jclass sub,
			       /*[out]*/ jclass* super)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetSuperclass\n"));

  *super = env->GetSuperclass(sub);

  return NS_OK;
}


nsresult
japharSecureEnv::IsAssignableFrom(/*[in]*/  jclass sub,
                                /*[in]*/  jclass super,
                                /*[out]*/ jboolean* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::IsAssignableFrom\n"));
  *result = env->IsAssignableFrom(sub, super);
  return NS_OK;
}


nsresult
japharSecureEnv::Throw(/*[in]*/  jthrowable obj,
		       /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::Throw\n"));
  *result = env->Throw(obj);
  return NS_OK;
}


nsresult
japharSecureEnv::ThrowNew(/*[in]*/  jclass clazz,
			  /*[in]*/  const char *msg,
			  /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ThrowNew(%s)\n", msg ? msg : "(null msg)"));
  *result = env->ThrowNew(clazz, msg);
  return NS_OK;
}


nsresult
japharSecureEnv::ExceptionOccurred(/*[out]*/ jthrowable* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ExceptionOccurred\n"));
  *result = env->ExceptionOccurred();
  return NS_OK;
}


nsresult
japharSecureEnv::ExceptionDescribe(void)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ExceptionDescribe\n"));
  env->ExceptionDescribe();
  return NS_OK;
}


nsresult
japharSecureEnv::ExceptionClear(void)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ExceptionClear\n"));
  env->ExceptionClear();
  return NS_OK;
}


nsresult
japharSecureEnv::FatalError(/*[in]*/ const char* msg)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::FatalError\n"));
  env->FatalError(msg);
  return NS_OK;
}


nsresult
japharSecureEnv::NewGlobalRef(/*[in]*/  jobject lobj, 
			      /*[out]*/ jobject* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::NewGlobalRef\n"));
  *result = env->NewGlobalRef(lobj);
  return NS_OK;
}


nsresult
japharSecureEnv::DeleteGlobalRef(/*[in]*/ jobject gref)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::DeleteGlobalRef\n"));
  env->DeleteGlobalRef(gref);
  return NS_OK;
}


nsresult
japharSecureEnv::DeleteLocalRef(/*[in]*/ jobject obj)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::DeleteLocalRef\n"));
  env->DeleteLocalRef(obj);
  return NS_OK;
}


nsresult
japharSecureEnv::IsSameObject(/*[in]*/  jobject obj1,
			      /*[in]*/  jobject obj2,
			      /*[out]*/ jboolean* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::IsSameObject\n"));
  *result = env->IsSameObject(obj1, obj2);
  return NS_OK;
}


nsresult
japharSecureEnv::AllocObject(/*[in]*/  jclass clazz,
			     /*[out]*/ jobject* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::AllocObject\n"));
  *result = env->AllocObject(clazz);
  return NS_OK;
}


nsresult
japharSecureEnv::GetObjectClass(/*[in]*/  jobject obj,
				/*[out]*/ jclass* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetObjectClass\n"));
  *result = env->GetObjectClass(obj);
  return NS_OK;
}


nsresult
japharSecureEnv::IsInstanceOf(/*[in]*/  jobject obj,
			      /*[in]*/  jclass clazz,
			      /*[out]*/ jboolean* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::IsInstanceOf\n"));
  *result = env->IsInstanceOf(obj, clazz);
  return NS_OK;
}


nsresult
japharSecureEnv::GetMethodID(/*[in]*/  jclass clazz, 
			     /*[in]*/  const char* name,
			     /*[in]*/  const char* sig,
			     /*[out]*/ jmethodID* id)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetMethodID(%s,%s)\n", name, sig));
  *id = env->GetMethodID(clazz, name, sig);
  return NS_OK;
}


nsresult
japharSecureEnv::GetFieldID(/*[in]*/  jclass clazz, 
			    /*[in]*/  const char* name,
			    /*[in]*/  const char* sig,
			    /*[out]*/ jfieldID* id)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetFieldID(%s,%s)\n", name, sig));
  *id = env->GetFieldID(clazz, name, sig);
  return NS_OK;
}


nsresult
japharSecureEnv::GetStaticMethodID(/*[in]*/  jclass clazz, 
				   /*[in]*/  const char* name,
				   /*[in]*/  const char* sig,
				   /*[out]*/ jmethodID* id)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStaticMethodID(%s,%s)\n", name, sig));
  *id = env->GetStaticMethodID(clazz, name, sig);
  return NS_OK;
}


nsresult
japharSecureEnv::GetStaticFieldID(/*[in]*/  jclass clazz, 
                                /*[in]*/  const char* name,
                                /*[in]*/  const char* sig,
                                /*[out]*/ jfieldID* id)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStaticFieldID(%s,%s)\n", name, sig));
  *id = env->GetStaticFieldID(clazz, name, sig);
  return NS_OK;
}


nsresult
japharSecureEnv::NewString(/*[in]*/  const jchar* unicode,
			   /*[in]*/  jsize len,
			   /*[out]*/ jstring* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::NewString\n"));
  *result = env->NewString(unicode, len);
  return NS_OK;
}


nsresult
japharSecureEnv::GetStringLength(/*[in]*/  jstring str,
				 /*[out]*/ jsize* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStringLength\n"));
  *result = env->GetStringLength(str);
  return NS_OK;
}

    
nsresult
japharSecureEnv::GetStringChars(/*[in]*/  jstring str,
				/*[in]*/  jboolean *isCopy,
				/*[out]*/ const jchar** result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStringChars\n"));
  *result = env->GetStringChars(str, isCopy);
  return NS_OK;
}


nsresult
japharSecureEnv::ReleaseStringChars(/*[in]*/  jstring str,
				    /*[in]*/  const jchar *chars)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ReleaseStringChars\n"));
  env->ReleaseStringChars(str, chars);
  return NS_OK;
}


nsresult
japharSecureEnv::NewStringUTF(/*[in]*/  const char *utf,
			      /*[out]*/ jstring* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::NewStringUTF(%s)\n", utf));
  *result = env->NewStringUTF(utf);
  return NS_OK;
}


nsresult
japharSecureEnv::GetStringUTFLength(/*[in]*/  jstring str,
				    /*[out]*/ jsize* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStringUTFLength\n"));
  *result = env->GetStringUTFLength(str);
  return NS_OK;
}

    
nsresult
japharSecureEnv::GetStringUTFChars(/*[in]*/  jstring str,
				   /*[in]*/  jboolean *isCopy,
				   /*[out]*/ const char** result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetStringUTFChars\n"));
  *result = (const char*)env->GetStringUTFChars(str, isCopy);
  return NS_OK;
}


nsresult
japharSecureEnv::ReleaseStringUTFChars(/*[in]*/  jstring str,
				       /*[in]*/  const char *chars)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ReleaseStringUTFChars\n"));
  env->ReleaseStringUTFChars(str, chars);
  return NS_OK;
}


nsresult
japharSecureEnv::GetArrayLength(/*[in]*/  jarray array,
				/*[out]*/ jsize* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetArrayLength\n"));
  *result = env->GetArrayLength(array);
  return NS_OK;
}


nsresult
japharSecureEnv::NewObjectArray(/*[in]*/  jsize len,
				/*[in]*/  jclass clazz,
				/*[in]*/  jobject init,
				/*[out]*/ jobjectArray* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::NewObjectArray\n"));
  *result = env->NewObjectArray(len, clazz, init);
  return NS_OK;
}


nsresult
japharSecureEnv::GetObjectArrayElement(/*[in]*/  jobjectArray array,
				       /*[in]*/  jsize index,
				       /*[out]*/ jobject* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetObjectArrayElement\n"));
  *result = env->GetObjectArrayElement(array, index);
  return NS_OK;
}


nsresult
japharSecureEnv::SetObjectArrayElement(/*[in]*/  jobjectArray array,
				       /*[in]*/  jsize index,
				       /*[in]*/  jobject val)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::SetObjectArrayElement\n"));
  env->SetObjectArrayElement(array, index, val);
  return NS_OK;
}


nsresult
japharSecureEnv::NewArray(/*[in]*/ jni_type element_type,
			  /*[in]*/  jsize len,
			  /*[out]*/ jarray* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::NewArray\n"));
  switch (element_type)
    {
    case jboolean_type:
      *result = env->NewBooleanArray(len);
      break;
    case jbyte_type:
      *result = env->NewByteArray(len);
      break;
    case jchar_type:
      *result = env->NewCharArray(len);
      break;
    case jshort_type:
      *result = env->NewShortArray(len);
      break;
    case jint_type:
      *result = env->NewIntArray(len);
      break;
    case jlong_type:
      *result = env->NewLongArray(len);
      break;
    case jfloat_type:
      *result = env->NewFloatArray(len);
      break;
    case jdouble_type:
      *result = env->NewDoubleArray(len);
      break;
    case jobject_type:
    case jvoid_type:
      *result = NULL;
      break;
    }

  return NS_OK;
}


nsresult
japharSecureEnv::GetArrayElements(/*[in]*/  jni_type type,
                                /*[in]*/  jarray array,
                                /*[in]*/  jboolean *isCopy,
                                /*[out]*/ void* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetArrayElements\n"));
  switch (type)
    {
    case jboolean_type:
      *(jboolean**)result = env->GetBooleanArrayElements(array, isCopy);
      break;
    case jbyte_type:
      *(jbyte**)result = env->GetByteArrayElements(array, isCopy);
      break;
    case jchar_type:
      *(jchar**)result = env->GetCharArrayElements(array, isCopy);
      break;
    case jshort_type:
      *(jshort**)result = env->GetShortArrayElements(array, isCopy);
      break;
    case jint_type:
      *(jint**)result = env->GetIntArrayElements(array, isCopy);
      break;
    case jlong_type:
      *(jlong**)result = env->GetLongArrayElements(array, isCopy);
      break;
    case jfloat_type:
      *(jfloat**)result = env->GetFloatArrayElements(array, isCopy);
      break;
    case jdouble_type:
      *(jdouble**)result = env->GetDoubleArrayElements(array, isCopy);
      break;
    case jobject_type:
    case jvoid_type:
      *(void**)result = NULL;
      break;
    }

  return NS_OK;
}


nsresult
japharSecureEnv::ReleaseArrayElements(/*[in]*/ jni_type type,
				      /*[in]*/ jarray array,
				      /*[in]*/ void *elems,
				      /*[in]*/ jint mode)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::ReleaseArrayElements\n"));
  switch (type)
    {
    case jboolean_type:
      env->ReleaseBooleanArrayElements(array, (jboolean*)elems, mode);
      break;
    case jbyte_type:
      env->ReleaseByteArrayElements(array, (jbyte*)elems, mode);
      break;
    case jchar_type:
      env->ReleaseCharArrayElements(array, (jchar*)elems, mode);
      break;
    case jshort_type:
      env->ReleaseShortArrayElements(array, (jshort*)elems, mode);
      break;
    case jint_type:
      env->ReleaseIntArrayElements(array, (jint*)elems, mode);
      break;
    case jlong_type:
      env->ReleaseLongArrayElements(array, (jlong*)elems, mode);
      break;
    case jfloat_type:
      env->ReleaseFloatArrayElements(array, (jfloat*)elems, mode);
      break;
    case jdouble_type:
      env->ReleaseDoubleArrayElements(array, (jdouble*)elems, mode);
      break;
    }

  return NS_OK;
}


nsresult
japharSecureEnv::GetArrayRegion(/*[in]*/  jni_type type,
				/*[in]*/  jarray array,
				/*[in]*/  jsize start,
				/*[in]*/  jsize len,
				/*[out]*/ void* buf)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetArrayRegion\n"));
  switch (type)
    {
    case jboolean_type:
      env->GetBooleanArrayRegion(array, start, len, (jboolean*)buf);
      break;
    case jbyte_type:
      env->GetByteArrayRegion(array, start, len, (jbyte*)buf);
      break;
    case jchar_type:
      env->GetCharArrayRegion(array, start, len, (jchar*)buf);
      break;
    case jshort_type:
      env->GetShortArrayRegion(array, start, len, (jshort*)buf);
      break;
    case jint_type:
      env->GetIntArrayRegion(array, start, len, (jint*)buf);
      break;
    case jlong_type:
      env->GetLongArrayRegion(array, start, len, (jlong*)buf);
      break;
    case jfloat_type:
      env->GetFloatArrayRegion(array, start, len, (jfloat*)buf);
      break;
    case jdouble_type:
      env->GetDoubleArrayRegion(array, start, len, (jdouble*)buf);
      break;
    }

  return NS_OK;
}


nsresult
japharSecureEnv::SetArrayRegion(/*[in]*/  jni_type type,
				/*[in]*/  jarray array,
				/*[in]*/  jsize start,
				/*[in]*/  jsize len,
				/*[in]*/  void* buf)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::SetArrayRegion\n"));
  switch (type)
    {
    case jboolean_type:
      env->SetBooleanArrayRegion(array, start, len, (jboolean*)buf);
      break;
    case jbyte_type:
      env->SetByteArrayRegion(array, start, len, (jbyte*)buf);
      break;
    case jchar_type:
      env->SetCharArrayRegion(array, start, len, (jchar*)buf);
      break;
    case jshort_type:
      env->SetShortArrayRegion(array, start, len, (jshort*)buf);
      break;
    case jint_type:
      env->SetIntArrayRegion(array, start, len, (jint*)buf);
      break;
    case jlong_type:
      env->SetLongArrayRegion(array, start, len, (jlong*)buf);
      break;
    case jfloat_type:
      env->SetFloatArrayRegion(array, start, len, (jfloat*)buf);
      break;
    case jdouble_type:
      env->SetDoubleArrayRegion(array, start, len, (jdouble*)buf);
      break;
    }

  return NS_OK;
}


nsresult
japharSecureEnv::RegisterNatives(/*[in]*/  jclass clazz,
				 /*[in]*/  const JNINativeMethod *methods,
				 /*[in]*/  jint nMethods,
				 /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::RegisterNatives\n"));
  *result = env->RegisterNatives(clazz, methods, nMethods);
  return NS_OK;
}


nsresult
japharSecureEnv::UnregisterNatives(/*[in]*/  jclass clazz,
				   /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::UnregisterNatives\n"));
  *result = env->UnregisterNatives(clazz);
  return NS_OK;
}


nsresult
japharSecureEnv::MonitorEnter(/*[in]*/  jobject obj,
			      /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::MonitorEnter(%p)\n", obj));
  *result = env->MonitorEnter(obj);
  return NS_OK;
}


nsresult
japharSecureEnv::MonitorExit(/*[in]*/  jobject obj,
			     /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::MonitorExit(%p)\n", obj));
  *result = env->MonitorExit(obj);
  return NS_OK;
}


nsresult
japharSecureEnv::GetJavaVM(/*[in]*/  JavaVM **vm,
			   /*[out]*/ jint* result)
{
  PR_LOG (japharOJILm, PR_LOG_DEBUG, ("+japharSecureEnv::GetJavaVM\n"));
  *result = env->GetJavaVM(vm);
  return NS_OK;
}
