/* -*- Mode: C; c-file-style: "gnu" -*-
   jvmdisfm.c -- JVMDI Stack Frame Functions
   Created: Hernan Otero <hernan_otero@bigfoot.com>, 18-Jul-1998.
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "jvmdiint.h"

JNIEXPORT jvmdiError JNICALL
JVMDI_GetCurrentFrame(JNIEnv *env,
		      jthread thread,
		      jframeID *framePtr)
{
  japhar_object* thread_obj = (japhar_object*)thread;
  HungryEnv *henv;

  /* usual NULL checks */
  if (env == NULL
      || thread_obj == NULL)
    return JVMDI_ERROR_NULL_POINTER;

  /* make sure the object refers to a thread (and a valid one at that.) */
  henv = (HungryEnv*)HVM_ObjectGetNativeState(thread_obj);
  if (henv == NULL)
    return JVMDI_ERROR_INVALID_THREAD;

  /* the thread must be suspended for this operation. */
  if (henv->current_state != STATE_SUSPENDED)
    return JVMDI_ERROR_THREAD_NOT_SUSPENDED;

  *framePtr = (jframeID)henv->top_frame;
  return JVMDI_ERROR_NONE;
}

JNIEXPORT jvmdiError JNICALL
JVMDI_GetCallerFrame(JNIEnv *env,
		     jframeID called,
		     jframeID *framePtr)
{
  HungryEnv *henv;

  /* usual NULL checks */
  if (env == NULL
      || framePtr == NULL)
    return JVMDI_ERROR_NULL_POINTER;

  henv = ((StackFrame*)called)->_env;

  /* sanity check for a valid frame id */
  if (called == NULL)
    return JVMDI_ERROR_INVALID_FRAMEID;

  *framePtr = (jframeID)((StackFrame*)called)->parent;

  if (*framePtr >= (jframeID)henv->stack_highwater)
    {
      *framePtr = NULL;
      return JVMDI_ERROR_NO_MORE_FRAMES;
    }

  return JVMDI_ERROR_NONE;
}

JNIEXPORT jvmdiError JNICALL
JVMDI_GetFrameMethod(JNIEnv *env,
		     jframeID frame, 
		     jclass *classPtr,
		     jmethodID *methodPtr)
{
  HungryEnv *henv = HVM_ThreadGetEnv();
  StackFrame *sframe = (StackFrame*)frame;
  /* usual NULL checks */
  if (env == NULL
      || classPtr == NULL
      || methodPtr == NULL)
    return JVMDI_ERROR_NULL_POINTER;

  /* sanity check for a valid frame id */
  if (frame == NULL)
    return JVMDI_ERROR_INVALID_FRAMEID;

  *methodPtr = sframe->method;
  *classPtr = clazzfile_to_jclass(henv, sframe->method->clazz);
  return JVMDI_ERROR_NONE;
}

JNIEXPORT jvmdiError JNICALL
JVMDI_GetFrameLocation(JNIEnv *env,
		       jframeID frame, jclass *classPtr, jmethodID *methodPtr,
		       jlocation *locationPtr)
{
  StackFrame *sframe = (StackFrame*)frame;
  HungryEnv *henv = sframe->_env;

  /* usual NULL checks */
  if (env == NULL
      || classPtr == NULL
      || methodPtr == NULL
      || locationPtr == NULL)
    return JVMDI_ERROR_NULL_POINTER;

  /* sanity check for a valid frame id */
  if (frame == NULL)
    return JVMDI_ERROR_INVALID_FRAMEID;

  *classPtr = clazzfile_to_jclass(henv, sframe->method->clazz);
  *methodPtr = sframe->method;
  *locationPtr = sframe->pc;

  return JVMDI_ERROR_NONE;
}

JNIEXPORT jvmdiError JNICALL
JVMDI_NotifyFramePop(JNIEnv *env,
		     jframeID frame)
{
  PR_ASSERT(0);
  return JVMDI_ERROR_NONE;
}
