/* -*- Mode: C; c-file-style: "gnu"; indent-tabs-mode:nil -*-
   jnisfld.c -- Java Native Interface methods relating to static fields.
   Created: Chris Toshok <toshok@hungry.com>, 26-Jul-1997
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1997, 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "jniint.h"

jfieldID
JNIFUNC(GetStaticFieldID)(JNIEnv *env,
                          jclass clazz,
                          const char *name,
                          const char *sig)
{
  HungryEnv *henv = JNIEnvToHEnv(env);
  FieldStruct *field = NULL;
  ClazzFile *cf = jclass_to_clazzfile(henv, clazz);
  int field_num;

  for (field_num = 0; field_num < cf->num_fields; field_num++)
    {
      FieldStruct *cur_field = cf->fields[field_num];

      if ((cur_field->access_flags & ACC_STATIC) == 0)
        continue;

      if (!PL_strcmp(cur_field->name, name)
          && !PL_strcmp(cur_field->sig_str, sig))
        {
          field = cur_field;
          break;
        }
    }

  if (field == NULL)
    {
      jclass no_such_field;
      (*env)->ExceptionClear(env);
      no_such_field = (*env)->FindClass(env, java_lang_NoSuchFieldException);
      if (no_such_field)
        (*env)->ThrowNew(env, no_such_field, name);
      else
        (*env)->FatalError(env,
                           "Could not load java/lang/NoSuchFieldException");
      return NULL;
    }
  return field;
}

#define GETSET_STATIC_FIELD(type, pretty_type, sig_type, union_element) \
type \
JNIFUNC(GetStatic##pretty_type##Field)(JNIEnv *env, \
                                       jclass clazz, \
                                       jfieldID fieldID) \
{ \
    HungryEnv *henv = JNIEnvToHEnv(env); \
    FieldStruct *field = (FieldStruct*)fieldID; \
    InterpValue val; \
    ClazzFile *cls = jclass_to_clazzfile(henv, clazz); \
        \
    initialize_class(henv, cls); \
    HVM_FieldGetStatic(cls, field, &val); \
        \
    return val.union_element; \
}  \
void \
JNIFUNC(SetStatic##pretty_type##Field)(JNIEnv *env, \
                                       jclass clazz, \
                                       jfieldID fieldID, \
                                       type value) \
{ \
    HungryEnv *henv = JNIEnvToHEnv(env); \
    FieldStruct *field = (FieldStruct*)fieldID; \
    InterpValue val; \
    ClazzFile *cls = jclass_to_clazzfile(henv, clazz); \
        \
    initialize_class(henv, cls); \
        \
    val.union_element = value; \
        \
    HVM_FieldSetStatic(cls, field, val);  \
}


GETSET_STATIC_FIELD(jboolean, Boolean, SIG_JBOOLEAN, z)
GETSET_STATIC_FIELD(jbyte, Byte, SIG_JBYTE, b)
GETSET_STATIC_FIELD(jchar, Char, SIG_JCHAR, c)
GETSET_STATIC_FIELD(jshort, Short, SIG_JSHORT, s)
GETSET_STATIC_FIELD(jint, Int, SIG_JINT, i)
GETSET_STATIC_FIELD(jlong, Long, SIG_JLONG, j)
GETSET_STATIC_FIELD(jfloat, Float, SIG_JFLOAT, f)
GETSET_STATIC_FIELD(jdouble, Double, SIG_JDOUBLE, d)
GETSET_STATIC_FIELD(jobject, Object, SIG_JOBJECT, l)
