/* -*- Mode: C; c-file-style: "gnu"; indent-tabs-mode:nil -*-
   abort.c -- our own version of abort that calls the vm hook.
   Created: Chris Toshok <toshok@hungry.com>, 15-Aug-1997
*/
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1997, 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "runtimeint.h"
#include "arch.h"

void
abort_with_message(const char *msg)
{
  fprintf (stderr, "Fatal Japhar Error: %s\n", msg);

  /* our version of abort() will take care of calling
     the abort hook for this vm, if any. */
  abort();
}

#ifndef PROFILING
void
abort(void)
{
  HungryEnv *henv = HVM_ThreadGetEnv();
  if (henv)
    {
      HungryVM *vm = henv->vm;

      if (vm->abort)
        {
          (*vm->abort)();
        }
    }

#ifdef DEBUG
  SIGNAL_crash();
#else
  exit(-1);
#endif
}
#endif

void
fatal_signal_handler(int signal)
{
  HungryEnv *henv = HVM_ThreadGetEnv();
  StackFrame *frame = henv->top_frame;

  fprintf (stderr, "fatal signal handler called, signal = %d\n", signal);

  while (frame < henv->stack_highwater)
    {
      if (frame->flags & FRAME_NATIVE)
        {
          fprintf (stderr, "native frame:\t%s.%s\n",
                   getClassName(henv, frame->method->clazz),
                   frame->method->name);
        }
      else 
        {
          int line_number = method_pc_to_line_number(henv, frame->method, frame->pc);
          if (-1 == line_number)
            fprintf (stderr, "java frame:\t%s.%s (pc = %d)\n",
                     getClassName(henv, frame->method->clazz),
                     frame->method->name, frame->pc);
          else
            fprintf (stderr, "java frame:\t%s.%s (%d, pc = %d)\n",
                     getClassName(henv, frame->method->clazz),
                     frame->method->name, line_number, frame->pc);
        }

      frame = frame->parent;
    }

#ifdef DEBUG
  SIGNAL_crash();
#else
  exit(-1);
#endif
}
