/*
    $Id: k74.c,v 1.6 2000/09/08 12:13:31 root Exp root $

    k74, kitsrus.com kit 74 parallel port relay driver
    Copyright (C) 2000  James Cameron (quozl@us.netrek.org)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef lint
static char vcid[] = "$Id: k74.c,v 1.6 2000/09/08 12:13:31 root Exp root $";
#endif /* lint */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>

/* port access functions borrowed from prog84 */
int dev_port_fd = -1;

void open_io()
{
  dev_port_fd = open("/dev/port", O_RDWR);
  if(dev_port_fd < 0)
  {
    perror("/dev/port");
    exit(errno);
  }
}

void close_io()
{
  close(dev_port_fd);
  dev_port_fd = -1;
}

void out_byte(int port, unsigned char byte)
{
  off_t s;
  int r;

  s = lseek(dev_port_fd, port, 0);
  if (s < 0) perror("lseek");
  else if (s != port)
    fprintf(stderr, "hmmm: seeking to %d, went to %ld.\n", port, (long)s);

  r = write(dev_port_fd, &byte, 1);
  if (r != 1)
  {
    fprintf(stderr, "hmmm: write returned %d\n", r);
    if (r < 0) perror("write");
  }
}

unsigned char in_byte(int port)
{
  off_t s;
  int r;
  unsigned char ch = 0;

  s = lseek(dev_port_fd, port, 0);
  if (s < 0)
    perror("lseek");
  else if (s != port)
    fprintf(stderr, "hmmm: seeking to %d, went to %ld.\n", port, (long)s);

  r = read(dev_port_fd, &ch, 1);
  if (r != 1)
  {
    fprintf(stderr, "hmmm: read returned %d\n", r);
    if (r < 0) perror("read");
  }

  return ch;
}

static void copyright()
{
    fprintf(stderr, "Kit 74 Parallel Port Relay Driver, 1.0\n"
	    "Copyright (C) 2000  James Cameron (quozl@us.netrek.org)\n"
	    "%s\n\n"
"This program comes with ABSOLUTELY NO WARRANTY; for details see source.\n"
"This is free software, and you are welcome to redistribute it under certain\n"
"conditions; see source for details.\n\n", vcid);
}

static void usage()
{
  copyright();
  
  fprintf(stderr, 
	  "Usage: k74 [options] [bit] [command] ...\n"
"\n"
"Options\n"
"    -v  --verbose     issue copyright statement during operation\n"
"    -V  --version     display version information\n"
"    -h  --help        display program usage information\n"
"    -p  --port n      set parallel port for next operation (0,1,2)\n"
"                      (n may also be 0x378, 0x278, or 0x3bc)\n"
"Bits\n"
"    n                 bit number, zero through seven\n"
"    all               all bits, mask 0xff\n"
"    lower             lower half of byte, mask 0x0f\n"
"    upper             upper half of byte, mask 0xf0\n"
"\n"
"Commands\n"
"    set               set the bits\n"
"    clear             clear the bits\n"
"    toggle            change the state of the bits\n"
"    write n           write decimal number n to the port\n"
"\n"
"    read              Variation form the original k74 !!!!! \n"
"                      if Bit n [0..7] reads the selectet bit of \n"
"                      the port and print on stdout 1/0\n"
"\n"
"Examples\n"
"    # k74 all set                (turns on all bits)\n"
"    # k74 6 read                 (returns  bit six )\n"
	  );

}

int main ( int argc, char *argv[] )
{
  int i, bit,readout, port = 0x378;

  open_io();

  /* process command line arguments */
  for(i=1;i<argc;i++) {
    
    if (!strcmp(argv[i], "-v")||!strcmp(argv[i], "--verbose")) {
      copyright();
      continue;
    }
    
    if (!strcmp(argv[i], "-V")||!strcmp(argv[i], "--version")) {
      copyright();
      printf("%s\n", vcid);
      exit(0);
    }
    
    if (!strcmp(argv[i], "-h")||!strcmp(argv[i], "--help")) {
      usage();
      exit(1);
    }

    if (!strcmp(argv[i], "-p")||!strcmp(argv[i], "--port")) {
      if ( i++ == argc ) break;
      port = atoi(argv[i]);
      switch (port) {
      case 0: port = 0x378; break;
      case 1: port = 0x278; break;
      case 2: port = 0x3bc; break;
      case 0x378:
      case 0x278:
      case 0x3bc:
		break;
      default:
			fprintf(stderr, "k74: port %s not valid\n", argv[i]);
			exit(2);
      		}
      continue;
    }

   
	 if (!strcmp(argv[i], "all")) {
      bit = 0xff;
      continue;
    }
    
    if (!strcmp(argv[i], "lower")) {
      bit = 0xf;
      continue;
    }
    
    if (!strcmp(argv[i], "upper")) {
      bit = 0xf0;
      continue;
    }
    
    if (!strcmp(argv[i], "aa")) {
      bit = 0xaa;
      continue;
    }
    
    if (!strcmp(argv[i], "55")) {
      bit = 0x55;
      continue;
    }
    
    if (!strcmp(argv[i], "set")||!strcmp(argv[i], "on")) {
      out_byte(port,in_byte(port)|(bit));
      continue;
    }
    
    if (!strcmp(argv[i], "clear")||!strcmp(argv[i], "off")) {
      out_byte(port,in_byte(port)&~(bit));
      continue;
    }
    
    if (!strcmp(argv[i], "toggle")) {
      out_byte(port,in_byte(port)^(bit));
      continue;
    }
    
    if (!strcmp(argv[i], "write")) {
      if ( i++ == argc ) break;
      out_byte(port,atoi(argv[i]));
    }

    if (!strcmp(argv[i], "read")) {
    readout = in_byte(port);
	 switch (bit) 
	  {
      case 128: readout=readout / 128;  break;
      case  64: readout=(readout % 128) / 64 ;   break;
      case  32: readout=((readout % 128) % 64) /32 ;  break;
      case  16: readout=(((readout % 128) % 64) %32) / 16;  break;
		case   8: readout=((((readout % 128) % 64) %32) % 16) /8;  break;
      case   4: readout=(((((readout % 128) % 64) %32) % 16) %8) /4;  break;
		case   2: readout=((((((readout % 128) % 64) %32) % 16) %8) %4) /2;  break;
		case   1: readout=(((((((readout % 128) % 64) %32) % 16) %8) %4) %2) /1;  break;
		break;
     }
	 //fprintf(stdout,"%d\n",bit);  
	 fprintf(stdout,"%d\n",readout);
    }

    bit = 1<<atoi(argv[i]);
  }
  
}
