//----------------------------------------------------------------------------
// Creation:    10.04.2007 16:41
// Author:      Roland Spitzer
//
// Programm:    irlogger.c
//
// Last Update: 11.04.2007 09:04 Umbau zum daemon-Prozess
//              12.04.2007 11:28 SIGNAL Handling
//              12.04.2007 19:56 Port-Auswahl eingebaut
//
// Kleines Programm, dass einen Mini-Server fr IRTrans darstellt.
// Es wird ein Socket auf Port 21001 Protokoll UDP aufgemacht und alle ankommenden
// Pakete werden mit Datum und Uhrzeit ausgegeben. Die Ausgabe wird in ein Log
// geschrieben werden und kann dann mittels Logfile-Adapter ausgewertet werden.
//
//----------------------------------------------------------------------------

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <signal.h>

#define MAX_MSG 256

//----------------------------------------------------------------------------
//  functions
//----------------------------------------------------------------------------

void    signal_process (int);
void    time_string (void);
void    log_print (void);

//----------------------------------------------------------------------------
//  public definitions
//----------------------------------------------------------------------------

FILE    *pidfp, *logfp;
char    program [256];
char    logfile [256];
char    st      [256];
char    time_st [80];
int     sd;
int     port    = 21001;

//****************************************************************************
//  irlogger
//****************************************************************************

int main(int argc, char *argv[])
{
    int       rc, n, cliLen;
    struct    sockaddr_in cliAddr, servAddr;
    char      msg [MAX_MSG];
    struct    sigaction sig_struct ;

//----------------------------------------------------------------------------
//  parametr
//----------------------------------------------------------------------------

    strcpy (program, argv[0]);
    if   (argc != 3)
         {
         fprintf(stderr, "%s first parameter port-number, second logfile\n", program);
         exit(1);
         }
    port = atoi (argv[1]);
    strcpy (logfile, argv[2]);

//----------------------------------------------------------------------------
//  start as daemon
//
//  fork creates a copy of the parent-process
//  fork () == 0 child-process, write pid in pid-file and close parent-process
//               with exit(0)
//----------------------------------------------------------------------------

    if   (fork () == 0)
         {
         setsid ();
         pidfp = fopen ("/var/run/irlogger.pid","w");
         if   (pidfp)
              {
              fprintf(pidfp, "%d\n", getpid ());
              fclose(pidfp);
              }
        }
    else
        exit(0);
    
//----------------------------------------------------------------------------
//  signal handling
//----------------------------------------------------------------------------

    sig_struct.sa_handler = signal_process;
    sigemptyset(&sig_struct.sa_mask);
    sig_struct.sa_flags = 0;
    
    if (sigaction(SIGTERM, &sig_struct, NULL) != 0)
       {
       fprintf(stderr, "%s error signal processing SIGTERM\n", program);
       exit(1);
       }
    if (sigaction(SIGHUP, &sig_struct, NULL) != 0)
       {
       fprintf(stderr, "%s error signal processing SIGHUP\n", program);
       exit(1);
       }

//----------------------------------------------------------------------------
//  open logfile
//----------------------------------------------------------------------------

    logfp = fopen (logfile, "w");
    if   (!logfp)
         {
         fprintf(stderr, "%s unable to open logfile %s\n", program, logfile);
         exit(1);
         }

//----------------------------------------------------------------------------
//  socket creation
//----------------------------------------------------------------------------

    sd = socket(AF_INET, SOCK_DGRAM, 0);
    if   (sd < 0)
         {
         fprintf(stderr, "%s cannot open socket\n", program);
         fclose(logfp);
         exit(1);
         }

//----------------------------------------------------------------------------
//  bind local server port
//----------------------------------------------------------------------------

    servAddr.sin_family = AF_INET;
    servAddr.sin_addr.s_addr = htonl(INADDR_ANY);
    servAddr.sin_port = htons(port);
    rc = bind (sd, (struct sockaddr *) &servAddr, sizeof(servAddr));
    if   (rc < 0)
         {
         fprintf(stderr, "%s cannot bind port number %d\n", program, port);
         fclose(logfp);
         close(sd);
         exit(1);
         }

//----------------------------------------------------------------------------
//  close standard-files and wait for data
//----------------------------------------------------------------------------

    fclose(stdin);
    fclose(stdout);
    fclose(stderr);
    stderr = NULL;

    sprintf(st, "waiting for data on port UDP %d\n", port);
    log_print();

    while (1)
         {
         memset(msg, 0x0, MAX_MSG);

         cliLen = sizeof(cliAddr);
         n = recvfrom(sd, msg, MAX_MSG, 0, (struct sockaddr *) &cliAddr, &cliLen);

         if   (n < 0)
              {
              sprintf(st, "cannot receive data\n");
              log_print();
              continue;
              }

         sprintf(st, "%s[%u]: %s\n", inet_ntoa(cliAddr.sin_addr), ntohs(cliAddr.sin_port), msg);
         log_print();
         }

    return 0;
}

//****************************************************************************
//  signal processing
//****************************************************************************

void  signal_process (int sig)
{
    char    logold  [256];
    
    if   (sig == SIGTERM)
         {
         sprintf(st, "get signal %d (SIGTERM) and stop\n", sig);
         log_print();
     
         fclose(logfp);
         close(sd);
         
         exit(0);
         }

    if   (sig == SIGHUP)
         {
         sprintf(st, "get signal %d (SIGHUP)\n", sig);
         log_print();

         fclose(logfp);
         
         time_string();
         sprintf(logold, "%s.%s", logfile, time_st); 
         rename(logfile, logold);
         
         logfp = fopen (logfile, "w"); 
         }

    return;
}

//****************************************************************************
//  make time
//****************************************************************************

void time_string ()
{
    time_t    now;
    struct    tm *zgr;

    time(&now);
    zgr = localtime(&now);
    strftime(time_st, 80, "%d.%m.%y-%X", zgr);

    return;
}

//****************************************************************************
//  write logfile
//****************************************************************************

void log_print ()
{
    time_string();
    
    fprintf(logfp, "%s %s %s", program, time_st, st);
    fflush(logfp);

    return;
}

//-------------------------------- Ende --------------------------------------


