/*----------------------------------------------------------------
 *       Copyright (C) 2007
 *       Gernot Miksch
 *----------------------------------------------------------------
 *
 * sf2t_sub.c,v 1.00 2007/02/19 17:39:17 
 *
 * Converts SFF to TIFF files
 *
 * original author: Kolja Waschk
 *
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

//#include "config.h"

#define REVERSE_SFFTIFF 1     /* while converting from SFF to TIFF */

// *****************************************************************************
// TIFF structures 
struct tag_s 
{
  short unsigned int tag_type;
  short unsigned int data_type;  
  long  unsigned int data_len;
  union 
  { 
    long unsigned int ptr;
    long unsigned int value;
  } data;
};

// TIFF data-types
#define TT_ASCII    0x02
#define TT_SHORT    0x03
#define TT_LONG     0x04
#define TT_RATIONAL 0x05

#define IFD_TAGS 11

// ImageFileDirectory of TIFF-file *=mandatory for bilevel
struct ifd_s 
{
  struct tag_s t_ImageWidth;                  // 0x100 *
  struct tag_s t_ImageLength;                 // 0x101 *
  struct tag_s t_BitsPerSample;               // 0x102 
  struct tag_s t_Compression;                 // 0x103 *
  struct tag_s t_PhotometricInterpretation;   // 0x106 *
  struct tag_s t_StripOffset;                 // 0x111 *
  struct tag_s t_RowsPerStrip;                // 0x116 *
  struct tag_s t_StripByteCounts;             // 0x117 *
  struct tag_s t_XResolution;                 // 0x11a *
  struct tag_s t_YResolution;                 // 0x11b *
  struct tag_s t_ResolutionUnit;              // 0x128 *
  unsigned long next_ifd;
  unsigned long xres[2];
  unsigned long yres[2];
} *cifd,                                      // pointer to current IFD
  *nifd,                                      // pointer to next IFD
  *ifdhead = NULL,                            // pointer to 1st IFD
  default_ifd =                               // template IFD
  {
    { 0x0100, TT_SHORT, 1L, { 0L } },         // Image Width
    { 0x0101, TT_SHORT, 1L, { 0L } },         // Image Length
    { 0x0102, TT_SHORT, 1L, { 1L } },         // Bits per Sample
    { 0x0103, TT_SHORT, 1L, { 2L } },         // Compression
    { 0x0106, TT_SHORT, 1L, { 0L } },         // Photometric Interpretation
    { 0x0111, TT_LONG,  1L, { 0L } },         // Strip Offset
    { 0x0116, TT_LONG,  1L, { 0L } },         // Rows per Strip
    { 0x0117, TT_LONG,  1L, { 0L } },         // Strip Byte Counts
    { 0x011A, TT_RATIONAL, 1L, { 0L } },      // X Resolution
    { 0x011B, TT_RATIONAL, 1L, { 0L } },      // Y Resolution
    { 0x0128, TT_SHORT, 1L, { 1L } },         // Resolution Unit
    0L,                                       // pointer to next IFD
    { 203L, 1L },                             // horizontal resulution
    { 196L, 1L },                             // vertical resolution
  };

struct
{
  unsigned short byte_order;
  unsigned short version;
  unsigned long  ifd1_offset;
} tif_header = 
  {
    0x4949,                                   // 'II' fuer Little Endian, 'MM' fuer Big Endian
    0x002A,                                   // Versionnumber (42)
    sizeof(tif_header)                        // offset to first IFD, filled in later
  };

// *****************************************************************************
// sff-structures
struct 
{
  unsigned char magic[4];
  unsigned char version;
  unsigned char reserved;
  unsigned short user_info;
  unsigned short page_count;
  unsigned short ofs_firstpage;
  unsigned long  ofs_lastpage;
  unsigned long  ofs_doc_end;
} sff_header;

struct {
  unsigned char vres;
  unsigned char hres;
  unsigned char encoding;
  unsigned char reserved;
  unsigned short line_len;
  unsigned short page_len;
  unsigned long ofs_prev;
  unsigned long ofs_next;
} page_header;

// *****************************************************************************
// reverse bits (B7->B0, B6->B1 ...
// multiply operations seen on http://graphics.stanford.edu/~seander/bithacks.html
inline unsigned char revbits(unsigned char corg)                      
{
  #if REVERSE_SFFTIFF
/*    return  ((corg & 128) >> 7) +
            ((corg &  64) >> 5) +
            ((corg &  32) >> 3) +
            ((corg &  16) >> 1) +
            ((corg &   8) << 1) +
            ((corg &   4) << 3) +
            ((corg &   2) << 5) +
            ((corg &   1) << 7);
*/
  return((corg * 0x0802LU & 0x22110LU) | (corg * 0x8020LU & 0x88440LU)) * 0x10101LU >> 16; 
  #else
    return corg;
  #endif
}

// *****************************************************************************
// output an empty line
#define WANDEOL 3                                                     // EOL is 3 char long

int output_emptyline(long len, FILE *output)                          // output an empty line, 
{                                                                     // len is the length of the line (future use)
  #if REVERSE_SFFTIFF
    fputc(0x4D, output);                                              // 01001101 10011010 10000000 00000001
    fputc(0x9A, output);                                              // 010011011 00110101 00000000000 0001
    fputc(0x80, output);                                              // M1728W    T0W      EOLB         Fill 
  #else
    fputc(0xB2, output); /* 1728 pel white */
    fputc(0x59, output); /* 0 pel black */
    fputc(0x01, output);
  #endif
  return WANDEOL;
}

// *****************************************************************************
// to allow output on stdout without resetting the file pointer
// input data has to be processed in two passes; first the
// format and number of pages is extracted, then the actual
// data is copied in a second pass. */

int pages_in_output = 0, current_page = 0;
long g_StripOffset = 0L;

int convert_sff_to_tiff(char *sffname, char *docname, FILE *output) 
{
  short i, j, k, l, pgo, secondpass, scale=2;
  long nph, written=0, pos;
  FILE *sff;
    
  sff = fopen(sffname, "rb");                                         // open source-file
  if(!sff) 
  {
    fprintf(stderr,"Open error: %s: %s\n", sffname, strerror(errno));
    return -1;
  };
  
  i = fread(&sff_header, sizeof(sff_header), 1, sff);                 // read sff-header into structure
  if(i<0) 
  {
    fprintf(stderr,"Can't read header of %s: %s\n", sffname, strerror(errno));
    return -1;
  };
  
  if(!i || strncmp(sff_header.magic,"Sfff",4))                       // check, if its really an sff
  {
    fprintf(stderr,"%s has no SFF header\n", sffname);
    return -1;
  };

/*  if(docname && *docname)                                             // if we got a doc-name, fill into struct
  {
    default_ifd.t_DocName.data_len = strlen(docname) + 1;
    default_ifd.t_DocName.data.ptr = sizeof(tif_header);              // doc-name follows header
    tif_header.ifd1_offset += (strlen(docname) + 1) & 0xFFFE;         // adjust offset to point behind the name but on the next word-boundary
  };
*/
  for (secondpass=0;secondpass<2;secondpass++)
  {
    cifd=NULL;
    if (secondpass)
    {
      fwrite(&tif_header, sizeof(tif_header), 1, output);             // write the tiff-header
      written = tif_header.ifd1_offset;                               // reset byte-counter of written byte
    }
    fseek(sff, sff_header.ofs_firstpage, SEEK_SET);                   // seek to page-header
    if(i<0) 
    {
      fprintf(stderr,"Can't seek to first page\n");
      return -1;
    };
  
    pgo = 0;                                                          // 0=page-header, !0=page-data
    nph = 0L;                                                         // offset to next page-header
  
    while(!feof(sff))                                                 // read file byte by byte
    {
      i = fgetc(sff);                                                 // get character from file
      if(i<0) 
      {
        fprintf(stderr,"error while reading %s: %s\n", sffname, strerror(errno));
        return -1;
      };
  
      if(pgo)                                                         // we were inside page-data
      {
        switch(i) 
        {
          case 255:                                                   // (error) white line 
            i = fgetc(sff);                                           // check next character
            if(!i)                                                    // if 0 -> empty line
            {             
              if(!secondpass)
              {
                cifd->t_ImageLength.data.value += scale;							// increase length
                cifd->t_StripByteCounts.data.value += scale*WANDEOL;	// 1728 white
              }
              else
              {
                written += output_emptyline(cifd->t_ImageWidth.data.value, output); // output empty line, add number of bytes written
                if(scale-1)
                {
                	written += output_emptyline(cifd->t_ImageWidth.data.value, output); // output empty line, add number of bytes written
                }
              }
            } 
            else                                                      // if <>0, ignore data  
            {
              fseek(sff, i, SEEK_CUR);                                // skip user-data
            };
            break;
  
          case 254:                                                   // new page header or end of file
            pgo = 0;
            break;
  
          default: 
            if(i >= 217)                                              // white lines to skip
            {
              i -= 216;                                               // calculate amount
              if(!secondpass)
              {
                cifd->t_ImageLength.data.value += i*scale;            // add to line-count
                cifd->t_StripByteCounts.data.value += i * scale* WANDEOL; // add to byte-count (1728w each)
              }
              else
              {
                while(i--) 
                {
                  written += output_emptyline(cifd->t_ImageWidth.data.value, output); // output empty lines, add number of bytes written
                  if (scale-1)
                  {
                  	written += output_emptyline(cifd->t_ImageWidth.data.value, output); // output empty lines, add number of bytes written
                  }
                };
              }
            } 
            else                                                      // real data, i contains the number of bytes in this row
            {
              if(!i)                                                  // 0=ESC for more than 216 bytes per line
              {
                fread(&i, sizeof(i), 1, sff);                         // read number of bytes
              }
              
              if (!secondpass)
              {
                cifd->t_ImageLength.data.value += scale;              // increment line-count
      
                if(i>1)                                               // more than 1 byte
                {
                  fseek(sff, i-1, SEEK_CUR);                          // seek to next line
                }
                l = fgetc(sff);                                       // dummy-read (###warum nicht if(i>0) seek(sff,i,SEEK_CUR);??)
          
                cifd->t_StripByteCounts.data.value += i * scale;      // add to byte-count
              }
              else
              {
                written += i * scale;                                 // add number of bytes written (in the next step)
								pos=ftell(sff);																				// save actual fileposition
								j=scale;																		  				// prepare scaling
								while(j--)
								{
									fseek(sff,pos, SEEK_SET);														// seek to saved position	
									k=i;
	                while(k--)                                          // write all data of this line
	                { 
	                  fputc(revbits(fgetc(sff)), output);               // reverse bit-order
	                }
	              }
              };
            };
        };
      };
      
      if(!pgo)                                                        // read and check page-header
      { 
        if(i != 254)                                                  // check if header, no -> Error
        {
          fprintf(stderr,"expected page header, found 0x%02x at %08lx@%srecord\n",i, ftell(sff)-1, sffname);
          if(nph && nph != ftell(sff))                                // we know the position of the next header?
          {
            fprintf(stderr, "skipping to next page header\n");
            fseek(sff, nph, SEEK_SET);
            nph = 0L;
          } 
          else 
          {
            fprintf(stderr, "skipping remainder of SFF file %s\n", sffname);
          };
        } 
        else                                                          // header found:
        { 
          l = fgetc(sff);                                             // header-length, 0=EOF
          if(l<1) break;                                              // EOF -> leave
  
          i = fread(&page_header, sizeof(page_header), 1, sff);       // read rest of page-header
          if(i<1) 
          {
            fprintf(stderr,"error while reading page header from %s: %s\n",sffname, strerror(errno));
            break;
          };
  
          if(!secondpass)
          {
            if(!ifdhead)                                              // do we have already a TIFF-IFD allocated?
            {
              ifdhead = malloc(sizeof(struct ifd_s));                 // no, reserve space
              cifd = ifdhead;                                         // set current IFD to point to allocated space
            }
            else                                                      // new page, allocate space
            {
              cifd->next_ifd = (unsigned long)malloc(sizeof(struct ifd_s)); // save pointer to next IFD in current IFD
              cifd = (struct ifd_s *) cifd->next_ifd;                 // make next IFD the current
            };
          
            memcpy(cifd, &default_ifd, sizeof(struct ifd_s));         // preset current IFD
  
            pages_in_output++;                                        // increment no of pages
  
//            cifd->yres[0] = page_header.vres ? 196L : 98L;          // set vertical resolution in accordance to SFF-Header
            scale = page_header.vres ? 1 : 2;            							// select to scale vertical resolution in accordance to SFF-Header
            cifd->t_ImageWidth.data.value = page_header.line_len;     // set horizontal length
          }
          else
          {
            if(!cifd)                                                 // current IFD already set?
            {
              cifd = ifdhead;                                         // no, set to 1st
            }
            else
            {
              cifd = nifd;                                            // yes, set to next
            };

            if (written%2)                                            // filler to start next IFD at word-boundary
            {
              fputc(0,output);
              written++;
            }

            i = IFD_TAGS;                                             // set number of tags in IFD
            fwrite(&i, sizeof(i), 1, output);                         // write this number to the file
            written += sizeof(i);                                     // increase bytes written
    
            current_page++;
            nifd = (struct ifd_s *) cifd->next_ifd;                   // read pointer to next IFD (here in memory)
    
            if(current_page != pages_in_output)                       // if not at the last page,
            {
              cifd->next_ifd = (written + sizeof(struct ifd_s) +      // set structure-element to conatin offset of the next IFD
                               cifd->t_StripByteCounts.data.ptr + 1) & 0xFFFFFFFE;  // within the file
            };
    
            cifd->t_StripOffset.data.ptr = written + sizeof(struct ifd_s);
            cifd->t_XResolution.data.ptr = written + (unsigned long)&(cifd->xres) - (unsigned long)cifd;
            cifd->t_YResolution.data.ptr = written + (unsigned long)&(cifd->yres) - (unsigned long)cifd;
            cifd->t_RowsPerStrip.data.value = cifd->t_ImageLength.data.value;
            
            fwrite(cifd, sizeof(struct ifd_s), 1, output);
            written += sizeof(struct ifd_s);
          }
          
          nph = page_header.ofs_next;
          if(nph == 1L) nph = 0;
          fseek(sff, l - sizeof(page_header), SEEK_CUR);              // seek to data
          pgo = 1;                                                    // set flag to indicate that data has to be processed
        };
      };
    };
  };
  return 0;
};

