/*
 * wrapd.c: program to control the pushbutton switch
 * on a WRAP board
 *
 * portions based on
 * (c) 2005 by Torsten Georg <fli4l<at>foddy.de>
 * 
 * $Id: wrapd.c 11670 2006-06-29 07:17:25Z bastard $
 * 
 * wrapd reads from a fifo file commands to execute.
 * valid commands are: led1|led2|led3 on|off|blink
 *
 * (c) Copyright 2006 Stefan Krister <stefan@fli4l.de>
 * and the fli4l developer-team, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * Version 2 as published by the Free Software Foundation.
 * 
 * I don't accept liability nor provide warranty for this Software.
 * This material is provided "AS-IS" and at no charge.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <syslog.h>
#include <sys/io.h>
#include <sys/types.h>
#include <sys/stat.h>

// Define I/O port for the LEDs
#define GPIOBASE 0xF400 // Only valid with tinyBIOS

// buffer for fifo
#define BUFFERSIZE 1024

//------------------------------------------------------------------------------
// usage
//
// print help text / usage information
//------------------------------------------------------------------------------
// Print usage message 
int
usage(char * argv)
{
    printf("--------------------------------------------------------------\n");
    printf(" wrapd usage\n");
    printf("\n");
    printf(" Usage: %s command keyprog [debug]\n", argv);
    printf("--------------------------------------------------------------\n");
    exit (1);
}

int
led1on()
{
    outb(inb( GPIOBASE ) & 0xFB, GPIOBASE);
}

int
led1off()
{
    outb(inb( GPIOBASE ) | 0x04, GPIOBASE);
}

int
led2on()
{
    outb(inb( GPIOBASE ) & 0xF7, GPIOBASE);
}

int
led2off()
{
    outb(inb( GPIOBASE ) | 0x08, GPIOBASE);
}

int
led3on()
{
    outb(inb( GPIOBASE + 2 ) & 0xFB, GPIOBASE + 2);
}

int
led3off()
{
    outb(inb( GPIOBASE + 2) | 0x04, GPIOBASE + 2);
}

/******************************************************************************/
/*                                                                            */
/*                          M A I N   P R O G R A M                           */
/*                                                                            */
/******************************************************************************/
int
main (int argc, char **argv)
{
    char * debug   = (char *) NULL;
    char   command[1024];
    char   buf[BUFFERSIZE], zeichen; 

    int switch_status, old_switch_status, res, fd, i;
    int led1, led2, led3;
    
    openlog("wrapd", LOG_PID, LOG_USER);
    
    // check given arguments
    //
    // check number of arguments
    if (argc > 2)
    {
	snprintf (command, sizeof(command) - sizeof(argv[2]) - 1, "%s %s", argv[2], argv[1]);
	syslog(LOG_INFO, "starting ...");
	syslog(LOG_INFO, "command = %s", command);
    }
    else           // give help
    {
        usage(argv[0]);
    }
                                             
    if (argc > 3)
    {
	debug = argv[3];
	syslog(LOG_INFO, "debug mode enabled");
    }

    // switch_status
    // 1 = released
    // 0 = pressed
  
    switch_status = 1;
    old_switch_status = 1;

    // LED status
    // 0 = off
    // 1 = on
    // 2 = blink, current on
    // 3 = blink, current off
    
    led1 = 0;
    led2 = 0;
    led3 = 0;

    // Get access to the I/O ports
    if ( iopl(3) < 0 )
    {
	syslog(LOG_ERR, "could not set I/O permissions to level 3");
	exit (1);
    }

    led1off();
    led2off();
    led3off();
    
    // Delete (old) FIFO file
    unlink("/var/run/wrap.fifo");

    // Create FIFO file
    res = mkfifo("/var/run/wrap.fifo", 0777);
    if(res != 0)
    {
	syslog(LOG_ERR, "could not create FIFO file");
	exit (1);
    }
    
    if (daemon(1, 1) == -1)
    {
        printf("Cannot detach!\n");
	syslog(LOG_ERR, "cannot detach");
	exit(1);
    }

    fd = open("/var/run/wrap.fifo", O_RDONLY | O_NONBLOCK);
    if(fd == -1)
    {
	syslog(LOG_ERR, "could not open FIFO file");
	exit (1);
    }
    
    while (1) {
	// read status of pushbutton switch
        // 1 = released
        // 0 = pressed

	switch_status = inb( GPIOBASE + 0x15 ) & 1;
	if(debug)
	{
	    syslog(LOG_INFO, "button is %s", switch_status ? "released" : "pressed");
	}
	if ( old_switch_status == 1 && switch_status == 0)
	{
            syslog(LOG_INFO, "button pressed, executing command %s", command);
	    system(command);
	}
	old_switch_status = switch_status;

	// read FIFO file
	res = read(fd, &zeichen, sizeof(char));
	i=0;
	while(res != 0)
	{
            buf[i]=zeichen;
	    res =  read(fd, &zeichen, sizeof(char));
	    i++;
	    if('\n' == zeichen || BUFFERSIZE - 1 < i ) // avoid buffer overflow
	    {
		    break;
	    }
	}
        buf[i]='\0';
	
	if(i > 0)
	{
            if(debug)
	    {
	        syslog(LOG_INFO, "gelesen: \"%s\"", buf);
	    }
            if(! strcmp(buf, "LED1 on"))
	    {
		led1on();
		led1 = 1;
		syslog(LOG_INFO,"switching LED1 on");
	    }
	    if(! strcmp(buf, "LED1 off"))
	    {
		led1off();
		led1 = 0;
		syslog(LOG_INFO,"switching LED1 off");
	    }
	    if(! strcmp(buf, "LED1 blink"))
            {
	        led1 = 3;
		if(debug)
		    syslog(LOG_INFO,"blinking LED1");
            }
            if(! strcmp(buf, "LED2 on"))
	    {
		led2on();
		led2 = 1;
		    syslog(LOG_INFO,"switching LED2 on");
	    }
	    if(! strcmp(buf, "LED2 off"))
	    {
		led2off();
		led2 = 0;
		    syslog(LOG_INFO,"switching LED2 off");
	    }
	    if(! strcmp(buf, "LED2 blink"))
	    {
	        led2 = 3;
		    syslog(LOG_INFO,"blinking LED2");
	    }
            if(! strcmp(buf, "LED3 on"))
	    {
		led3on();
		led3 = 1;
		    syslog(LOG_INFO,"switching LED3 on");
	    }
	    if(! strcmp(buf, "LED3 off"))
	    {
		led3off();
		led3 = 0;
		    syslog(LOG_INFO,"switching LED3 off");
	    }
	    if(! strcmp(buf, "LED3 blink"))
	    {
	        led3 = 3;
		    syslog(LOG_INFO,"blinking LED3");
	    }
	    if(! strcmp(buf, "STATUS"))
	    {
	        switch(led1)
		{
		    case 0:
		        syslog(LOG_INFO,"status: LED1 off");
			break;
		    case 1:
		        syslog(LOG_INFO,"status: LED1 on");
			break;
		    case 2 | 3:
		        syslog(LOG_INFO,"status: LED1 blink");
			break;
		}
	        switch(led2)
		{
		    case 0:
		        syslog(LOG_INFO,"status: LED2 off");
			break;
		    case 1:
		        syslog(LOG_INFO,"status: LED2 on");
			break;
		    case 2 | 3:
		        syslog(LOG_INFO,"status: LED2 blink");
			break;
		}
	        switch(led3)
		{
		    case 0:
		        syslog(LOG_INFO,"status: LED3 off");
			break;
		    case 1:
		        syslog(LOG_INFO,"status: LED3 on");
			break;
		    case 2 | 3:
		        syslog(LOG_INFO,"status: LED3 blink");
			break;
		}
	        switch(switch_status)
		{
		    case 0:
		        syslog(LOG_INFO,"status: button pressed");
			break;
		    case 1:
		        syslog(LOG_INFO,"status: button released");
			break;
		}

	    }
	}
	// handle blinking LEDs
	if(led1 == 2)
	{
            led1off();
	    led1=3;
	}
	else if (led1 == 3)
	{
	    led1on();
	    led1=2;
	}
	
	if(led2 == 2)
	{
            led2off();
	    led2=3;
	}
	else if(led2 == 3)
	{
	    led2on();
	    led2=2;
	}
	
	if(led3 == 2)
	{
            led3off();
	    led3=3;
	}
	else if(led3 == 3)
	{
	    led3on();
	    led3=2;
	}
        
	sleep(1);
    }
    return 0;
}

