#include <termios.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <limits.h>
#include <sys/stat.h>
#include <errno.h>
#include <getopt.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>

#define LOCK_PATH "/var/lock"
#define LOCK_FILE_PREFIX "LCK.."
#define CALLER_NUMBER "CALLER NUMBER:"
#define MAX_NUMBER_LEN 100
#define BAUD_RATE B115200
#define DLE 0x10
#define ETX 0x03
#define DC4 0x14
#define TIMEOUT 2
#define RINGS 2

static int debug=1;
static int interrupted;
static void sigint(int signal);
static pid_t pid=0;

static void sigint(int signal)
{
  (void) signal;
  interrupted = 1;
  if(pid>0)
  {
    kill(pid, SIGINT);
  }
  return;
}


class getTty
{
  protected:
    getTty(char *dev=NULL);
    ~getTty(void);
  public:
    int getFd(void);
  private:
    void lockTty(char *device);
    void lockTty(void);
    void unlockTty(void);
    void openFd(void);
    void closeFd(void);
    char *basename(char *f);
    int fd;
    char *lockFile;
    char *device;
    struct termios oldTermios;
};

class initTty : public getTty
{
  protected:
    initTty(char *localMsn, char *dev=NULL);
    ~initTty(void);
    int executeCommand(
      char *command, 
      char *expectedResponse, 
      int timeout, 
      int stopFd=0);
    char *modemResponse(void);
    void sendCommand(char *command);
  private:
    int waitForResponse(char *expectedResponse, int timeout, int stopFd=0);
    char realResponse[256];
    static char *const initCommands[];
};

class call : public initTty
{
  public:
    call(char *localMsn, char *dev=NULL);
    ~call(void);
    int connect(char* remoteNumber=NULL, int stopFd=0);
    char *playMessageAndGetTouchTones(char *fileName);
    void trash(int stopFd);
    int detectCarrier(void);
    bool msnIsValid(char *msnNumbers);
    bool pinIsValid(char *pinNumbers, char *pinNumber);
  private:
    int switchToVoice(void);
    char detectTouchTones(char *data, int size);
    char touchtoneNumberBuffer[MAX_NUMBER_LEN];
    char *getCallingNumber(void);
    char callingNumberBuffer[MAX_NUMBER_LEN];
};

class connection
{
  public:
    connection(
      char *localMsnIn, 
      char *localMsnOut, 
      char *devIn=NULL, 
      char *devOut=NULL);
    ~connection(void);
    void establishCall(
      char *numberMsg=NULL, 
      char *pinMsg=NULL, 
      char *errorMsg=NULL);
  private:
    void copyData(void);
    call *incomming, *outgoing;
    char numberMsg[100];
    char pinMsg[100];
    char errorMsg[100];
    char numberTxt[100];
    char pinTxt[100];
};


getTty::getTty(char *dev)
{
  if(debug)
  {
    fprintf(stderr, "executing getTty constructor\n");
  }
  fd=0;
  interrupted=0;
  signal(SIGINT, &sigint);

  if(dev)
  {
    lockTty(dev);
  }
  else
  {
    lockTty();
  }

  if(lockFile)
  {
    openFd();
    if(fd<0)
    {
      fprintf(stderr, "Could open device %s.\n", device);
    }
  }
  else
  {
    fprintf(stderr, "Could not get lock on an ISDN tty.\n");
  }
  return;
}

getTty::~getTty(void)
{
  if(debug)
  {
    fprintf(stderr, "executing getTty destructor\n");
  }
  if(fd>0)
  {
    closeFd();
  }
  if(lockFile)
  {
    unlockTty();
    free(lockFile);
    free(device);
  }
  signal(SIGINT, SIG_DFL);
  return;
}

int getTty::getFd(void)
{
  return fd;
}

void getTty::lockTty(void)
{
  int i;
  char dev[256];

  for (i = 2; i <= 20; i++)
  {
    snprintf(dev, sizeof(dev), "/dev/ttyI%i", i);
    lockTty(dev);
    if(lockFile)
    {
      if (debug)
      {
        fprintf(stderr, "foud free device: %s\n", dev);
      }
      return;
    }
  }
  if (debug)
  {
    fprintf(stderr, "Could not find free device.\n");
  }
  return;
}

void getTty::lockTty(char *dev)
{
  struct stat stt;
  char buf[256];

  if (stat(LOCK_PATH, &stt) == 0)
  {
    int fdLock;
    char lf[PATH_MAX]="";
    snprintf(
      lf, 
      sizeof(lf), 
      "%s/%s%s", 
      LOCK_PATH, 
      LOCK_FILE_PREFIX, 
      basename(dev));
    lockFile=strdup(lf);
      
    if ((fdLock = open(lockFile, O_RDONLY)) >= 0) 
    {
      int n = read(fdLock, buf, sizeof(buf) - 1);
      close(fdLock);

      if (n > 0)
      {
        int lockPid;
        buf[n] = 0;
        sscanf(buf, "%d", &lockPid);
      
        if (lockPid > 0)
        {
          if (kill((pid_t) lockPid, 0) < 0 && errno == ESRCH) 
          {
            fprintf(stderr, "Lockfile is stale. Overriding it.\n");
            unlink(lockFile);
          }
          else
          {
            lockFile=NULL;
            return;
          }
        }
      }
    }

    if ((fdLock = open(lockFile, O_WRONLY | O_CREAT | O_EXCL, 0666)) < 0)
    {
      lockFile=NULL;
      return;
    }
      
    snprintf(buf, sizeof(buf), "%ld\n", (long) getpid());
    write(fdLock, buf, strlen(buf));
    close(fdLock);

    device=strdup(dev);
    return;
  }
  else
  {
    lockFile=NULL;
    return;
  }
}

void getTty::unlockTty(void)
{
  unlink(lockFile);
}

void getTty::openFd(void)
{
  struct termios pts; 
  int flag;
   
  fd = open(device, O_RDWR|O_NONBLOCK);
  if (fd < 0)
  {
    return;
  }
  // flag = fcntl(fd, F_GETFL, 0);
  // fcntl(fd, F_SETFL, flag & ~O_NONBLOCK);

  if (tcgetattr(fd, &oldTermios) != 0)
  {
    close(fd);
    return;
  }
   
  bzero(&pts, sizeof pts);
  pts.c_cflag = CRTSCTS | IGNPAR | HUPCL | CS8;
  pts.c_iflag = IGNPAR;
  pts.c_oflag = 0;
  pts.c_lflag = 0;

  pts.c_cc[VMIN] = 1;
  pts.c_cc[VTIME] = 0;

  cfsetospeed(&pts, BAUD_RATE);
  cfsetispeed(&pts, BAUD_RATE);

  tcflush(fd, TCIFLUSH);
  if (tcsetattr(fd, TCSANOW, &pts) != 0)
  {
    close(fd);
    return;
  }

  return;
}

void getTty::closeFd(void)
{
  tcflush(fd, TCIFLUSH);
  tcsetattr(fd, TCSANOW, &oldTermios);
  close(fd);
}

char *getTty::basename(char *f)
{
  char *i = strrchr(f, '/');

  return i ? i+1 : f;
}


initTty::initTty(char *localMsn, char *dev)
  : getTty(dev)
{
  if(debug)
  {
    fprintf(stderr, "executing initTty constructor\n");
  }
  if(getFd()>0)
  {
    int i;
    for(i=0; initCommands[i]!=0; i+=2)
    {
      realResponse[0]='\0';
      if(!executeCommand(initCommands[i], initCommands[i+1], TIMEOUT))
      {
        fprintf(stderr, "Initialisation failure, sent: %s", initCommands[i]);
        fprintf(stderr, "expected response: %s", initCommands[i+1]);
        fprintf(stderr, "got response: %s", realResponse);
        return;
      }
    }
    char buffer[1024];
    snprintf(buffer, sizeof(buffer), "AT&E%s\n", localMsn);
    realResponse[0]='\0';
    if(!executeCommand(buffer, "OK\r\n", TIMEOUT))
    {
        fprintf(stderr, "Initialisation failure, sent: %s", buffer);
        fprintf(stderr, "expected response: OK\r\n");
        fprintf(stderr, "got response: %s\n", realResponse);
        return;
    }
  }
  return;
}

initTty::~initTty(void)
{
  if(debug)
  {
    fprintf(stderr, "executing initTty destructor\n");
  }
  if(getFd()>0)
  {
    sendCommand("AT&F\n");
    sleep(1);
  }
  // ~getTty();
}

int initTty::executeCommand(
  char *command, 
  char *expectedResponse, 
  int timeout, 
  int stopFd)
{
  int ret;
  sendCommand(command);
  ret=waitForResponse(expectedResponse, timeout, stopFd);
  return ret;
}

void initTty::sendCommand(char *command)
{
  write (getFd(), command, strlen(command));
  
  if (debug) 
  {
    fprintf(stderr, "-> %s", command);
  }
}

char *initTty::modemResponse(void)
{
  return realResponse;
}

int initTty::waitForResponse(char *expectedResponse, int timeout, int stopFd)
{
  char *p = expectedResponse;
  int quit = 0;

  if (debug)
  {
    fprintf(stderr, "<- ");
  }

  int j=0;
  while (! quit)
  {
    int t, i, l;
    unsigned char c;
    fd_set fds;
    struct timeval tv, *tvp;

    if (interrupted)
    {
      break;
    }
      
    if (timeout != -1)
    {
      tv.tv_sec = timeout;
      tv.tv_usec = 0;
     
      tvp = &tv;
    }
    else 
    {
      tvp = NULL;
    }

    FD_ZERO(&fds);
    FD_SET(getFd(), &fds);
    if(stopFd>0)
    {
      FD_SET(stopFd, &fds);
    }

    select(FD_SETSIZE, &fds, 0, 0, tvp);

    if(stopFd>0)
    {
      if(FD_ISSET(stopFd, &fds))
      {
        char buf[100];
        l = read(stopFd, buf, sizeof(buf));
        if(l>=0)
        {
          fprintf(stderr, "stopped call attempt\n");
          break;
        }
        else
        {
          errno=0;
          snprintf(buf, sizeof(buf), "read() from pipe returned %d", l);
          perror(buf);
          return 0;
        }
      }
    }

    t = read(getFd(), &c, 1);

    if (t == 1)
    {
      if (debug)
      {
        fprintf(stderr, "%c", c >= 32 && c <= 126 ? c : '.');
      }

      realResponse[j]=c;
      j++;

      if (c == *p)
      {
        p++;
        if (*p == 0)
        {
          quit = 1;
          break;
        }
      }
      else
      {
        p = expectedResponse;
      }
    }
  }
  realResponse[j]='\0';
  
  if (debug)
  {
    fprintf(stderr, "\n");
  }
  return quit;
}

char *const initTty::initCommands[] =
{
  "AT&F\n",
  "OK\r\n",

  "ATI\n",
  "Linux ISDN\r\nOK\r\n",

  "AT&B128\n",
  "OK\r\n",

  "AT+FCLASS=8\n", 
  "OK\r\n",

  "AT+VSM=6\n", 
  "OK\r\n",

  "ATS18=1\n",
  "OK\r\n",

  "ATS14=4\n", 
  "OK\r\n",

  "ATS13.4=1\n",
  "OK\r\n",

  "ATS23=1\n",
  "OK\r\n",

  0
};


call::call(char *localMsn, char *dev)
  : initTty(localMsn, dev)
{
  return;
}

call::~call(void)
{
  char buffer[2];
  touchtoneNumberBuffer[0]='\0';
  buffer[0] = DLE;
  buffer[1] = DC4;
  write (getFd(), buffer, 2);
  buffer[0] = DLE;
  buffer[1] = ETX;
  write (getFd(), buffer, 2);
}

int call::connect(char *remoteNumber, int stopFd)
{
  char buffer[256];
  if(remoteNumber)
  {
    snprintf(buffer, sizeof(buffer), "ATD%s\n", remoteNumber);
  }
  else
  {
    snprintf(buffer, sizeof(buffer), "ATS0=%i\n", RINGS);
  }
  if(!executeCommand(buffer, "VCON\r\n", -1, stopFd))
  {
    fprintf(stderr, "Connect failure, sent: %s\n", buffer);
    fprintf(stderr, "expected response: VCON\r\n");
    fprintf(stderr, "got response: %s\n", modemResponse());
    return 1;
  }
  if(debug)
  {
    fprintf(stderr, "call accepted: %s\n", modemResponse());
  }
  if(!remoteNumber)
  {
    char *c;
    if(c=getCallingNumber())
    {
      strncpy(callingNumberBuffer, c, MAX_NUMBER_LEN);
    }
  }
  return switchToVoice();
}

char *call::playMessageAndGetTouchTones(char *fileName)
{
  fd_set fdReadSet, fdWriteSet;
  int messageFd;
  int isdnFd=getFd();
  char buf[1024], c;
  int l, numCount=0;
  if((messageFd=open(fileName, O_RDONLY))<0)
  {
    snprintf(buf, sizeof(buf), "error opening %s\n", fileName); 
    perror(buf);
  }
  while(!interrupted)
  {
    FD_ZERO(&fdReadSet);
    FD_ZERO(&fdWriteSet);
    FD_SET(isdnFd, &fdReadSet);
    FD_SET(isdnFd, &fdWriteSet);
    if(detectCarrier()<=0)
    {
      if(debug)
      {
        fprintf(stderr, "exit due to hangup while gathering touchtones\n");
        return NULL;
      }
    }
    if(select(FD_SETSIZE, &fdReadSet, &fdWriteSet, NULL, NULL)<=0)
    {
      perror("select() returned error");
      return NULL;
    }
    if(FD_ISSET(isdnFd, &fdWriteSet))
    {
      l = read(messageFd, buf, sizeof(buf));
      if(l<0)
      {
        snprintf(buf, sizeof(buf), "read() from message file returned %d", l);
        perror(buf);
        return NULL;
      }
      write(isdnFd, buf, l);
    }
    if(FD_ISSET(isdnFd, &fdReadSet))
    {
      l = read(isdnFd, buf, sizeof(buf));
      if(l<0)
      {
        snprintf(buf, sizeof(buf), "read() from ISDN device returned %d", l);
        perror(buf);
        return NULL;
      }
      if(c=detectTouchTones(buf, l))
      {
        if(c=='#' || numCount==MAX_NUMBER_LEN-2)
        {
          touchtoneNumberBuffer[numCount]='\0';
          break;
        }
        else
        {
          touchtoneNumberBuffer[numCount++]=c;
        }
      }
    }
  }
  return touchtoneNumberBuffer;
}

void call::trash(int stopFd)
{
  fd_set fdReadSet;
  int isdnFd=getFd();
  char buf[1024];
  int l;
  while(!interrupted)
  {
    FD_ZERO(&fdReadSet);
    FD_SET(isdnFd, &fdReadSet);
    FD_SET(stopFd, &fdReadSet);
    if(detectCarrier()<=0)
    {
      if(debug)
      {
        fprintf(stderr, "exit due to hangup while trashing\n");
      }
      return;
    }
    if(select(FD_SETSIZE, &fdReadSet, NULL, NULL, NULL)<=0)
    {
      perror("select() returned error");
      return;
    }
    if(FD_ISSET(isdnFd, &fdReadSet))
    {
      l = read(isdnFd, buf, sizeof(buf));
      if(l<0)
      {
        errno=0;
        snprintf(buf, sizeof(buf), "read() from ISDN device returned %d", l);
        perror(buf);
        return;
      }
    }
    if(FD_ISSET(stopFd, &fdReadSet))
    {
      l = read(stopFd, buf, sizeof(buf));
      if(l>=0)
      {
        fprintf(
          stderr, 
          "outgoing call established, stop trashing incomming data\n");
        return;
      }
      else
      {
        errno=0;
        snprintf(buf, sizeof(buf), "read() from pipe returned %d", l);
        perror(buf);
        return;
      }
    }
  }
  return;
}

int call::switchToVoice(void)
{
  if (!executeCommand("AT+VTX+VRX\n", "CONNECT\r\n\r\nCONNECT\r\n", TIMEOUT))
  {
    fprintf(stderr, "Could not switch to full-duplex audio, ");
    fprintf(stderr, "sent: AT+VTX+VRX\n");
    fprintf(stderr, "expected response: CONNECT\r\n\r\nCONNECT\r\n");
    fprintf(stderr, "got response: %s\n", modemResponse());
    return 1;
  }
  return 0;
}

char *call::getCallingNumber(void)
{
  int i;
  char *c=modemResponse();
  int l1=strlen(c);
  int l2=strlen(CALLER_NUMBER);
  for(i=0; i<l1-l2; i++)
  {
    if(!strncmp(c++, CALLER_NUMBER, l2))
    {
      int j;
      c+=l2;
      for(j=0; j<strlen(c); j++)
      {
        if(c[j]=='\n' || c[j]=='\r')
        {
          c[j]='\0';
          break;
        }
      }
      if(debug)
      {
        fprintf(stderr, "calling number: %s\n", c);
      }
      return c;
    }
  }
  return NULL;
}

int call::detectCarrier(void)
{
  unsigned int arg;
  int status = ioctl(getFd(), TIOCMGET, &arg);
  if (status == 0)
  {
    return !! (arg & TIOCM_CAR);
  }
  else
  {
    return -1;
  }
}

bool call::msnIsValid(char *msnNumbers)
{
  FILE *fp;
  bool ret=false;
  char buffer[256];
  int len;
  if(!callingNumberBuffer)
  {
    return false;
  }
  if(!(fp=fopen(msnNumbers, "r")))
  {
    fprintf(stderr, "could not open %s\n", msnNumbers);
    return false;
  }
  while(fgets(buffer, sizeof(buffer), fp))
  {
    len=strlen(buffer)-1;
    if(buffer[len]=='\n')
    {
      buffer[len]='\0';
    }
    if(!strcmp(buffer, callingNumberBuffer))
    {
      return true;
    }
  }
  return false;
}

bool call::pinIsValid(char *pinNumbers, char *pinNumber)
{
  FILE *fp;
  bool ret=false;
  char buffer[256];
  int len;
  if(!pinNumber)
  {
    return false;
  }
  if(!(fp=fopen(pinNumbers, "r")))
  {
    fprintf(stderr, "could not open %s\n", pinNumbers);
    return false;
  }
  while(fgets(buffer, sizeof(buffer), fp))
  {
    len=strlen(buffer)-1;
    if(buffer[len]=='\n')
    {
      buffer[len]='\0';
    }
    if(!strcmp(buffer, pinNumber))
    {
      return true;
    }
  }
  return false;
}

char call::detectTouchTones(char *data, int size)
{
  char c='\0';
  int touchToneCount=0, i;
  for(i=0; i<size; i++)
  {
    if(data[i]==DLE)
    {
      if(i==size-1)
      {
        fprintf(stderr, "DLE at end of buffer received, ignored!\n");
        break;
      }
      switch(data[++i])
      {
        case DLE:
          // fprintf(stderr, "DLE received at pos: %d\n", i);
          break;
        case ETX:
          fprintf(stderr, "ETX received at pos: %d\n", i);
          break;
        case DC4:
          fprintf(stderr, "DC4 received at pos: %d\n", i);
          break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
        case '#':
          touchToneCount++;
          if(debug)
          {
            fprintf(stderr, "received touchtone %c at pos: %d\n", data[i], i);
          }
          c=data[i];
          break;
        case '*':
        case 'A':
        case 'B':
        case 'C':
        case 'D':
          fprintf(stderr, "received touchtone %c at pos: %d\n", data[i], i);
          break;
        default:
          break;
      }
    }
  }
  if(touchToneCount>1)
  {
    fprintf(stderr, "%d touchtones received in one block, ");
    fprintf(stderr, "previous touchtones ignored\n", touchToneCount);
  }
  return c;
}


connection::connection(
  char *localMsnIn, 
  char *localMsnOut, 
  char *devIn, 
  char *devOut)
{
  incomming=new call(localMsnIn, devIn);
  outgoing=new call(localMsnOut, devOut);
  return;
}

connection::~connection(void)
{
  delete incomming;
  delete outgoing;
  return;
}

void connection::establishCall(char *number, char *pin, char *error)
{
  char *calledNumber;
  char *pinNumber;
  int status;
  int fdp[]={0, 0};
  
  if(number)
  {
    snprintf(numberMsg, sizeof(numberMsg), "%s.msg", number); 
    snprintf(numberTxt, sizeof(numberTxt), "%s.txt", number); 
  }
  else
  {
    numberMsg[0]='\0';
    numberTxt[0]='\0';
  }
  if(pin)
  {
    snprintf(pinMsg, sizeof(pinMsg), "%s.msg", pin); 
    snprintf(pinTxt, sizeof(pinTxt), "%s.txt", pin); 
  }
  else
  {
    pinMsg[0]='\0';
    pinTxt[0]='\0';
  }
  if(error)
  {
    snprintf(errorMsg, sizeof(errorMsg), "%s.msg", error); 
  }
  else
  {
    errorMsg[0]='\0';
  }

  incomming->connect();
  if(!incomming->msnIsValid(numberTxt))
  {
    if(!pin)
    {
      return;
    }
    pinNumber=incomming->playMessageAndGetTouchTones(pinMsg);
    for(int i=0; !incomming->pinIsValid(pinTxt, pinNumber); i++)
    {
      if(i>3)
      {
        return;
      }
      pinNumber=incomming->playMessageAndGetTouchTones(errorMsg);
    }
  }

  calledNumber=incomming->playMessageAndGetTouchTones(numberMsg);
  if(debug)
  {
    fprintf(stderr, "received touchtones: %s\n", calledNumber);
  }
  if(pipe(fdp)!=0)
  {
    perror("failed to create pipe");
    return;
  }
  if(!interrupted)
  {
    if(pid=fork())
    { // parent
      outgoing->connect(calledNumber, fdp[0]);
      write(fdp[1], "1", 1);
      wait(&status);
      copyData();
      close(fdp[0]);
      close(fdp[1]);
    }
    else
    { // child
      incomming->trash(fdp[0]);
      write(fdp[1], "1", 1);
      exit(0);
    }
  }
  return;
}

void connection::copyData(void)
{
  fd_set fdReadSet;
  int incommingFd=incomming->getFd();
  int outgoingFd=outgoing->getFd();
  char buf[1024];
  int l;
  while(!interrupted)
  {
    FD_ZERO(&fdReadSet);
    FD_SET(incommingFd, &fdReadSet);
    FD_SET(outgoingFd, &fdReadSet);
    if(incomming->detectCarrier()<=0)
    {
      if(debug)
      {
        fprintf(stderr, "exit due to local hangup\n");
      }
      return;
    }
    if(outgoing->detectCarrier()<=0)
    {
      if(debug)
      {
        fprintf(stderr, "exit due to remote hangup\n");
      }
      return;
    }
    if(select(FD_SETSIZE, &fdReadSet, NULL, NULL, NULL)<=0)
    {
      perror("select() returned error");
      return;
    }
    if(FD_ISSET(incommingFd, &fdReadSet))
    {
      l = read(incommingFd, buf, sizeof(buf));
      if(l<0)
      {
        errno=0;
        snprintf(
          buf, 
          sizeof(buf), 
          "read() from incomming ISDN device returned %d", 
          l);
        perror(buf);
        return;
      }
      write(outgoingFd, buf, l);
    }
    if(FD_ISSET(outgoingFd, &fdReadSet))
    {
      l = read(outgoingFd, buf, sizeof(buf));
      if(l<0)
      {
        errno=0;
        snprintf(
          buf, 
          sizeof(buf), 
          "read() from outgoing ISDN device returned %d", 
          l);
        perror(buf);
        return;
      }
      write(incommingFd, buf, l);
    }
  }
  return;
}


int main(int argc, char *argv[])
{
  char buffer[256];
  if(argc!=4 && argc!=6)
  {
    fprintf(stderr, "Usage: %s InMsn OutMsn Number [Pin Error]\n", argv[0]);
    fprintf(stderr, "       InMsn:  the listening MSN for incomming calls\n");
    fprintf(stderr, "       OutMsn: the MSN used for an outgoing call\n");
    fprintf(stderr, "       Number: both name of the file that contains the\n");
    fprintf(stderr, "               MSNs that are allowed to call in, but\n");
    fprintf(stderr, "               with the extension .txt and the name of\n");
    fprintf(stderr, "               the file that contains the voice\n");
    fprintf(stderr, "               message that asks for the remote number\n");
    fprintf(stderr, "               to be connected to, but with the\n");
    fprintf(stderr, "               extension .msg\n");
    fprintf(stderr, "       Pin:    optional, both name of the file that\n");
    fprintf(stderr, "               contains the pin codes that are allowed\n");
    fprintf(stderr, "               to dial in, but with the extension .txt\n");
    fprintf(stderr, "               and the name of the file that contains\n");
    fprintf(stderr, "               the voice message that asks for the\n");
    fprintf(stderr, "               the pin, but with the extension .msg\n");
    fprintf(stderr, "       Error:  optional, but mandatory when Pin is\n");
    fprintf(stderr, "               present, name of the file that contains\n");
    fprintf(stderr, "               the voice message that informs about\n");
    fprintf(stderr, "               an invalid pin, but with the extension\n");
    fprintf(stderr, "               .msg\n");
    fprintf(stderr, "The .txt files must contain one (MSN or pin) number\n");
    fprintf(stderr, "per line, without any non-numeric characters.\n");
    fprintf(stderr, "The .msg files need to contain law data.\n");
    return 1;
  }
  while(!interrupted)
  {
    connection *makeConnection=new connection(argv[1], argv[2]);
    if(argc==4)
    {
      makeConnection->establishCall(argv[3]);
    }
    else
    {
      makeConnection->establishCall(argv[3], argv[4], argv[5]);
    }
    delete makeConnection;
  }
  return 0;
}

