#! /bin/sh
#
# mkcd.bat 
# bootCD v.0.4 package for fli4l 2.0.4
# author: Art Pogoda <a.pogoda@web.de>

#------------------------------------------------------------
# print usage information to stderr and exit
#------------------------------------------------------------
function usage() {
	    echo "
usage: $0 -h | [-g] [-l] [-s] [config-directory]
 
    -h print this lot out
    -l use large 32MB root filesystem as default

" >&2
	    exit 1
}

#------------------------------------------------------------
# message if something went wrong on the floppy disk
#------------------------------------------------------------
function fl_failure() {
		echo "failed writing system files, check floppy fisk"
		echo "maybe it's full, damaged, write protected ?"
		echo "if it's full, you can try a configuration with less options..."
		exit 1
}

#------------------------------------------------------------
# message if something went wrong on the harddisk
#------------------------------------------------------------
function hd_failure() {
	echo "something went wrong creating disk image file"
	echo "maybe hard disk is full or some files are missing ?"
	exit 1
}


#------------------------------------------------------------
# parse command line options
#------------------------------------------------------------
isolinux=isolinux.cfg                 # default cfg

while [ 1 ]
do
    case "$1"
    in
        -l)                                     # use 32MB rootfs
            large=1
            shift
            ;;
	-?)
            usage
	    ;;
	*)
	    break
	    ;;
    esac
done

if [ "$1" != "" ]
then
    echo "using configuration directory $1 ..."
    config_dir=$1
else
    config_dir=config
fi

set -e	# exit on error

cd unix
make mkfli4l
cd ..

unix/mkfli4l $config_dir

echo "creating compressed tar archive..."
cd opt
rm -f ../img/opt.tgz ../img/opt.tar
tar -c -v -T opt.tmp -f ../img/opt.tar
gzip -9 ../img/opt.tar
mv ../img/opt.tar.gz ../img/opt.tgz
rm -f opt.tmp

cp ./etc/rc.cfg ../img/rc.cfg

cd ../img
mkdir -p cd/isolinux

echo "copying isolinux files ..."
cp -v isolinux.bin cd/isolinux
cp -v isolinux.msg cd/isolinux
if [ "$large" = "1" ]
then
	cp -v isolinux32.cfg cd/isolinux/isolinux.cfg
else
	cp -v isolinux.cfg   cd/isolinux/isolinux.cfg
fi

echo "copying rc.cfg ..."
cp -v rc.cfg cd

echo "copying opt.tgz ..."
cp -v opt.tgz cd

echo "copying kernel ..."
cp -v kernel cd/isolinux

echo "copying root fs images ..."
cp -v rootcd.gz   cd/isolinux/rootfs.gz
cp -v rootcd32.gz cd/isolinux/rootfs32.gz

cd cd

set +e  # don't exit on error any more, gonna catch'em ourselves

#------------------------------------------------------------
# create the iso image
#------------------------------------------------------------
mkisofs -pad -A "fli4l" -V "fli4l" -o ../fli4l.iso -J -r -b isolinux/isolinux.bin -c isolinux/boot.cat -no-emul-boot -boot-load-size 4 -boot-info-table .
if [ $? != 0 ]
then
	echo -e "\n\ncreation of iso image failed ... "	
	exit 1
fi

cd ..

#------------------------------------------------------------
# ask user if cd should be written using cdrecord
#------------------------------------------------------------
echo -e "\n\ncreation of iso image seems to have been successfull:\n"
ls -l fli4l.iso
echo -en "\ndo you now want to write this image to a CD medium now ? (y/N): " 
read yn
if [ "$yn" != "y" -a "$yn" != "Y" ]
then
	echo -e "finished...\n"
	exit 0
fi

#------------------------------------------------------------
# display list of installed cd writers / cd roms and ask for choice
#------------------------------------------------------------
clear
echo
cdrecord -scanbus | grep ' *[0-9],[0-9],[0-9] *'
if [ $? != 0 ]
then
	echo "there was a problem using the program CDRECORD"
	echo "either it's not installed on your system or you have"
	echo "not sufficient privileges to use it ..."
	exit 1
fi
echo -e  "\n\nEnter the three letter ID of the drive to use, e.g. 0,0,0"
echo -n "(0,0,0): > "
read CDID
[ -z "$CDID" ] && CDID="0,0,0"

#------------------------------------------------------------
# open the drive tray and tell user to insert a CD
#------------------------------------------------------------
cdrecord dev=$CDID -eject > /dev/null
if [ $? != 0 ]
then
	echo -e "\nthere was a problem accessing drive $CDID, make"
	echo -e "sure the ID is correct, aborting ...\n"
	exit 1
fi

echo -en "\n\ninsert a blank CD medium anmd press RETURN key\nto start write simulation ..."
read dummy
clear

#------------------------------------------------------------
# CD write simulation
#------------------------------------------------------------
cdrecord dev=$CDID speed=1 -dummy -v -data fli4l.iso > /dev/null
if [ $? != 0 ]
then
	echo -e "\nwrite simulation failed, aborting ...\n"
	exit 1
fi

#------------------------------------------------------------
# ask for final confirmation
#------------------------------------------------------------
echo -en "\n\nwrite simulation was successful,\ndo you now REALLY want to\nburn the image to CD ? (y/N) "
read yn
if [ "$yn" != "y" -a "$yn" != "Y" ]
then
	echo -e "aborted ...\n"
	exit 0
fi

#------------------------------------------------------------
# actual CD writing
#------------------------------------------------------------
cdrecord dev=$CDID speed=1 -v -data -eject fli4l.iso
if [ $? = 0 ]
then
	echo -e "\nsucessfully written CD, try to boot your fli4l box with it ...\n\n"
else
	echo -e "\nwriting CD failed ...\n\n"
fi

