#! /bin/sh
#
# mkcd.bat 
# bootCD v.0.2 package for fli4l 2.0.3
# author: Art Pogoda <a.pogoda@web.de>

#------------------------------------------------------------
# print usage information to stderr and exit
#------------------------------------------------------------
function usage() {
	    echo "
usage: $0 -h | [-g] [-l] [-s] [config-directory]
 
    -h print this lot out
    -g force generation of a boot image floppy even
       an image from a former run exists
    -l use large 32MB root filesystem instead of default
    -s use slow and stupid but secure mode with syslinix

" >&2
	    exit 1
}

#------------------------------------------------------------
# message if something went wrong on the floppy disk
#------------------------------------------------------------
function fl_failure() {
		echo "failed writing system files, check floppy fisk"
		echo "maybe it's full, damaged, write protected ?"
		echo "if it's full, you can try a configuration with less options..."
		exit 1
}

#------------------------------------------------------------
# message if something went wrong on the harddisk
#------------------------------------------------------------
function hd_failure() {
	echo "something went wrong creating disk image file"
	echo "maybe hard disk is full or some files are missing ?"
	exit 1
}


#------------------------------------------------------------
# parse command line options
#------------------------------------------------------------
suffix=''				# fast but not safe
size="1440"                             # only this works with cd's
syslinux=syslinuxcd.cfg                 # default cfg
fd0=/dev/fd0                            # default device: generic fd0

while [ 1 ]
do
    case "$1"
    in
	-g)
	    generate=1
	    shift
	    ;;
        -s)                                     # slow stupid safe
            suffix='-s'
            shift
            ;;
        -l)                                     # use 32MB rootfs
            large=1
            shift
            ;;
	-?)
            usage
	    ;;
	*)
	    break
	    ;;
    esac
done

if [ "$1" != "" ]
then
    echo "using configuration directory $1 ..."
    config_dir=$1
else
    config_dir=config
fi

set -e	# exit on error

cd unix
make mkfli4l
cd ..

unix/mkfli4l $config_dir

echo "creating compressed tar archive..."
cd opt
rm -f ../img/opt.tgz ../img/opt.tar
tar -c -v -T opt.tmp -f ../img/opt.tar
gzip -9 ../img/opt.tar
mv ../img/opt.tar.gz ../img/opt.tgz
rm -f opt.tmp

cp ./etc/rc.cfg ../img/rc.cfg

cd ../img

#------------------------------------------------------------
# check if floppy image exists and force generation otherwise
#------------------------------------------------------------
if [ "$generate" != "1" ]; then
	if [ ! -f cd$suffix/boot.img ]; then
		echo -e "\n\n------------------------------------------------------"
		echo "img/cd$suffix/boot.img does not exist yet, forcing -g parameter"
		echo "please insert a formatted 1.44 MB floppy ..."
		echo
		echo "Press enter to continue ..."
		read dummy
		generate=1
	elif [ `id -g` != 0 ]
	then
		echo -e "\n\n------------------------------------------------------"
		echo "since you are not root (thus not allowed to mount a loopback"
		echo "device), please insert a formatted 1.44 MB floppy ..."
		echo
		echo "Press enter to continue ..."
		read dummy
		generate=1
	else
		TMP=`mktemp -d /tmp/mkcd_XXXXXX`
		mkdir -p $TMP || hd_failure
		mount -o loop cd$suffix/boot.img $TMP
		if [ $? != 0 ]
		then
			echo "failed mounting cd$suffix/boot.img as loopback device"
			echo "make sure, your're kernel supports this and you're root"
			rm -r $TMP
			exit 1
		fi
		rm -f $TMP/syslinux.cfg $TMP/kernel $TMP/rootfs.gz || hd_failure

		echo "copying syslinux.cfg ..."
		cp -v $syslinux $TMP/syslinux.cfg || hd_failure

		echo "copying kernel ..."
		cp -v kernel $TMP || hd_failure

		echo "copying rootfs.gz ..."
		if [ "$large" = "1" ]
		then
			cp -v rootfscd32.gz $TMP/rootfs.gz || hd_failure
		else
			cp -v rootfscd.gz   $TMP/rootfs.gz || hd_failure
		fi
		
		umount $TMP || hd_failure
		rmdir $TMP
	fi
fi

#------------------------------------------------------------
# generate floppy image if -g parameter seen or if missing
#------------------------------------------------------------
if [ "$generate" = "1" ]; then

	fat_image_file=fd$size$suffix.gz

	echo "writing FAT and system files ..."
	gzip -d <$fat_image_file >$fd0 || fl_failure

	echo "copying syslinux.cfg ..."
	mcopy -o $syslinux a:SYSLINUX.CFG || failue

	echo "copying kernel ..."
	mcopy -o kernel a:KERNEL || fl_failure

	echo "copying rootfs.gz ..."
	if [ "$large" = "1" ]
	then
		mcopy -o rootfscd32.gz a:ROOTFS.GZ || fl_failure
	else
		mcopy -o rootfscd.gz   a:ROOTFS.GZ || fl_failure
	fi

	echo "creating boot floppy image ..."
	mkdir -p cd$suffix
	cat $fd0 > cd$suffix/boot.img || fl_failure
fi

echo "copying rc.cfg ..."
cp -v rc.cfg cd$suffix

echo "copying opt.tgz ..."
cp -v opt.tgz cd$suffix

#------------------------------------------------------------
# if the boot device should be mounted, we copy also
# the rootfs and the kernel to the root of the cd, 
# because thats what will be visible when mounted but not
# the files inside the floppy boot image 
#------------------------------------------------------------
MOUNT_BOOT=`grep "^MOUNT_BOOT" rc.cfg | sed -e 's/.*=//' -e "s/'//g"`
case $MOUNT_BOOT
in
	yes|ro|rw)
		echo "WARNING: copying also kernel and rootfs to CD root"
		cp kernel      cd$suffix/kernel    || hd_failure
		if [ "$large" = "1" ]
		then
			cp rootfscd32.gz cd$suffix/rootfs.gz || hd_failure
		else
			cp rootfscd.gz   cd$suffix/rootfs.gz || hd_failure
		fi
	;;

	*)
		echo "WARNING: MOUNT_BOOT='$MOUNT_BOOT' was set, CD will hide kernel and rootfs inside boot image"
		rm -f cd$suffix/kernel cd$suffix/rootfs.gz
	;;
esac

cd cd$suffix

set +e  # don't exit on error any more, gonna catch'em ourselves

#------------------------------------------------------------
# create the iso image
#------------------------------------------------------------
mkisofs -pad -A "CentriumFirewall" -V "CentriumFirewall" -o ../fli4l$suffix.iso -J -r -b boot.img -c boot.catalog .
if [ $? != 0 ]
then
	echo -e "\n\ncreation of iso image failed ... "	
	exit 1
fi

cd ..

#------------------------------------------------------------
# ask user if cd should be written using cdrecord
#------------------------------------------------------------
echo -e "\n\ncreation of iso image seems to have been successfull:\n"
ls -l fli4l.iso
echo -en "\ndo you now want to write this image to a CD medium now ? (y/N): " 
read yn
if [ "$yn" != "y" -a "$yn" != "Y" ]
then
	echo -e "finished...\n"
	exit 0
fi

#------------------------------------------------------------
# display list of installed cd writers / cd roms and ask for choice
#------------------------------------------------------------
clear
echo
cdrecord -scanbus | grep ' *[0-9],[0-9],[0-9] *'
if [ $? != 0 ]
then
	echo "there was a problem using the program CDRECORD"
	echo "either it's not installed on your system or you have"
	echo "not sufficient privileges to use it ..."
	exit 1
fi
echo -e  "\n\nEnter the three letter ID of the drive to use, e.g. 0,0,0"
echo -n "(0,0,0): > "
read CDID
[ -z "$CDID" ] && CDID="0,0,0"

#------------------------------------------------------------
# open the drive tray and tell user to insert a CD
#------------------------------------------------------------
cdrecord dev=$CDID -eject > /dev/null
if [ $? != 0 ]
then
	echo -e "\nthere was a problem accessing drive $CDID, make"
	echo -e "sure the ID is correct, aborting ...\n"
	exit 1
fi

echo -en "\n\ninsert a blank CD medium anmd press RETURN key\nto start write simulation ..."
read dummy
clear

#------------------------------------------------------------
# CD write simulation
#------------------------------------------------------------
cdrecord dev=$CDID speed=1 -dummy -v -data fli4l.iso > /dev/null
if [ $? != 0 ]
then
	echo -e "\nwrite simulation failed, aborting ...\n"
	exit 1
fi

#------------------------------------------------------------
# ask for final confirmation
#------------------------------------------------------------
echo -en "\n\nwrite simulation was successful,\ndo you now REALLY want to\nburn the image to CD ? (y/N) "
read yn
if [ "$yn" != "y" -a "$yn" != "Y" ]
then
	echo -e "aborted ...\n"
	exit 0
fi

#------------------------------------------------------------
# actual CD writing
#------------------------------------------------------------
cdrecord dev=$CDID speed=1 -v -data -eject fli4l.iso
if [ $? = 0 ]
then
	echo -e "\nsucessfully written CD, try to boot your firewall with it ...\n\n"
else
	echo -e "\nwriting CD failed ...\n\n"
fi

