/*
 *      Ersatz für timelog.sh
 *      Bietet unglaubliche Geschwindigkeitsvorteile, echt kaum zu glauben.  
 *
 *      Einschränkungen: Circuit-Name darf nicht mehr als 15 Zeichen(Einschränkung von Fli4l)
 *      enthalten, und die Anzahl der Circuits darf nicht größer als 10 sein
 *
 *  
 *      Copyright 2005 Enrico Troeger <fli4l@emgame.de>
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *      (AZR = Abrechnungszeitraum)
 *      ToDo: - imond.log wird auch bei der Hilfe eingelesen, unnötig.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <libgen.h>		// für basename()

// alle für die Socket-Operationen
#include <unistd.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <sys/socket.h>



#define TRUE 1
#define FALSE 0

#define VERSION				"0.8"
#define VERSION_C			"0.1"
#define CONFFILE			"/etc/timelog.conf"
#define MAX_CONN_PER_DAY 	75
#define SCRIPTNAME			"main_Timelog.cgi"
#define FARBE1				"#FFFFFF"
#define FARBE2				"#E7E7E7"
#define GRAPH_HEIGHT		350



unsigned short PRINT_SHORT;			// Flag, ob Ausgabe ohne führende Null erfolgen soll
unsigned short PRINT_ALL = 1;		// Ausgabe-Modus für den Kommandozeilen-Modus (alles ausgeben)
unsigned short PRINT_TIME = 2;		// Ausgabe-Modus für den Kommandozeilen-Modus (OnlineZeit ausgeben)
unsigned short PRINT_TRAFFIC = 3;	// Ausgabe-Modus für den Kommandozeilen-Modus (Traffic ausgeben)
unsigned short farb_count;			// Variable für die abwechselnde Tabellenhintergrundfarbe
unsigned int conns_len;				// Anzahl der Verbindungen in der imond.log
struct connection *conns;			// Array der Verbindungen
char *me;							// Zeiger auf argv[0] (Programmname)

// Heute
static time_t seconds;
static struct tm *tm;
static unsigned int this_year;
static unsigned short this_month;
static unsigned short today;


// eine Verbindung
struct connection {
    char cir_name[15];
	time_t start;
	time_t end;
	unsigned int duration;
	float tin;
	float tout;
	char device[10];
};


// die Konfiguration
struct config {
	char logfile[50];
	char lcd_day_file[50];
	char lcd_mon_file[50];
    char circuits[11][15];
    unsigned short circuits_anz;
	unsigned short default_circuit;
    unsigned short start_day;
} conf;


// Monatsnamen
char monatsnamen[12][10] = {
	"Januar",
	"Februar",
	"Maerz",
	"April",
	"Mai",
	"Juni",
	"Juli",
	"August",
	"September",
	"Oktober",
	"November",
	"Dezember",
};


// Prototypen

// Format
time_t get_time(int year, int month, int day);
struct tm get_time_as_tm(int year, int month, int day);
char *nice_tl_date(char *raw_date);
void format_tl_date(char *str, int *day, int *month, int *year);
int format_imond_time(char *str);
char *format_seconds(int s);
float format_traffic(float tr);
// Init
int read_conf_file(char* filename);
int read_file(void);
int file_exists(char *filename);
int do_checks();
// HTML
void data_summery(char *circuit, int last_mon, int yesterday);
void data_year(char *circuit, int year);
void data_month(char *circuit, int year, int month);
void data_day(char *circuit, int year, int month, int day);
void graph_year(char *circuit, int year);
void graph_month(char *circuit, int year, int month);
void graph_day(char *circuit, int year, int month, int day);
// CmdLine
void cmdl_date(int argc, char **argv, unsigned short mode);
void cmdl_gen(int argc, char **argv, unsigned short mode);
void cmdl_A(int argc, char **argv);
// Counting
void count_year(char *circuit, int year, int *dur, float *tin, float *tout);
void count_month(char *circuit, int year, int month, int *dur, float *tin, float *tout);
void count_day(char *circuit, int year, int month, int day, int *dur, float *tin, float *tout);
int get_min_year();
int get_monthmax(int month);
int get_prev_month(int this_month);
// Help
void show_header(char *me);
void show_usage(char *me);
void show_help(char *me);
void lcd_mode(void);
// Imond-Kommunikation
int service_connect (char *host_name, int port);
void send_command (int fd, char *str);
char *get_answer (int fd);
int get_time_from_imond(char* circuit);
#ifdef TEST		// Debug
void print_conns(void);
void print_conn(int i);
#endif


// Implementierungen


// die folgenden 3 Optionen stammen aus dem Code von GTK-Imonc
/* service_connect (host_name, port)	    - connect to tcp-service */
int service_connect (char *host_name, int port)
{
	struct sockaddr_in addr;
	struct hostent *host_p;
	int fd;
	int opt = 1;

	(void)memset((char *) &addr, 0, sizeof (addr));

	if ((addr.sin_addr.s_addr = inet_addr((char *) host_name)) == INADDR_NONE)
	{
		host_p = gethostbyname (host_name);
		(void) memcpy((char *)(&addr.sin_addr), host_p->h_addr, (size_t)host_p->h_length);
	}

	addr.sin_family  = AF_INET;
	addr.sin_port    = htons((unsigned short) port);

	if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
		return (-1);

	(void) setsockopt (fd, IPPROTO_TCP, TCP_NODELAY, (char *) &opt, sizeof (opt));

	if (connect (fd, (struct sockaddr *) &addr, sizeof (addr)) != 0)
	{
		(void) close (fd);
		return (-1);
	}

	return (fd);
}


void send_command (int fd, char *str)
{
	char buf[256];
	int len = strlen (str);

	snprintf(buf, 256, "%s\r\n", str);
	write (fd, buf, len + 2);
}


/* get_answer (fd) */
char *get_answer (int fd)
{
	static char buf[128];
	int len;

	len = read (fd, buf, 128);

	if (len <= 0)
		return ((char *) NULL);

	while (len > 1 && (buf[len - 1] == '\n' || buf[len - 1] == '\r'))
	{
		buf[len - 1] = '\0';
		len--;
	}

	if (! strncmp (buf, "OK ", 3))			/* OK xxxx	    */
		return (buf + 3);
	else if (len > 2 && ! strcmp (buf + len - 2, "OK"))
	{
		*(buf + len - 2) = '\0';
		return (buf);
	}
	else if (len == 2 && ! strcmp (buf + len - 2, "OK"))
		return (buf);

    return ((char *) NULL);				/* ERR xxxx	    */
}


void data_summery(char *circuit, int last_mon, int yesterday)
{
	time_t tings;
	char end_day[6];
	int month, last_mon_yesterday;	// um Monatswechsel zu merken
	int i, j, dur = 0, t_width, t_height;
	float tin = 0, tout = 0;
	
	// keine Ahnung ob das zuverlässig läuft
	if (today < conf.start_day)
	{
		month = get_prev_month(this_month);
		last_mon = get_prev_month(get_prev_month(this_month));
	}
	else
	{
		month = this_month;
		last_mon = get_prev_month(this_month);
	}
	
	// letzten Monat berechnen
	count_month(circuit, this_year, last_mon, &dur, &tin, &tout);
	printf("<table border=\"0\">");
    printf("<tr><td colspan=\"2\" style=\"font-size: 11pt\">Aktuelle Verbindungszeiten:</td><td width=\"25\">");
	printf("</td><td colspan=\"3\" align=\"center\" style=\"font-size: 11pt\">Traffic:</td></tr>");
    printf("<tr><td colspan=\"6\">&nbsp;</td></tr>");
    printf("<tr><td align=\"right\"><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">Letzter Monat</a>&nbsp;", 
			SCRIPTNAME, circuit, this_year, last_mon);
	printf("(<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=0\">Grafik</a>):</td><td>%s h</td>\n", 
			SCRIPTNAME, circuit, this_year, last_mon, format_seconds(dur));
	printf("<td width=\"25\"></td><td>\n");
    printf("<td>IN: %05.2f MB</td><td>OUT: %05.2f MB</td></tr>\n", format_traffic(tin), format_traffic(tout));

   	// diesen Monat berechnen
	count_month(circuit, this_year, month, &dur, &tin, &tout);
	printf("<tr><td align=\"right\"><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">Aktueller Monat</a>&nbsp;\n",
			SCRIPTNAME, circuit, this_year, month);
	printf("(<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=0\">Grafik</a>):</td><td><b>%s h</b></td>\n",
			SCRIPTNAME, circuit, this_year, month, format_seconds(dur));
	printf("<td width=\"25\"></td><td></td>");
    printf("<td>IN: %05.2f MB</td><td>OUT: %05.2f MB</td></tr>", format_traffic(tin), format_traffic(tout));
    
	// Gestern berechnen
	if (today == 1) last_mon_yesterday = last_mon;
	else last_mon_yesterday = this_month;
	count_day(circuit, this_year, last_mon_yesterday, yesterday, &dur, &tin, &tout);
	printf("<tr><td align=\"right\"><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=%02d\">Gestern</a>&nbsp;",
			SCRIPTNAME, circuit, this_year, last_mon_yesterday, yesterday);
	printf("(<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=%02d\">Grafik</a>):</td><td>%s h</td>\n",
			SCRIPTNAME, circuit, this_year, last_mon_yesterday, yesterday, format_seconds(dur));
	printf("<td width=\"25\"></td><td></td>");
    printf("<td>IN: %05.2f MB</td><td>OUT: %05.2f MB</td></tr>", format_traffic(tin), format_traffic(tout));
    
	// Heute berechnen
	count_day(circuit, this_year, this_month, today, &dur, &tin, &tout);
	printf("<tr><td align=\"right\"><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=%02d\">Heute</a>&nbsp;\n",
			SCRIPTNAME, circuit, this_year, this_month, today);
	printf("(<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=%02d\">Grafik</a>):</td>\n",
			SCRIPTNAME, circuit, this_year, this_month, today);
	printf("<td><b>%s h</b></td><td width=\"25\"></td><td></td>\n",format_seconds(dur));
	printf("<td>IN: %05.2f MB</td><td>OUT: %05.2f MB</td></tr>\n", format_traffic(tin), format_traffic(tout));
	printf("</table>\n");


#ifdef TEST
	printf("<!-- Lines: %d, Month: %d, last_mon: %d, last_mon_yesterday: %d, yesterday: %d  # just debug   ;-)  -->\n",
			conns_len, month, last_mon, last_mon_yesterday, yesterday);
#endif

	// Übersicht
	t_width = 50;
	t_height = 30;
	printf("<table border=\"0\">\n");
	printf("<tr><td colspan=\"14\">&nbsp;</td></tr>\n");
	printf("<tr><td colspan=\"14\" style=\"font-size: 11pt\">Monatsdaten(D) oder Monatsgrafiken(G) anzeigen:</td></tr>\n");
	printf("<tr><td colspan=\"14\">&nbsp;</td></tr>\n");
	printf("<tr><td align=\"right\" width=\"%d\" height=\"%d\">&nbsp;</td>\n", t_width, t_height);
	for(i = 1; i <= 12; i++)
	{
		tings = get_time(this_year, (i == 12) ? 1 : (i + 1), conf.start_day - 1);
		strftime(end_day, 6, "%d.%m", localtime(&tings));
		printf("<td valign=\"top\" width=\"%d\" height=\"%d\">%02d.%02d-<br>%s</td>\n",
			t_width, t_height, conf.start_day, i, end_day);

	}
	printf("<td valign=\"top\" width=\"%d\" height=\"%d\">Gesamt</td></tr>\n", t_width, t_height);

	for(i = get_min_year(); i <= this_year; i++)
	{
		printf("<tr><td align=\"right\">%d:&nbsp;</td>\n", i);
		for(j = 1; j <= 12; j++)
		{
			if (seconds < get_time(i, j, conf.start_day))
				printf("<td></td>\n");
			else
				printf("<td><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">D</a>&nbsp;\n<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=0\">G</a></td>\n",
						SCRIPTNAME, circuit, i, j, SCRIPTNAME, circuit, i, j);
		}
		printf("<td align=\"center\"><a href=\"%s?circuit=%s&jahr=%d&monat=0&tag=0\">D</a>&nbsp;<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=0&tag=0\">G</a></td></tr>\n",
						SCRIPTNAME, circuit, i, SCRIPTNAME, circuit, i);
	}

}


void data_year(char *circuit, int year)
{
	int i, max_monat, dur = 0;
	float tin = 0, tout = 0;
	
	farb_count = 1;
	// im aktuellen Jahr nur bis zum aktuellen Monat berechnen, weil vom nähesten Monat gibt es ja noch keine Daten ;-)
	if (this_year == year) max_monat=this_month;
	else max_monat=12;

	printf("<table border=\"0\">\n");
	printf("<tr><td colspan=\"2\">Jahresansicht f&uuml;r <b>%d</b> (f&uuml;r Circuit: %s):&nbsp;&nbsp;&nbsp;</td>\n", year, circuit);
	printf("<td align=\"right\">Traffic IN:&nbsp;&nbsp;</td><td width=\"15\"></td><td align=\"right\">Traffic OUT:</td></tr>\n");

	for(i = 1; i <= max_monat; i++)
	{
		count_month(circuit, year, i, &dur, &tin, &tout);
		printf("<tr bgcolor=\"%s\"><td><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">",
				(farb_count % 2 != 0) ? FARBE1 : FARBE2, SCRIPTNAME, circuit, year, i);
		printf("%s:</a></td><td>%s</td></td><td align=\"right\">%05.2f MB</td><td></td>",
				monatsnamen[i - 1], format_seconds(dur), format_traffic(tin));
		printf("<td align=\"right\">%05.2f MB</td></tr>\n", format_traffic(tout));
		// Tabellen-Hintergrundfarbe festlegen
		farb_count++;
	}
	count_year(circuit, year, &dur, &tin, &tout);
	printf("<tr bgcolor=\"%s\"><td>Gesamt:</td><td>%s</td></td><td align=\"right\">%05.2f MB</td><td>",
			(farb_count % 2 != 0) ? FARBE1 : FARBE2, format_seconds(dur), format_traffic(tin));
	printf("</td><td align=\"right\">%05.2f MB</td></tr>\n", format_traffic(tout));
	printf("</table>\n<br><br><a href=\"%s?circuit=%s&jahr=0&monat=0&tag=0\">Zur&uuml;ck</a>&nbsp;&nbsp;&nbsp;", SCRIPTNAME, circuit);
	printf("<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=0&tag=0\">Grafik</a>\n", SCRIPTNAME, circuit, year);

}


void data_month(char *circuit, int year, int month)
{
	time_t azr_start, azr_end;
	int i, max_tage, dur = 0, end_day, sd_year, sd_month, sd_day;	// sd_* Variablen für START_DAY
	float tin = 0, tout = 0;
	
	farb_count = 1;
	max_tage = get_monthmax(month);
	
	// im aktuellen Monat nur bis zum heutigen Tag berechnen, weil von morgen gibt es ja noch keine Daten ;-) 
	// keine Ahnung ob das zuverlässig läuft
	azr_start = get_time(year, month, conf.start_day);
	azr_end = get_time(year, month +1, conf.start_day - 1);
	//printf("%d<br>%d<br>%d\n<br>", (int)seconds, (int)azr_start, (int)azr_end);
	if (seconds > azr_start && seconds < azr_end)
	{
		if (today < conf.start_day)
			end_day = conf.start_day + max_tage - (conf.start_day - today);
		else
			end_day = today;	
	}
	else
		end_day = conf.start_day + max_tage - 1;

	//printf("end_day: %d, max_tage: %d, today: %d, <br>", end_day, max_tage, today);
	
	printf("<table border=\"0\">\n");
	printf("<tr><td colspan=\"2\">Monatsanzeige f&uuml;r %s %d (f&uuml;r Circuit: %s):&nbsp;&nbsp;&nbsp;</td>\n",
			monatsnamen[month - 1], year, circuit);
	printf("<td align=\"right\">Traffic IN:&nbsp;&nbsp;</td><td width=\"15\"></td><td align=\"right\">Traffic OUT:</td></tr>\n");

	for (i = conf.start_day; i <= end_day; i++)
	{
		if (i > max_tage)
		{
			sd_day = i - max_tage;
			if (month == 12)
			{
				sd_month = 1;
				sd_year = year + 1;
			}
			else
			{
				sd_month = month + 1;
				sd_year = year;
			}
		}
		else
		{
			sd_day =  i;
			sd_month = month;
			sd_year = year;
		}
		count_day(circuit, sd_year, sd_month, sd_day, &dur, &tin, &tout);
		printf("<tr bgcolor=\"%s\"><td><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=%02d\">\n",
				(farb_count % 2 != 0) ? FARBE1 : FARBE2, SCRIPTNAME, circuit, sd_year, sd_month, sd_day);
		printf("%02d.%02d.%d</a>:</td><td>%s</td></td><td align=\"right\">%05.2f MB</td><td></td>\n",
				sd_day, sd_month, sd_year, format_seconds(dur), format_traffic(tin));
		printf("<td align=\"right\">%05.2f MB</td></tr>\n", format_traffic(tout));
		farb_count++;
	}
	count_month(circuit, year, month, &dur, &tin, &tout);
	printf("<tr bgcolor=\"%s\"><td>Gesamt:</td><td>%s</td></td><td align=\"right\">%05.2f MB</td><td>\n",
			(farb_count % 2 != 0) ? FARBE1 : FARBE2, format_seconds(dur), format_traffic(tin));
	printf("</td><td align=\"right\">%05.2f MB</td></tr>\n", format_traffic(tout));
	printf("</table><br><br><a href=\"%s?circuit=%s&jahr=0&monat=0&tag=0\">Zur&uuml;ck zur &Uuml;bersicht</a>&nbsp;&nbsp;&nbsp;", SCRIPTNAME, circuit);
	printf("<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=0\">Grafik</a>\n", SCRIPTNAME, circuit, year, month);
}


void data_day(char *circuit, int year, int month, int day)
{
	int i, dur = 0, sd_month, sd_year;
	float tin = 0, tout = 0;
	char tmp[11];
	
	farb_count = 1;

	printf("Tagesanzeige f&uuml;r %02d.%02d.%d (f&uuml;r Circuit: %s):<br>\n",
			day, month, year, circuit);
	printf("<table border=\"0\">\n");
	printf("<tr><td width=\"65\">Circuit:</td><td width=\"75\">Start-Zeit:</td><td width=\"75\">End-Zeit:</td><td>Online-Zeit:</td><td width=\"15\"></td>\n");
	printf("<td width=\"75\" align=\"center\">Traffic IN:&nbsp;&nbsp;</td><td width=\"15\"></td><td width=\"75\" align=\"center\">Traffic OUT:</td></tr>\n");
    
	time_t day_start = get_time(year, month, day);
	time_t day_end = get_time(year, month, day + 1);

	for(i = 0; i < conns_len; i++)
	{
		if (! strcmp(conns[i].cir_name, circuit)
			|| ! strcmp("all", circuit) || ! strcmp("Alle", circuit))
		{
			if (conns[i].start >= day_start && conns[i].start < day_end)
			{
				strftime(tmp, 11, "%H:%M:%S ", localtime(&conns[i].start));
				printf("<tr bgcolor=\"%s\"><td align=\"left\">%s</td><td align=\"center\">%s</td>\n",
						(farb_count % 2 != 0) ? FARBE1 : FARBE2, conns[i].cir_name, tmp);
				
				strftime(tmp, 11, "%H:%M:%S ", localtime(&conns[i].end));
				printf("<td align=\"center\">%s</td><td align=\"center\">%s</td><td></td><td align=\"center\">%05.2f MB</td><td></td>\n<td align=\"center\">%05.2f MB</td></tr>\n",
						tmp, format_seconds(conns[i].duration), format_traffic(conns[i].tin), format_traffic(conns[i].tout));
				
				farb_count++;
			}
		}
	}
	
	count_day(circuit, year, month, day, &dur, &tin, &tout);
	printf("<tr bgcolor=\"%s\"><td align=\"left\" colspan=\"3\">Gesamt:</td><td align=\"center\">%s</td><td>\n",
			(farb_count % 2 != 0) ? FARBE1 : FARBE2, format_seconds(dur));
	printf("</td><td align=\"center\">%05.2f MB</td><td></td><td align=\"center\">%05.2f MB</td></tr>\n",
			format_traffic(tin), format_traffic(tout));
	// Monat verringern, um im richtigen AZR zu bleiben
	sd_month = month;
	sd_year = year;
	if (day < conf.start_day)
	{
		sd_year = (month == 1) ? (year - 1) : year;
		sd_month = (month == 1) ? 12 : (month - 1);
	}
	printf("</table><br><br><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">Zur&uuml;ck</a>&nbsp;&nbsp;&nbsp;",
			SCRIPTNAME, circuit, sd_year, sd_month);
	printf("<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=%02d\">Grafik</a>\n",
			SCRIPTNAME, circuit, year, month, day);

}


void graph_year(char *circuit, int year)
{
    int i, max_monat, max_dur = 0, sum = 0,  dur = 0, tmp;
	float months_dur[12][3] = { { 0 } };	// Array zum Erfassen der Online-Zeit der Verbindungen im Jahr
	float tin = 0, tout = 0;
	
	// im aktuellen Jahr nur bis zum aktuellen Monat berechnen, weil vom nähesten Monat gibt es ja noch keine Daten ;-)
	if (this_year == year) max_monat=this_month;
	else max_monat=12;
	
	printf("grafische Jahresansicht f&uuml;r %d (f&uuml;r Circuit: %s)<br><br><table>\n", year, circuit);
	for(i = 1; i <= max_monat; i++)
	{
		count_month(circuit, year, i, &dur, &tin, &tout);
		if (dur > max_dur)
			max_dur = dur;
		months_dur[i-1][0] = dur;
		months_dur[i-1][1] = tin;
		months_dur[i-1][2] = tout;
		sum += dur;
	}
    
	if (max_dur)
	{
		for(i = 1; i <= max_monat; i++)
		{
			printf("<td valign=\"bottom\"><img src=\"timelog_bar.gif\" height=\"");
			tmp = ((int)months_dur[i-1][0] * GRAPH_HEIGHT) / max_dur;
			printf("%d\" width=\"20\" title=\"%s: %s h, In-Traffic: %05.2f MB, Out-Traffic: %05.2f MB\"><br>\n",
					tmp, monatsnamen[i - 1], format_seconds((int)months_dur[i-1][0]), format_traffic(months_dur[i-1][1]), format_traffic(months_dur[i-1][2]));
			if (tmp > 0)
			{
				printf("<a href=\"%s?circuit=%s&graph=1&jahr=%d&monat=%02d&tag=0\">%02d</a>\n",
						SCRIPTNAME, circuit, year, i, i);
			}
			else
			{
				printf("%02d", i);
			}
			printf("</td>");
		}

		//Durchschnitt berechnen & Rest ausgeben
		tmp = (sum / max_monat);
		printf("<td align=\"center\" valign=\"bottom\"><img src=\"timelog_bar.gif\" height=\"%d\" ",
				(tmp * GRAPH_HEIGHT) / max_dur);
		printf("width=\"20\" title=\"Durchschnitt: %s h\" ><br>&#216;</td></tr>\n", format_seconds(tmp));
	}
	else
	{
		printf("<tr><td>Keine Verbindungen vorhanden.</td>");
	}
	printf("</tr></table>\n");
	
	printf("<br><br><a href=\"%s?circuit=%s&jahr=0&monat=0&tag=0\">Zur&uuml;ck</a>&nbsp;&nbsp;&nbsp;\n",
			SCRIPTNAME, circuit);
	printf("<a href=\"%s?circuit=%s&jahr=%d&monat=0&tag=0\">Daten</a>",
			SCRIPTNAME, circuit, year);
}


void graph_month(char *circuit, int year, int month)
{
    time_t azr_start, azr_end;
	int i, x, max_tage, max_dur = 0, sum = 0,  dur = 0, tmp, end_day, sd_year, sd_month, sd_day;
	float days_dur[31][3] = { { 0 }  };	// Array zum Erfassen der Online-Zeit der Verbindungen im Monat
	float tin = 0, tout = 0;
	
	max_tage = get_monthmax(month);
	
	// im aktuellen Monat nur bis zum heutigen Tag berechnen, weil von morgen gibt es ja noch keine Daten ;-) 
	// keine Ahnung ob das zuverlässig läuft
	azr_start = get_time(year, month, conf.start_day);
	azr_end = get_time(year, month +1, conf.start_day - 1);
	//printf("%d<br>%d<br>%d\n<br>", (int)seconds, (int)azr_start, (int)azr_end);
	if (seconds > azr_start && seconds < azr_end)
		if (today < conf.start_day)
			end_day = conf.start_day + max_tage - (conf.start_day - today);
		else
			end_day = today;	
	else
		end_day = conf.start_day + max_tage - 1;

	
	printf("grafische Monatsanzeige f&uuml;r %s %d (f&uuml;r Circuit: %s)<br><br><table>\n",
			monatsnamen[month - 1], year, circuit);

	x = 0;
	for (i = conf.start_day; i <= end_day; i++)
	{
		if (i > max_tage)
		{
			sd_day = i - max_tage;
			if (month == 12)
			{
				sd_month = 1;
				sd_year = year + 1;
			}
			else
			{
				sd_month = month + 1;
				sd_year = year;
			}
		}
		else
		{
			sd_day =  i;
			sd_month = month;
			sd_year = year;
		}

		count_day(circuit, year, month, i, &dur, &tin, &tout);
		if (dur > max_dur)
			max_dur = dur;
		days_dur[x][0] = dur;
		days_dur[x][1] = tin;
		days_dur[x++][2] = tout;
		sum += dur;
	}

	//printf("<tr><!-- max: %d anzahl: %d -->", max_dur, x);
	if (max_dur)
	{
		x = 0;
		for (i = conf.start_day; i <= end_day; i++)
		{
			if (i > max_tage)
			{
				sd_day = i - max_tage;
				if (month == 12)
				{
					sd_month = 1;
					sd_year = year + 1;
				}
				else
				{
					sd_month = month + 1;
					sd_year = year;
				}
			}
			else
			{
				sd_day =  i;
				sd_month = month;
				sd_year = year;
			}
			printf("<td valign=\"bottom\"><img src=\"timelog_bar.gif\" height=\"");
			tmp = ((int)days_dur[x][0] * GRAPH_HEIGHT) / max_dur;
			printf("%d\" width=\"15\" title=\"%02d.%02d.%d: %s h, In-Traffic: %05.2f MB, Out-Traffic: %05.2f MB\"><br>\n",
					tmp, sd_day, sd_month, sd_year, format_seconds((int)days_dur[x][0]), format_traffic(days_dur[x][1]), format_traffic(days_dur[x][2]));
			if (tmp > 0)
			{
				printf("<a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=%02d\">%02d</a>\n",
						SCRIPTNAME, circuit, sd_year, sd_month, sd_day, sd_day);
			}
			else
			{	// Tag ohne Verbindungen
				printf("%02d", sd_day);
			}
			printf("</td>");
			x++;
		}
		
		// Durchschnitt berechnen & Rest ausgeben
		tmp = (sum / max_tage);
		printf("<td align=\"center\" valign=\"bottom\"><img src=\"timelog_bar.gif\" height=\"%d\" ",
				(tmp * GRAPH_HEIGHT) / max_dur);
		printf("width=\"15\" title=\"Durchschnitt: %s h\" ><br>&#216;</td>\n", format_seconds(tmp));
		
		printf("<td align=\"center\" valign=\"bottom\"><img src=\"timelog_bar.gif\" ");
		printf("height=\"%d\" width=\"15\" title=\"Summe: %s h\" ><br>&Sigma;</td></tr>\n", GRAPH_HEIGHT, format_seconds(sum));
	}
	else
	{
		printf("<tr><td>Keine Verbindungen vorhanden.</td>");
	}
	printf("</tr></table>\n");
	printf("<br><br><a href=\"%s?circuit=%s&jahr=0&monat=0&tag=0\">Zur&uuml;ck zur &Uuml;bersicht</a>&nbsp;&nbsp;&nbsp;",
			SCRIPTNAME, circuit);
	printf("<a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">Daten</a>",
			SCRIPTNAME, circuit, year, month);

}


void graph_day(char *circuit, int year, int month, int day)
{
    int i, j = 0, max_dur = 0, sum = 0, sd_month, sd_year, idx;
	char sz[11], ez[11];
	// Array zum Erfassen der zutreffenden Indizes aus dem conns-Array
	int conns_indices[MAX_CONN_PER_DAY];
	
	printf("grafische Tagesanzeige f&uuml;r %02d.%02d.%d (f&uuml;r Circuit: %s)<br><br><table>\n",
			day, month, year, circuit);
	
	time_t day_start = get_time(year, month, day);
	time_t day_end = get_time(year, month, day + 1);

	for(i = 0; i < conns_len; i++)
	{
		if (! strcmp(conns[i].cir_name, circuit) || ! strcmp("Alle", circuit))
		{
			if (conns[i].start >= day_start && conns[i].start < day_end)
			{
				conns_indices[j] = i;
				// Maximum finden
				if (conns[i].duration > max_dur)
					max_dur = conns[i].duration;
				// Summe bilden
				sum += conns[i].duration;
				j++;
			}
		}
	}

	if (max_dur)
	{
		//printf("<tr><!-- max: %d -->", max_dur);
		for(i = 0; i < j; i++)
		{
			idx = conns_indices[i];		// eigentlich unnötig, liest sich als Array-Index aber schöner
			printf("<td align=\"center\" valign=\"bottom\"><img src=\"timelog_bar.gif\" height=\"");
			seconds = (time_t) conns[idx].start;
			strftime(sz, 11, "%H:%M", localtime(&seconds));
			seconds = (time_t) conns[idx].end;
			strftime(ez, 11, "%H:%M", localtime(&seconds));
			if (! strcmp(circuit, "Alle"))
			{
				printf("%d\" width=\"15\" title=\"%s: %s - %s Uhr: %s h, In-Traffic: %05.2f MB, Out-Traffic: %05.2f MB\"><br>%s</td>\n",
					(conns[idx].duration * GRAPH_HEIGHT) / max_dur, conns[idx].cir_name, sz, ez,
					format_seconds(conns[idx].duration), format_traffic(conns[idx].tin), format_traffic(conns[idx].tout), sz);
			}
			else
			{
				printf("%d\" width=\"15\" title=\"%s - %s Uhr: %s h, In-Traffic: %05.2f MB, Out-Traffic: %05.2f MB\"><br>%s</td>\n",
						(conns[idx].duration * GRAPH_HEIGHT) / max_dur, sz, ez,
						format_seconds(conns[idx].duration), format_traffic(conns[idx].tin), format_traffic(conns[idx].tout), sz);
			}
		}
	
		// Durchschnitt & Summe berechnen & Rest ausgeben
		printf("<td align=\"center\" valign=\"bottom\"><img src=\"timelog_bar.gif\" ");
		printf("height=\"%d\" width=\"15\" title=\"Durchschnitt: %s h\" ><br>&#216;</td>\n",
				(sum / j * GRAPH_HEIGHT) / max_dur, format_seconds(sum / j));
		printf("<td align=\"center\" valign=\"bottom\"><img src=\"timelog_bar.gif\" ");
		printf("height=\"%d\" width=\"15\" title=\"Summe: %s h\" ><br>&Sigma;</td>\n", GRAPH_HEIGHT, format_seconds(sum));
	}
	else
	{
		printf("<tr><td>Keine Verbindungen vorhanden.</td>");
	}
	printf("</tr></table>");
	
	// Monat verringern, um im richtigen AZR zu bleiben
	// TODO: gegen den Code aus data_day() tauschen, wenn er sich bewährt hat
	sd_month = month;
	sd_year = year;
	if (day < conf.start_day)
	{
		if (month == 1)
		{	
			sd_month = 12;
			sd_year = year - 1;
		}
		else sd_month--;
	}
	printf("<br><br><a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=0\">Zur&uuml;ck</a>&nbsp;&nbsp;&nbsp;",
			SCRIPTNAME, circuit, sd_year, sd_month);
	printf("<a href=\"%s?circuit=%s&jahr=%d&monat=%02d&tag=%02d\">Daten</a>",
			SCRIPTNAME, circuit, year, month, day);
}


// count_* aka als count() aus den Shellscripten
void count_month(char *circuit, int year, int month, int *dur, float *tin, float *tout)
{
	int i;
	time_t month_start = get_time(year, month, conf.start_day);
	time_t month_end = get_time(year, month + 1, conf.start_day);
	
	*dur = 0;
	*tin = 0.0;
	*tout = 0.0;
	
	for (i = 0; i < conns_len; i++)
	{
		if (! strcmp(conns[i].cir_name, circuit) || ! strcmp("all", circuit) || ! strcmp("Alle", circuit))
		{
			if (conns[i].start >= month_start && conns[i].start < month_end)
			{
				*dur += conns[i].duration;
				*tin += conns[i].tin;
				*tout += conns[i].tout;
			}
		}
	}
	
}


void count_day(char *circuit, int year, int month, int day, int *dur, float *tin, float *tout)
{
	int i;
	time_t day_start = get_time(year, month, day);
	time_t day_end = get_time(year, month, day + 1);
	
	*dur = 0;
	*tin = 0.0;
	*tout = 0.0;
	
	// Sommerzeit-Hack, da muss für Oktober noch was her, sonst gehts wieder in die Hose
	if ((tm->tm_year + 1900) == 2005 &&
		(tm->tm_mon + 1) == 3 &&
		(tm->tm_mday) == 28)
	day_start -= 3600;

	for (i = 0; i < conns_len; i++)
	{
		if (! strcmp(conns[i].cir_name, circuit) || ! strcmp("all", circuit) || ! strcmp("Alle", circuit))
		{
			if (conns[i].start >= day_start && conns[i].start < day_end)
			{
				//printf("%d - %s\n", i, conns[i].cir_name);
				//printf("%s\n", ctime(&conns[i].end));
				*dur += conns[i].duration;
				*tin += conns[i].tin;
				*tout += conns[i].tout;
			}
		}
	}

	// Zeit aus der aktuellen Verbindung zu "heute" addieren
	if ((tm->tm_year + 1900) == year &&
		(tm->tm_mon + 1) == month &&
		(tm->tm_mday) == day)
		*dur += get_time_from_imond(circuit);

}


void count_year(char *circuit, int year, int *dur, float *tin, float *tout)
{
	int i;
	time_t day_start = get_time(year, 1, 1);
	time_t day_end = get_time(year + 1, 1, 1);
	
	*dur = 0;
	*tin = 0.0;
	*tout = 0.0;
	
	for (i = 0; i < conns_len; i++)
	{
		if (! strcmp(conns[i].cir_name, circuit)
			|| ! strcmp("all", circuit) || ! strcmp("Alle", circuit))
		{
			if (conns[i].start >= day_start && conns[i].start < day_end)
			{
				*dur += conns[i].duration;
				*tin += conns[i].tin;
				*tout += conns[i].tout;
			}
		}
	}
	
}


int get_min_year()
{
	int i, min_year = this_year;	// jüngere Verbindungen sollten nicht existieren ;-)
	char tmp_year[5];
	for (i = 0; i < conns_len; i++)
	{
		strftime(tmp_year, 5, "%Y", localtime(&conns[i].end));
		if (atoi(tmp_year) < min_year) min_year = atoi(tmp_year);
	}
	
	return min_year;
}


int get_monthmax(int month)
{
	int result;
	switch (month)
	{
        case 1:
		case 3:
		case 5:
		case 7:
		case 8:
		case 10:
		case 12: result = 31; break;
        case 4:
		case 6:
		case 9:
		case 11: result = 30; break;
		case 2:  result = 28; break;
		default: result = 30;
	}
	return result;
}


int get_prev_month(int month)
{
	return (month == 12) ? 1 : (month - 1);
}


void cmdl_date(int argc, char **argv, unsigned short mode)
{
	int dur = 0, day = 0, month = 0, year, sd_month, sd_year;
	float tin = 0, tout = 0;
	
	// Circuit festlegen, Default-Circuit oder den von der CmdLine
	char *cir = conf.circuits[conf.default_circuit];
	if (argc > 3) cir = argv[3];

	// Datum kopieren für nice_tl_date()
	char date[strlen(argv[2])];
	strcpy(date, argv[2]);
					
	// Datum auslesen
	format_tl_date(argv [2], &day, &month, &year);
					
	// Monat verringern, um im richtigen AZR zu bleiben
	sd_month = month;
	sd_year = year;
	if (day < conf.start_day)
	{
		sd_year = (month == 1) ? (year - 1) : year;
		sd_month = (month == 1) ? 12 : (month - 1);
	}

	if (day > 0 && month > 0)
	{	// Tageswert
		count_day(cir, year, month, day, &dur, &tin, &tout);
	}
	else if (month > 0)
	{	// Monatswert
		count_month(cir, sd_year, sd_month, &dur, &tin, &tout);
	}
	else
	{	// Jahreswert
		count_year(cir, year, &dur, &tin, &tout);
	}
	// Ausgabe
	if (mode == PRINT_ALL || mode == PRINT_TIME)
	{
		printf("Online-Zeit fuer %s: %s h\n", nice_tl_date(date), (char*)format_seconds(dur));
	}
	if (mode == PRINT_ALL || mode == PRINT_TRAFFIC)
	{
		printf("Incoming Traffic: %05.2f MB\n",	format_traffic(tin));
		printf("Outgoing Traffic: %05.2f MB\n", format_traffic(tout));
	}
}


void cmdl_gen(int argc, char **argv, unsigned short mode)
{
	int dur = 0, sd_month, sd_year;
	float tin = 0, tout = 0;
	
	// Monat verringern, um im richtigen AZR zu bleiben
	sd_month = this_month;
	sd_year = this_year;
	if (today < conf.start_day)
	{
		sd_year = (this_month == 1) ? (this_year - 1) : this_year;
		sd_month = (this_month == 1) ? 12 : (this_month - 1);
	}
					
	// Default-Circuit festlegen
	char *cir = conf.circuits[conf.default_circuit];


	count_month(cir, sd_year, sd_month, &dur, &tin, &tout);
	if (mode == PRINT_ALL || mode == PRINT_TIME)
	{
		printf("Online-Zeit fuer diesen Monat (Circuit: %s): %s h\n", cir, (char*)format_seconds(dur));
	}
	else
	{
		printf("Traffic fuer diesen Monat (Circuit: %s):\n", cir);
	}
	if (mode == PRINT_ALL || mode == PRINT_TRAFFIC)
	{
		printf("Incoming Traffic: %05.2f MB\n",	format_traffic(tin));
		printf("Outgoing Traffic: %05.2f MB\n\n", format_traffic(tout));
	}

	count_day(cir, this_year, this_month, today, &dur, &tin, &tout);
	if (mode == PRINT_ALL || mode == PRINT_TIME)
	{
		printf("Online-Zeit fuer heute (Circuit: %s): %s h\n", cir, (char*)format_seconds(dur));
	}
	else
	{
		printf("Traffic fuer heute (Circuit: %s):\n", cir);
	}
	if (mode == PRINT_ALL || mode == PRINT_TRAFFIC)
	{
		printf("Incoming Traffic: %05.2f MB\n",	format_traffic(tin));
		printf("Outgoing Traffic: %05.2f MB\n", format_traffic(tout));
	}
}


void cmdl_A(int argc, char **argv)
{
	int dur = 0, day = 0, month = 0, year, sd_month, sd_year;
	float tin = 0, tout = 0;

	// Datum festlegen bzw. auslesen
	if (argc > 2 && argv[2])
		format_tl_date(argv [2], &day, &month, &year);
	else
	{	// kein Datum angegeben, also nehmen wir "heute"
		day = today;
		month = this_month;
		year = this_year;
	}

	// Monat verringern, um im richtigen AZR zu bleiben
	sd_month = month;
	sd_year = year;
	if (month > 0 && day < conf.start_day)
	{
		sd_year = (month == 1) ? (year - 1) : year;
		sd_month = (month == 1) ? 12 : (month - 1);
	}
					
	// Circuit festlegen, Default-Circuit oder den von der CmdLine
	char *cir = conf.circuits[conf.default_circuit];
	if (argc > 3) cir = argv[3];

	if (day > 0 && month > 0)
	{	// Tageswert
		count_day(cir, sd_year, sd_month, day, &dur, &tin, &tout);
	}
	else if (month > 0)
	{	// Monatswert
		count_month(cir, sd_year, sd_month, &dur, &tin, &tout);
	}
	else
	{	// Jahreswert
		count_year(cir, year, &dur, &tin, &tout);
	}

	// Ausgabe
	printf("%s %05.2f:%05.2f\n", (char*)format_seconds(dur), format_traffic(tin), format_traffic(tout));

}


int do_checks()
{
	if (! read_conf_file(CONFFILE))
	{
		fprintf(stderr, "Fehler beim Lesen der Konfiguration!\n");
		return 0;
	}
	
	if (! file_exists(conf.logfile))
	{
		fprintf(stderr, "Fehler %s existiert nicht!\n", conf.logfile);
		return 0;
	}

	if (! read_file())
	{
		fprintf(stderr, "Fehler beim Auslesen der Log-Datei!\n");
		return 0;
	}

	return 1;
}


int file_exists(char* filename)
{
	FILE *fp1 = fopen(filename, "r");
   	if (fp1 == NULL)
   	{
      	return 0;
   	}
	else
	{
	   	fclose(fp1);
		return 1;
	}
	
}


#ifdef TEST
void print_conns(void)
{
	int i;
	for (i = 0; i < conns_len; i++) {
		printf("%d:\t", i);
		printf("Name:\t%s\t", conns[i].cir_name);
		printf("In:\t%f\t", conns[i].tin);
		printf("Out:\t%f\t", conns[i].tout);
		printf("Device:\t%s\n", conns[i].device);
	}
}


void print_conn(int i)
{
	printf("Name:\t%s\t", conns[i].cir_name);
	printf("In:\t%f\t", conns[i].tin);
	printf("Out:\t%f\t", conns[i].tout);
	printf("Device:\t%s\n", conns[i].device);

}
#endif


int read_file(void) {
	
	FILE *fp1;
	struct tm tmse;
	int sd, sm, sj, sh, smin, ss;
	int ed, em, ej, eh, emin, es;
	int c = 0, line = 0, size;
	float t;
	char tmp[150];
	
	fp1 = fopen(conf.logfile, "r");
   	if (fp1 == NULL)
   	{
      	return FALSE;
   	}
	
	// Zeilen zählen
	conns_len = 0;
	while(!feof(fp1))
	{
		fgets(tmp, 150, fp1);
		conns_len++;
		//printf("%d\t%d: %s", conns_len, strlen(tmp), tmp);
	}
	//conns_len--;	// TODO, dirty Quick-Hack, weil immer eine Zeile zuviel gezählt wird
	
	// Array für die Verbindungen anlegen
	size = sizeof(struct connection) * conns_len;
	conns = malloc(size);
	
	rewind(fp1);	// Dateizeiger zurücksetzen
	
	// T-Online 2004/04/01 00:10:00 2004/04/30 23:50:00  290760  290760   0.00 0 149851996 0 0 ippp0 1 0.0000
   	do
   	{
      	if (c != EOF)
		{
      		c = fscanf(fp1,"%s %4d/%2d/%2d %2d:%2d:%2d %4d/%2d/%2d %2d:%2d:%2d %d %f %f %f %f %f %f %s %f %f", 
							conns[line].cir_name,
							&sj, &sm, &sd,
							&sh, &smin, &ss,
							&ej, &em, &ed,
							&eh, &emin, &es,
							&conns[line].duration,
							&t,
							&t,
							&t,
							&conns[line].tin,
							&t,
							&conns[line].tout,
							conns[line].device,
							&t,
							&t
			);

			tmse.tm_sec = ss;
			tmse.tm_min = smin;
			tmse.tm_hour = sh;
			tmse.tm_mday = sd;
			tmse.tm_mon = sm - 1;
			tmse.tm_year = sj - 1900;
			conns[line].start = mktime(&tmse);
			
			tmse.tm_sec = es;
			tmse.tm_min = emin;
			tmse.tm_hour = eh;
			tmse.tm_mday = ed;
			tmse.tm_mon = em - 1;
			tmse.tm_year = ej - 1900;
			conns[line].end = mktime(&tmse);

			//print_conn();
			//printf("%d: %d/%d/%d %d:%d:%d %d/%d/%d %d:%d:%d\n", line, sj, sm, sd, sh, smin, ss, ej, em, ed, eh, emin, es);
			
			line++;
			
		}
   	} while (c != EOF);
	
   	fclose(fp1);
	conns_len--;
	
	return TRUE;
}


int read_conf_file(char* filename)
{
	FILE *fp1;
	int c, idx = 1;
	char *key, *value;
	char line[200];
	
	key = malloc(100);
	value = malloc(100);
	
	strcpy(conf.circuits[0], "all");
	
	fp1 = fopen(filename, "r");
   	if (fp1 == NULL)
   	{
      	return FALSE;
   	}

   	do 
   	{
      	c = fscanf(fp1,"%s", line);
      	if (c != EOF)
		{
         	key = strtok(line, "=");
			value = strtok(NULL, "'");
			
			if (strcmp(key, "ISDN_CIRCUITS_N") == 0)
			{
				conf.circuits_anz = atoi(value) + 1;
			}
			else if (strcmp(key, "IMOND_LOGDIR") == 0)
			{
				strcpy(conf.logfile, value);
				strcat(conf.logfile, "/imond.log");
			}
			else if (strcmp(key, "TIMELOG_CIRCUIT") == 0)
			{
				conf.default_circuit = atoi(value);
			}
			else if (strcmp(key, "TIMELOG_START_DAY") == 0)
			{
				conf.start_day = atoi(value);
			}
			else if (strcmp(key, "TIMELOG_LCD_MONTH") == 0)
			{
				if (value != NULL) strcpy(conf.lcd_mon_file, value);
				else strcpy(conf.lcd_mon_file, "disabled");
			}
			else if (strcmp(key, "TIMELOG_LCD_DAY") == 0)
			{
				if (value != NULL) strcpy(conf.lcd_day_file, value);
				else strcpy(conf.lcd_day_file, "disabled");
			}
			else if (strncmp(key, "ISDN_CIRC_", 10) == 0)
			{
				strcpy(conf.circuits[idx++], value);
			}

			//printf("%s=%s\n", key, value);
		}
   	} while (c != EOF);

   	fclose(fp1);

   	return TRUE;
}


char *format_seconds(int s)
{
        int h, m;
        char *hour = malloc(10);

        m = s / 60;
        s = s % 60;
        h = m / 60;
        m = m % 60;
        if (PRINT_SHORT)
                sprintf(hour, "%d:%02d", h, m);
        else
                sprintf(hour, "%02d:%02d", h, m);

        return hour;
}


float format_traffic(float tr)
{
        return tr / 1048576.0f;
}


time_t get_time(int year, int month, int day)
{
        struct tm t;
		
        t.tm_sec = 0;
        t.tm_min = 0;
        t.tm_hour = 0;
        t.tm_mday = day;
        t.tm_mon = month - 1;
        t.tm_year = year - 1900;

		return mktime(&t);

}


struct tm get_time_as_tm(int year, int month, int day)
{
        struct tm t;
        t.tm_sec = 0;
        t.tm_min = 0;
        t.tm_hour = 0;
        t.tm_mday = day;
        t.tm_mon = month - 1;
        t.tm_year = year - 1900;

        return t;

}


/* wertet das Datum von der commandline, z.B. 2005/02/16, aus
   und gibt es z.B. als 12.03.2005 zurück */
char *nice_tl_date(char *raw_date)
{
	int year, month = 0, day = 0;
	char *result = malloc(10);
	char *tmp = malloc(5);
	year = atoi(strtok(raw_date, "/"));
	tmp = strtok(NULL, "/");
	if (tmp != NULL)
	{
		month = atoi(tmp);
		tmp = strtok(NULL, "/");
		if (tmp != NULL)
		{
			day = atoi(tmp);
		}
	}
	if (day > 0)
		sprintf(result, "den %02d.%02d.%d", day, month, year);
	else if (month > 0)
		sprintf(result, "%s %d", monatsnamen[month - 1], year);
	else
		sprintf(result, "%d", year);
		
	return result;
}


/* wertet das Datum von der commandline, z.B. 2005/02/16, aus */
void format_tl_date(char *str, int *day, int *month, int *year)
{
	char *tmp = malloc(5);
	*year = atoi(strtok(str, "/"));
	tmp = strtok(NULL, "/");
	if (tmp != NULL)
	{
		*month = atoi(tmp);
		tmp = strtok(NULL, "/");
		if (tmp != NULL)
		{
			*day = atoi(tmp);
		}
	}
}


/* wertet die Online-Zeit vom Imond(00:00:00) aus */
int format_imond_time(char *str)
{
	int h, m, s;
	if (strlen(str) == 8)
	{
		sscanf(str, "%02d:%02d:%02d", &h, &m, &s);
	}
	return ((h * 3600) + (m * 60) + s);
}


int get_time_from_imond(char *circuit)
{
    /* todo:
		- Channels bei DSL?
	*/
	int	port = 5000;
    int	fd;
	int ztime = 0;
	char buf[128];
    FILE *fp;
	
	fp = fopen ("/var/run/imond.port", "r");
	if (fp)
	{
		(void) fscanf (fp, "%d", &port);
		fclose (fp);
	}


#ifdef TEST
	// nur für mich, damit ich timelog auch im LAN mit dem imond testen kann
    if ((fd = service_connect ("uranus", port)) < 0)
#else
    if ((fd = service_connect ("127.0.0.1", port)) < 0)
#endif	
    {
		perror("get_time_from_imond");
		return 0;
    }
	
	send_command (fd, "pass");
	if (atoi(get_answer (fd)) < 4)
	{
		return 0;
	}
	
	// Channel 1 -  00:12:32
	send_command (fd, "status 1");
	if (! strcmp(get_answer (fd), "Online"))
	{
		// Check, ob aktuelle Verbindung dem gewählten Circuit entspricht
		send_command(fd, "phone 1");
		strcpy(buf, get_answer (fd));
		if (! strcmp(buf, circuit) || ! strcmp(buf, "Alle") || ! strcmp(buf, "all"))
		{
			send_command(fd, "online-time 1");
			ztime += format_imond_time(get_answer(fd));
		}
	}
	// Channel 2
	send_command (fd, "status 2");
	if (! strcmp(get_answer (fd), "Online"))
	{
		send_command(fd, "phone 2");
		strcpy(buf, get_answer (fd));
		if (! strcmp(buf, circuit) || ! strcmp(buf, "Alle") || ! strcmp(buf, "all"))
		{
			send_command(fd, "online-time 2");
			ztime += format_imond_time(get_answer(fd));
		}
	}
	
	close (fd);
	return ztime;
	
}


void show_header(char *me)
{
	printf("%s %s\n\n", basename(me), VERSION);
}


void show_usage(char *me)
{
	show_header(me);
	printf("Usage: %s [-atzAh] date [circuit]\n\n", basename(me));
}


void show_help(char *me)
{
	char date[11], date_short[9];

	strftime(date, 11, "%Y/%m/%d", tm);
	strftime(date_short, 9, "%Y/%m/", tm);
	
	show_header(me);
	printf("Usage: %s options date [circuit]\n\n", basename(me));
	printf("Optionen:\n");
	printf("-z    - Zeitwerte ausgeben\n");
	printf("-t    - Trafficwerte ausgeben\n");
	printf("-a    - Traffic- und Zeitwerte ausgeben (-A fuer kurze Ausgaben)\n\n");
	printf("Format von date:\n");
	printf("yyyy/mm/dd um einen Tag anzugeben\n");
	printf("yyyy/mm/ um einen Monat anzugeben\n");
	printf("yyyy/ um ein Jahr anzugeben\n");
	printf("(z.B. %s fuer heute, %s fuer diesen Monat, \"date +%%Y/%%m/%%d\" in Scripten)\n\n", date, date_short);
	printf("Bei Weglassen des Circuits, wird der in der config/timelog.txt eingestellte Standard-Circuit(%s) benutzt.\n\n", conf.circuits[conf.default_circuit]);
    printf("Beispiele:\n");
    printf("%s -z %s\n", basename(me), date);
    printf("%s -z %s all\n", basename(me), date);
    printf("%s -a %s %s\n", basename(me), date_short, conf.circuits[1]);
}


void lcd_mode(void)
{
	FILE *fp;
	int dur = 0, month = this_month;
	float tin = 0, tout = 0;
	PRINT_SHORT = TRUE;
	
	// Circuit festlegen: Default-Circuit
	char *cir = conf.circuits[conf.default_circuit];

	// Tageswert
	fp = fopen(conf.lcd_day_file, "w");
   	if (fp != NULL)
 	{
		count_day(cir, this_year, this_month, today, &dur, &tin, &tout);
		fputs((char*)format_seconds(dur), fp);
		fclose(fp);
	}
	else
	{
		perror("timelog: lcd_mode");
	}
	
	// Monatswert
	fp = fopen(conf.lcd_mon_file, "w");
   	if (fp != NULL)
 	{
		if (today <= conf.start_day)
		{
			month = get_prev_month(this_month);
		}
		count_month(cir, this_year, month, &dur, &tin, &tout);
		fputs((char*)format_seconds(dur), fp);
		fclose(fp);
	}		
	else
	{
		perror("timelog: lcd_mode");
	}

}



int main(int argc, char** argv)
{
	// Checkt versch. Sachen, und schmeißt uns evtl. gleich wieder raus
	if (! do_checks()) exit(EXIT_FAILURE);

	me = argv[0];
	
	seconds = time((time_t *) 0);
	tm = localtime(&seconds);
	this_year = tm->tm_year + 1900;
	this_month = tm->tm_mon + 1;
	today = tm->tm_mday;

	if (argc > 1)
	{
		if (argv[1][0])
    	{
			PRINT_SHORT = FALSE;	// Default-Vorbelegung
			switch(argv[1][1])
			{
            	case 'H': 
            	// HTML-Ausgabe
		    	{
					if (argc == 6 && ! strcmp(argv[2], "sum"))
					{
						// Übersicht timelog -H sum Circuit Letzer Monat Gestern
						data_summery(argv[3], atoi(argv[4]), atoi(argv[5]));
					}
					else if (argc == 5 && ! strcmp(argv[2], "year"))
					{
						// Jahres-Daten timelog -H year Circuit year
						data_year(argv[3], atoi(argv[4]));
					}
					else if (argc == 6 && ! strcmp(argv[2], "month"))
					{
						// Monats-Daten timelog -H month Circuit year month
						data_month(argv[3], atoi(argv[4]), atoi(argv[5]));
					}
					else if (argc == 7 && ! strcmp(argv[2], "day"))
					{
						// Tages-Daten timelog -H day Circuit year month day
						data_day(argv[3], atoi(argv[4]), atoi(argv[5]), atoi(argv[6]));
					}
					else if (argc == 5 && ! strcmp(argv[2], "gyear"))
					{
						// Jahres-Grafik timelog -H gyear Circuit year
						graph_year(argv[3], atoi(argv[4]));
					}
					else if (argc == 6 && ! strcmp(argv[2], "gmonth"))
					{
						// Monats-Grafik timelog -H gmonth Circuit year month
						graph_month(argv[3], atoi(argv[4]), atoi(argv[5]));
					}
					else if (argc == 7 && ! strcmp(argv[2], "gday"))
					{
						// Tages-Grafik timelog -H gday Circuit year month day
						graph_day(argv[3], atoi(argv[4]), atoi(argv[5]), atoi(argv[6]));
					}
			    	break;
            	}
        
            	case 'L':
            	{                
                	lcd_mode();		// LCD-Modus
					break;
				}

            	case 'A':
            	{                
					cmdl_A(argc, argv);
					
					break;
				}

            	case 'a':
				{
					show_header(argv[0]);
					
					if (argc > 2 && argv[2])
					{
						cmdl_date(argc, argv, PRINT_ALL);
					}
					else
					{
						cmdl_gen(argc, argv, PRINT_ALL);
					}
					
					break;
				}
    	        case 't':
					show_header(argv[0]);
					
					if (argc > 2 && argv[2])
					{
						cmdl_date(argc, argv, PRINT_TIME);
					}
					else
					{
						cmdl_gen(argc, argv, PRINT_TIME);
					}
					
					break;
				case 'z':
					show_header(argv[0]);
					
					if (argc > 2 && argv[2])
					{
						cmdl_date(argc, argv, PRINT_TRAFFIC);
					}
					else
					{
						cmdl_gen(argc, argv, PRINT_TRAFFIC);
					}
					
					break;
		        case 'h':
				{
					show_help(argv[0]);
					break;
				}
				case '-':
				{
					if (! strcmp(argv[1], "--help"))
					{
						show_help(argv[0]);
						break;
					}

				}
				default:
				{
					show_usage(argv[0]);
					break;
				}
			}
		}
	}
	else 
	{
		show_usage(argv[0]);
	}

	free(conns);	// Speicher freigeben
	return EXIT_SUCCESS;
}
