/*
 MultiImonC - a multiplatform imonc for fli4l
 Copyright (C) 2003, 2004 Michael Hanselmann
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
 $Id: ImonC.cpp,v 1.16 2004/07/10 21:02:55 michael Exp $
 */

#include "ImonC.h"
#include "Md5.h"
#include <iostream>
#include <sstream>
#include "signal.h"

#include <errno.h>
#include <stdio.h>
#include <cstring>
#include <cstdlib>

#include <unistd.h>
#include <signal.h>

#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>				/* decl of inet_addr()	    */
#include <sys/socket.h>

#define COUT(t) std::cout << #t << " : " << t << std::endl

using namespace std;

ImonC::Socket* ImonC::Socket::SigAlarmObj = NULL;

ImonC::ImonC() : m_strHost(""), m_iPort(0), m_oSocket() {
}

bool ImonC::Connect(std::string i_strHostname,
                    unsigned int i_iPort) {
    m_strHost = i_strHostname;
    m_iPort = i_iPort;
    return m_oSocket.Connect(m_strHost, m_iPort);
}

string ImonC::GetHost() {
    return m_oSocket.GetHost();
}

unsigned int ImonC::GetPort() {
    return m_oSocket.GetPort();
}

ImonC::~ImonC() {
    Disconnect();
}

bool ImonC::IsConnected() {
    return m_oSocket.IsConnected();
}

void ImonC::Disconnect() {
    while(!m_oChannelData.empty()) {
        delete m_oChannelData.back();
        m_oChannelData.pop_back();
    }
    while(!m_oCircuitData.empty()) {
        delete m_oCircuitData.back();
        m_oCircuitData.pop_back();
    }
    m_oSocket.Disconnect();
}

ImonC::uint64_t ImonC::atoull(const char *Buffer) const {
    uint64_t Result = 0;
    for (; *Buffer; Buffer++) {
        uint64_t OldRes = Result;
        Result *= 10;
        if(isdigit(*Buffer)) {
            Result += *Buffer-'0';
        }
        if (Result < OldRes)   // Uh, oh, overflow detected!
            throw std::overflow_error("overflow");
    }
    return Result;
}


string ImonC::GetAnswer(string i_strCommand) {
    string strResult("");
    string strCommand(i_strCommand);
    
    if(m_oSocket.IsConnected()) {
        m_oSocket.SetConnectionLostObj(this);
        
#if 0
        cerr << "Write: " << strCommand << endl;
#endif

        // Do not send commands longer than 250 chars to imond.
        if(strCommand.length() > 250) {
            strCommand = strCommand.substr(0, 250);
        }
        
#if 0
        cerr << "Write2: " << strCommand << endl;
#endif
        
        m_oSocket.Write(strCommand + "\r\n");
        strResult = m_oSocket.Read();

#if 0
        cerr << "Read: " << strResult << endl;
#endif

        if(strResult.find("OK ") == 0) {
            strResult = strResult.substr(3, strResult.length()-3);
        }else{
            strResult = "";
        }
    }
    return strResult;
}

string ImonC::GetMultilineAnswer(string i_strCommand) {
    string strResult("");
    if(m_oSocket.IsConnected()) {
        m_oSocket.SetConnectionLostObj(this);
        
        m_oSocket.Write(i_strCommand + "\r\n");

        bool loop = true;
        do {
            string strCurrent = m_oSocket.Read();
            if(strCurrent.find(" ") == 0) {
                strResult.append(strCurrent).append("\n");
            }else{
                loop = false;
            }
        }while(loop);
    }
    return strResult;
}

void ImonC::Dial() {
    static_cast<void>(GetAnswer("dial"));
}

void ImonC::Hangup() {
    static_cast<void>(GetAnswer("hangup"));
}

void ImonC::AddLink() {
    ostringstream str;
    str << Route();
    static_cast<void>(GetAnswer(string("addlink ") + str.str()));
}

void ImonC::RemoveLink() {
    ostringstream str;
    str << Route();
    static_cast<void>(GetAnswer(string("removelink ") + str.str()));
}

int ImonC::Links(int i_iDefault) {
    ostringstream str;
    str << (i_iDefault = -1? Route(): i_iDefault);
    return atoi(GetAnswer(string("links ") + str.str()).c_str());
}

string ImonC::Date() {
    return GetAnswer("date");
}

string ImonC::Support(std::string i_strRootPassword) {
    return GetMultilineAnswer("support " + i_strRootPassword);
}

string ImonC::Timetable() {
    return GetMultilineAnswer("timetable");
}

unsigned long long int ImonC::Uptime() {
    return atoull(GetAnswer("uptime").c_str());
}

unsigned short int ImonC::CPU() {
    stringstream ss(GetAnswer("cpu"));
    unsigned short int result;
    ss >> result;
    return result;
}

ImonC::Dialmode_t ImonC::Dialmode(Dialmode_t dmSet) {
    switch(dmSet) {
        case Off:
            static_cast<void>(GetAnswer("dialmode off"));
            break;
        case Auto:
            static_cast<void>(GetAnswer("dialmode auto"));
            break;
        case Manual:
            static_cast<void>(GetAnswer("dialmode manual"));
            break;
        case _Default:
            break;
    }
    string dialmode = GetAnswer("dialmode");
    if(dialmode == "auto") {
        return Auto;
    } else if (dialmode == "off") {
        return Off;
    } else if (dialmode == "manual") {
        return Manual;
    }
    return _Default;
}

int ImonC::Circuits() {
    stringstream ss(GetAnswer("circuits"));
    int result;
    ss >> result;
    return result;
}

int ImonC::Channels() {
    stringstream ss(GetAnswer("channels"));
    int result;
    ss >> result;
    return result;
}

int ImonC::Route(int i_iRoute) {
    string cmd = "route";
    if(i_iRoute >= 0) {
        ostringstream str;
        str << i_iRoute;
        cmd += " " + str.str();
    }
    stringstream ss(GetAnswer(cmd));
    int result;
    ss >> result;
    return result;
}

bool ImonC::IsUser() {
    return (Pass() >= 2) && (Uptime() > 0);
}

bool ImonC::IsAdmin() {
    return (Pass() >= 4);
}

const ImonC::ChannelVector& ImonC::GetChannels() {
    if(m_oChannelData.empty() && m_oSocket.IsConnected()) {
        unsigned int channelcount = Channels();
        if(atoi(GetAnswer("pppoe").c_str()) == 1) {
            Channel* c = new Channel(*this, "pppoe");
            c->SetType(Channel::PPPoE);
            m_oChannelData.push_back(c);
        }

        for( unsigned int i = 1; i <= channelcount; i++) {
            ostringstream str;
            str << i;
            Channel* c = new Channel(*this, str.str());
            c->SetType(Channel::ISDN);
            m_oChannelData.push_back(c);
        }
    }
    
    return m_oChannelData;
}

const ImonC::CircuitVector& ImonC::GetCircuits() {
    if(m_oCircuitData.empty() && m_oSocket.IsConnected()) {
        unsigned int circuitcount = Circuits();
        for(unsigned int i = 1; i <= circuitcount; i++) {
            ostringstream str;
            str << i;
            Circuit* c = new Circuit(*this, i);
            m_oCircuitData.push_back(c);
        }
    }

    return m_oCircuitData;
}

string ImonC::Salt() {
    if(Version().protocol <= 10) {
        return "";
    }else{
        return GetAnswer("salt");
    }
}

unsigned short int ImonC::Pass(string i_strPassword) {
    string strPassword(i_strPassword);
    
    // Cut passwords to 63 chars.
    if(strPassword.length() > 63) {
        strPassword = strPassword.substr(0, 63);
    }
    
    if(Version().protocol <= 10) {
        // No MD5
        return atoi(GetAnswer("pass " + strPassword).c_str());
    }else{
        // MD5
        string strCommand = "md5pass";
        if(i_strPassword != "") {
            Md5 md5;
            strCommand += " " + md5(strPassword + Salt());
        }
        return atoi(GetAnswer(strCommand).c_str());
    }
}

ImonC::Version_t ImonC::Version() {
    if(m_oVersionCache.protocol == 0) {
        string v = GetAnswer("version");

        unsigned int loc = v.find(" ", 0);
        if(loc != string::npos) {
            m_oVersionCache.protocol = atoi(v.substr(0,loc).c_str());
            m_oVersionCache.version = v.substr(loc+1,v.length()-loc);
        }
    }
    
    return m_oVersionCache;
}

string ImonC::Channel::IP() {
    return m_oImonC.GetAnswer("ip " + m_strName);
}

string ImonC::Channel::Phone() {
    return m_oImonC.GetAnswer("phone " + m_strName);
}

string ImonC::Channel::InOut() {
    return m_oImonC.GetAnswer("inout " + m_strName);
}

string ImonC::Channel::OnlineTime() {
    return m_oImonC.GetAnswer("online-time " + m_strName);
}

string ImonC::Channel::Time() {
    return m_oImonC.GetAnswer("time " + m_strName);
}

string ImonC::Channel::ChargeTime() {
    return m_oImonC.GetAnswer("chargetime " + m_strName);
}

string ImonC::Channel::Charge() {
    return m_oImonC.GetAnswer("charge " + m_strName);
}

ImonC::Channel::Traffic_t ImonC::Channel::Quantity() {
    Traffic_t t;
    stringstream ss(m_oImonC.GetAnswer("quantity " + m_strName));
    
    if(m_oImonC.Version().protocol <= 10) {
        ss >> t.in;
        ss >> t.out;
    }else{
        uint64_t overflowsIn;
        uint64_t in;
        uint64_t overflowsOut;
        uint64_t out;

        ss >> overflowsIn;
        ss >> in;
        ss >> overflowsOut;
        ss >> out;

        // 2^32 = 4294967296 = "<< 32"
        t.in = (overflowsIn << 32) + in;
        t.out = (overflowsOut << 32) + out;
    }

    return t;
}

ImonC::Channel::Traffic_t ImonC::Channel::Rate() {
    Traffic_t t;
    stringstream ss(m_oImonC.GetAnswer("rate " + m_strName));
    ss >> t.in;
    ss >> t.out;
    return t;
}

ImonC::Channel::Channel(ImonC& imonc, string name) : m_oImonC(imonc), m_strName(name) {
}

ImonC::Channel::Type_t ImonC::Channel::Type() {
    return m_tType;
}

ImonC::Channel::Status_t ImonC::Channel::Status() {
    string r = m_oImonC.GetAnswer("status " + m_strName);
    if(r == "Online") {
        return Online;
    }else{
        return Offline;
    }
}

ImonC::Circuit::Circuit(ImonC& imonc, int number) : m_oImonC(imonc), m_iNumber(number) {
}

bool ImonC::Circuit::CanSetHupTimeOut() {
    return (Device() != "pppoe");
}

string ImonC::Circuit::Name() {
    ostringstream str;
    str << m_iNumber;
    return m_oImonC.GetAnswer(string("circuit ") + str.str());
}

string ImonC::Circuit::Device() {
    ostringstream str;
    str << m_iNumber;
    return m_oImonC.GetAnswer(string("device ") + str.str());
}

long ImonC::Circuit::HupTimeOut(long timeout) {
    ostringstream str;
    str << m_iNumber;
    
    string cmd = "hup-timeout " + str.str();

    if(CanSetHupTimeOut() && timeout > 0) {
        ostringstream str2;
        str2 << timeout;
        cmd = cmd + string(" ") + str2.str();
    }
    return atol(m_oImonC.GetAnswer(cmd.c_str()).c_str());
}

bool ImonC::IsAllowed(std::string i_strCommand) {
    return (atoi(GetAnswer("is-allowed " + i_strCommand).c_str()) == 1);
}

bool ImonC::IsDialAllowed() {
    return IsAllowed("dial");
}

bool ImonC::IsDialmodeAllowed() {
    return IsAllowed("dialmode");
}

bool ImonC::IsRouteAllowed() {
    return IsAllowed("route");
}

bool ImonC::IsAddLinkAllowed() {
    return IsAllowed("addlink");
}

bool ImonC::IsRebootAllowed() {
    return IsAllowed("reboot");
}

bool ImonC::IsHaltAllowed() {
    return IsAllowed("halt");
}

void ImonC::Halt() {
    static_cast<void>(GetAnswer("halt"));
}

void ImonC::Reboot() {
    static_cast<void>(GetAnswer("reboot"));
}

string ImonC::GetFile(std::string i_strRootPassword,
                      std::string i_strFilename) {
    std::string strResult;
    if(long lSize = atoi(GetAnswer(string("send ") + i_strFilename + " " +
                                   i_strRootPassword).c_str())) {
        char strBuffer[1025];
        long lReceived = 0;
        const char strACK_STRING = ACK_CHAR;
        while(lReceived < lSize) {
            const int iLen = read(m_oSocket.m_iFD, strBuffer, 1024);
            strBuffer[iLen] = '\0';

            strResult += strBuffer;
            lReceived += iLen;

            write(m_oSocket.m_iFD, &strACK_STRING, 1);
        }
        m_oSocket.Read();
    }
    return strResult;
}

bool ImonC::UploadFile(std::string i_strRootPassword,
                       std::string i_strFilename,
                       std::string i_strContent) {
    ostringstream str;
    str << i_strContent.size();
    string strCommand;

    strCommand.append("receive ")
              .append(i_strFilename)
              .append(" ")
              .append(str.str())
              .append(" ")
              .append(i_strRootPassword)
              .append("\r\n");

    write(m_oSocket.m_iFD, strCommand.c_str(), strCommand.size());

    char strBuffer4[4];

    read(m_oSocket.m_iFD, strBuffer4, 3);

    if(strBuffer4[0] != ACK_CHAR) {
        m_oSocket.Read();
        return false;
    }

    const long lSize = i_strContent.size();
    long lSent = 0;

    while(lSent < lSize) {
        const int iLen = ((lSize - lSent > 1024)?1024:lSize - lSent);
        lSent += iLen;
        write(m_oSocket.m_iFD, i_strContent.substr(0, iLen).c_str(), iLen);

        read(m_oSocket.m_iFD, strBuffer4, 3);
        if(strBuffer4[0] != ACK_CHAR) {
            m_oSocket.Read();
            return false;
        }
    }

    m_oSocket.Read();
    return true;
}

bool ImonC::UploadFile(std::string i_strRootPassword,
                       std::string i_strFilename,
                       std::istream& i_oStream) {
    i_oStream.seekg(0, std::ios::end);
    const long lSize = i_oStream.tellg();
    i_oStream.seekg(0, std::ios::beg);

    ostringstream str;
    str << lSize;
    string strCommand;

    strCommand.append("receive ")
              .append(i_strFilename)
              .append(" ")
              .append(str.str())
              .append(" ")
              .append(i_strRootPassword)
              .append("\r\n");

    write(m_oSocket.m_iFD, strCommand.c_str(), strCommand.size());

    char strBuffer4[4];

    read(m_oSocket.m_iFD, strBuffer4, 3);

    if(strBuffer4[0] != ACK_CHAR) {
        m_oSocket.Read();
        return false;
    }

    long lSent = 0;
    char strBuffer[1025];

    while(lSent < lSize) {
        const int iLen = ((lSize - lSent > 1024)?1024:lSize - lSent);

        memset(strBuffer, 0, 1024);

        i_oStream.seekg(lSent);
        i_oStream.readsome(strBuffer, iLen);

        write(m_oSocket.m_iFD, strBuffer, iLen);
        lSent += iLen;

        read(m_oSocket.m_iFD, strBuffer4, 3);
        if(strBuffer4[0] != ACK_CHAR) {
            m_oSocket.Read();
            return false;
        }
    }

    m_oSocket.Read();
    return true;
}

void ImonC::SetConnectionLostCallback(AbstractCallback* callback) {
    m_pConnectionLostCallbackObj = callback;
}

void ImonC::ConnectionLost() {
	(*m_pConnectionLostCallbackObj)();
}

void ImonC::InitSignalHandler() {
    ImonC::Socket::InitSignalHandler();
}

// ###############################################################################
// Socket

ImonC::Socket::Socket():
m_iFD(-1),
m_iTimeout(DefaultTimeout) {
    m_pImonC = NULL;
    SigAlarmObj = NULL;
}

ImonC::Socket::~Socket() {
    Disconnect();
}

void ImonC::Socket::SetConnectionLostObj(ImonC* imonc) {
    m_pImonC = imonc;
}

void ImonC::Socket::SetTimeout(const int i_iTimeout) {
    m_iTimeout = i_iTimeout;
}

void ImonC::Socket::Write(const string i_strCommand) {
    if(!IsConnected()) return;
    
    write(m_iFD, i_strCommand.c_str(), i_strCommand.length());
}

string ImonC::Socket::Read() {
    static char buf[8192];
    char *tmp;
    char c[1];
    int len = 0;

    if(!IsConnected()) return "";
    
    tmp = buf;

    SigAlarmObj = this;

    alarm (m_iTimeout);
    while (read(m_iFD, c, 1) > 0) {
        len ++;
        if (c[0] == '\n')
            break; /* break at end of line */
        if (c[0] != '\r')
            *tmp++ = c[0];
    }
    alarm (0);

    SigAlarmObj = NULL;
    
    *tmp = '\0';
    
    return string(buf);
}

void ImonC::Socket::ConnectionLost() {
    Disconnect();
    if(m_pImonC) {
        m_pImonC->ConnectionLost();
    }
}

bool ImonC::Socket::Connect(const string i_strHostname,
                            const unsigned int i_iPort) {
    int	on = 1;

    Disconnect();

    SigAlarmObj = this;

    alarm (m_iTimeout); // resolve & connect-timeout

    m_strHostname = i_strHostname;
    m_iPort = i_iPort;

    struct sockaddr_in addr;
    struct hostent *host = gethostbyname(i_strHostname.c_str());
    if(!host) {
        m_iFD = -1;
        throw ConnectError(std::string("Host ").
                           append(i_strHostname).
                           append(" not found."));
    }

    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = *(long*)(host->h_addr);
    addr.sin_port = htons(i_iPort);

    if((m_iFD = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        /* open socket */
        m_iFD = -1;
        throw ConnectError("Error opening socket");
    }

    setsockopt(m_iFD, IPPROTO_TCP, TCP_NODELAY, &on, sizeof(on));

    if(connect(m_iFD, (struct sockaddr*)&addr, sizeof(addr)) != 0) {
        char* strError = strerror(errno);
        close(m_iFD);

        m_iFD = -1;
        throw ConnectError(strError);
    }

    alarm (0);

    SigAlarmObj = NULL;

    return true;
}

string ImonC::Socket::GetHost() const {
    return m_strHostname;
}

unsigned int ImonC::Socket::GetPort() const {
    return m_iPort;
}

void ImonC::Socket::Disconnect() {
    if(IsConnected()) {
        close(m_iFD);
    }
    m_iFD = -1;
}

bool ImonC::Socket::IsConnected() const {
    return (m_iFD != -1);
}

void ImonC::Socket::InitSignalHandler() {
    struct sigaction act;
    struct sigaction act2;

    sigaction(SIGALRM, NULL, &act2);
    act2.sa_flags &= ~SA_RESTART;
    sigaction(SIGALRM, &act2, NULL);

    // Assign sig_pipe as SIGPIPE handler
    act.sa_handler = Socket::SigAlarm;

    // We don't want to block any other signals
    sigemptyset(&act.sa_mask);

    if (sigaction(SIGPIPE, &act, NULL) < 0) {
        std::cerr << "sigaction failed" << std::endl;
    }

    signal(SIGALRM, Socket::SigAlarm);
}

void ImonC::Socket::SigAlarm(int signo) {
    if(Socket::SigAlarmObj) {
        Socket::SigAlarmObj->ConnectionLost();
    }
}
