char ProgInfo[]=
"/*                                                \n"
"* File: run_tree.c                           \n"
"* Purpose: parse and run a decision tree          \n"
"* Author: Vincent Pagel ( pagel@tcts.fpms.ac.be ) \n"
"* Version : 0.4                                   \n"
"* Time-stamp: <1999-06-25 16:44:30 pagel>                          \n"
"*                                                 \n"
"* Copyright (c) 1998 Faculte Polytechnique de Mons (TCTS lab) \n"
"*                                                             \n"
"* This program is free software; you can redistribute it and/or modify \n"
"* it under the terms of the GNU General Public License as published by \n"
"* the Free Software Foundation version 1 \n"
"* \n"
"* This program is distributed in the hope that it will be useful, \n"
"* but WITHOUT ANY WARRANTY; without even the implied warranty of \n"
"* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n"
"* GNU General Public License for more details. \n"
"* \n"
"* You should have received a copy of the GNU General Public License \n"
"* along with this program; if not, write to the Free Software \n"
"* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. \n"
"* \n"
"* History: \n"
"* \n"
"*  12/08/98 : Created. To parse and run a decision tree in a .tree \n"
"*  file, whose structure is:                                   \n"
"*            SWITCH : '[' attrib  default_value CASE+ ']' \n"
"*            CASE   : feature SWITCH |  feature value          \n"
"*                                                              \n"
"*  attrib: [0..9]+  (index of the feature to test)             \n"
"*  feature: string                                             \n"
"*  val and default_value: string (phoneme)                     \n"
"*                                                              \n"
"*  17/08/98 : added a specific node counter, and change the method so that we\n"
"*      consider a big if_then_else in with each if count for 1, as well as each \n"
"*      return \n"
"*\n"
"*  16/05/99 : more explicit debug mode\n"
"*  21/06/99 : left to right transcription direction\n"
"*\n"
"*\n"
"*\n"
"*/\n";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <stdarg.h>

typedef char* Phoneme;
typedef char* Feature;
typedef Feature* TestVector;

/* Switch-case-default statement ! */
typedef struct 
{
  int feature_index;  /* index of the feature to carry the tests on */

  int nb_case;    /* number of cases available */
  struct Case ** cases;    /* table of case pointers */

  Phoneme default_phoneme; /* default return value */
} Switch;
#define feature_index(S) (S->feature_index)
#define nb_case(S) (S->nb_case)
#define cases(S) (S->cases)
#define default_phoneme(S) (S->default_phoneme)

enum 
{
  Terminal,							  /* Terminal node */
  Recursive							  /* Recursive node */
} TypeNode;

/* Case statement */
typedef struct Case
{
  Feature feature;    /* value of the tested feature */
  int type;          /* Terminal node or not */
  
  union 
  {
	 Switch* decision;  /* Embedded decision */
	 Phoneme return_phoneme;     /* or return value */
  } u;
} Case;
#define feature(C) (C->feature)
#define type(C)    (C->type)
#define decision(C)  (C->u.decision)
#define return_phoneme(C) (C->u.return_phoneme)

/*
 * Global variable
 */
int max_depth; /* volontary limit the depth of search in the tree */

int modulo; /* to print something each nth test */
int trace;  /* if true, display the decision path */


/* Attribute parsing in the file */
int  left_grapheme=0;
int  right_grapheme=0;
int  nb_feedback=0;
int  nb_skip=0;
int left_to_right;


/* 
 * Here we go ....
 */

void tabulate(int depth)
	  /* print the right number of space to justify the ouput */
{
  int i;
  
  for(i=0; i<depth; i++)
	 printf("  ");
}

void display_debug(int index)
	  /* print the origin of a vector's element (trace purpose) */
{
  /* we're in the TAG */
  if (index<=nb_skip)
	 printf("TAG%i",index);
  /* we're in the letters */
  else if (index<=nb_skip+left_grapheme+right_grapheme+1)
	 printf("L%i",index-nb_skip-left_grapheme-1);
  /* we're in the phoneme feedback */
  else
	 printf("P-%i",index-nb_skip-left_grapheme-right_grapheme-1);
}


Case* init_Case()
	  /* Alloc memory */
{
  Case* my_case= (Case*) malloc(sizeof(Case));

  return(my_case);
}

void close_Switch(Switch* sw);
/* Release the memory */

void close_Case(Case* cs)
	  /* Release the momory */
{
  free(feature(cs));
  switch (type(cs))
	 {
	 case Recursive:
		close_Switch(decision(cs));
		break;

	 case Terminal:
		free(return_phoneme(cs));
		break;

	 default:
		fprintf(stderr,"Unknown node type %i\n",type(cs));
		exit(1);
	 }
  
  free(cs);
}

void print_Switch(Switch* sw, int depth);

void print_Case(Case* cs,int depth)
{
  tabulate(depth);  
  printf(" case %s:\n", feature(cs));

  switch (type(cs))
	 {
	 case Recursive:
		print_Switch( decision(cs), depth+1);
		break;
		
	 case Terminal:
		tabulate(depth);  
		printf("  return %s\n",return_phoneme(cs));
		break;
		
	 default:
		fprintf(stderr,"Unknown node type %i\n",type(cs));
		exit(1);
	 }
}

int count_Switch(Switch* sw, int depth);

int count_Case(Case* cs, int depth)
	  /* node counter */
{
  int count=0;
  
  switch (type(cs))
	 {
	 case Recursive:
		count+= count_Switch( decision(cs), depth+1);
		break;
		
	 case Terminal:
		count=1;
		break;
		
	 default:
		fprintf(stderr,"Unknown node type %i\n",type(cs));
		exit(1);
	 }  
  return(count);
}


Switch* parse_Switch(FILE* decision_file);
/* parse a Switch statement in the decision file */


Case* parse_Case(FILE* decision_file,char* token)
	  /* Parse a Case statement in the decision file */
{
  Case* my_case=init_Case();
  char value[255];
  
  feature(my_case)= strdup(token);
  
  fscanf(decision_file," %s ",value);
  if (strcmp(value,"[")==0)
	 {
		/* This is a recursive Switch bloc */
		type(my_case)=Recursive;
		decision(my_case)= parse_Switch(decision_file);
	 }
  else
	 {
		/* This is a terminal node */
		type(my_case)=Terminal;
		return_phoneme(my_case)= strdup(value);
	 }
  return(my_case);
}

Phoneme run_Switch(Switch* sw, TestVector vector, int depth);
/* recursively run the decision tree */

Phoneme run_Case(Case* cs, TestVector vector, int depth)
	  /* Run the decision tree */
{
  if (type(cs)==Terminal)
	 {
		if (trace)
		  printf(" -> %s\n", return_phoneme(cs));
		return return_phoneme(cs) ;
	 }
  
  else
	 return run_Switch( decision(cs), vector, depth+1);
}


Switch* init_Switch()
{
  Switch* my_switch= (Switch*) malloc( sizeof(Switch) );

  nb_case(my_switch)=0;
  cases(my_switch)=NULL;
  default_phoneme(my_switch)=NULL;
  feature_index(my_switch)=-1;
  return(my_switch);
}

void close_Switch(Switch* sw)
	  /* Release the memory */
{
  int i;
  
  for(i=0; i<nb_case(sw); i++)
	 close_Case( cases(sw)[i] );

  if (default_phoneme(sw))
	 free(default_phoneme(sw));
  
  free(sw);
}

void print_Switch(Switch* sw, int depth)
{
  int i;
  
  tabulate(depth);  
  printf("switch (%i) {\n", feature_index(sw));

  tabulate(depth);  
  printf("/* %i %i %i */\n", feature_index(sw),depth,nb_case(sw));

  if (depth>=max_depth)
	 { /* well do go deeper and print the default value */
	 } 
  else
	 {
		for(i=0; i<nb_case(sw); i++)
		  print_Case(cases(sw)[i], depth);
	 }
  
  tabulate(depth);  
  printf(" default: return %s\n",default_phoneme(sw));
  tabulate(depth);  
  printf("}\n");
}

int count_Switch(Switch* sw, int depth)
	  /* node counter */
{
  int i;
  int count=1; /* one for the default return */
  
  if (depth>=max_depth)
	 {
		/* nothing ! */
	 } 
  else
	 {
		for(i=0; i<nb_case(sw); i++)
		  count+= 1 + count_Case(cases(sw)[i], depth); /* one for each condition */
	 }
  return(count);
}

void addcase_Switch(Switch* my_switch,Case* my_case)
	  /* A a new case statement to a Switch structure */
{
  /* make some room, somehow ugly ! -> if sorted insertion, dichotomic search later :-) */
  cases(my_switch)=(Case**) realloc(cases(my_switch), 
												sizeof(Case*) * (nb_case(my_switch)+1) );
  cases(my_switch)[nb_case(my_switch)]=my_case;

  nb_case(my_switch)++;
}

Switch* parse_Switch(FILE* decision_file)
	  /* parse a Switch statement in the decision file */
{
  char token[255];
  Switch* my_switch=init_Switch();
  
  if ( fscanf(decision_file," %i ",& feature_index(my_switch)) != 1)
	 {
		fprintf(stderr,"Lack FEATURE_INDEX\n");
		exit(1);
	 }
  
  fscanf(decision_file," %s ",token);
  default_phoneme(my_switch)=strdup(token);
  
  do
	 {
		fscanf(decision_file," %s ",token);
		
		if (strcmp(token,"]")==0)
		  {
			 break;  /* leave the level */
		  }
		else
		  {
			 /* The token was a feature */
			 Case* my_case=parse_Case(decision_file,token);
			 addcase_Switch(my_switch,my_case);
		  }
	 } while (1);

  return(my_switch);
}

void parse_vector_format(char* line)
	  /* format of the learning vector */
{
  int position=0;
  int  nb_LRfeedback=0;
  int  nb_RLfeedback=0;

  while ( line[position]!=0 )
	 {
		switch (line[position]) 
		  {
		  case 'L':
			 left_grapheme++; 
			 break;
			 
		  case 'R':
			 right_grapheme++;
			 break;
			 
		  case 'T': /* target, ignore for the moment. It's mostly visual */
			 break;
			 
		  case 'P':
			 nb_RLfeedback++;
			 break;

		  case 'Q':
			 nb_LRfeedback++;
			 break;
		 
		  case 'S':
			 nb_skip++;
			 break;

		  case ' ':	  /*  Ignore blanks */
		  case 10:
		  case 12:
		  case 13:
			 break;
			
		  default:
			 fprintf(stderr,"unknown attrib *%c*\n",line[position]);
			 exit(1);
			 break;
		  }
		position++;
	 }

  /* Those flags are incompatible */
  if (nb_RLfeedback*nb_LRfeedback!=0)
	 {
		fprintf(stderr,"Can't apply LEFT and RIGHT feedback at the same time\n");
		exit(-1);
	 }

  /* Decide the feedback direction */
  if (nb_RLfeedback>nb_LRfeedback)
	 {
		left_to_right=0;
		nb_feedback=nb_RLfeedback;
	 }
  else
	 {
		left_to_right=1;
		nb_feedback=nb_LRfeedback;
	 }
}

Switch* read_Switch(char* decision_name)
	  /* Read the tree file */
{
  FILE* decision_file;
  Switch* my_decision;
  int position=0;
  char line[1024];
  
  decision_file=fopen(decision_name,"rt");
  if (!decision_file)
	 {
		fprintf(stderr,"FATAL Error, can't open %s\n",decision_name);
		exit(1);
	 }

  /* Skip copyright notice */
  do
	 {
		fgets(line,sizeof(line),decision_file);
	 }
  while (line[0]=='#');
  
  /* Read the format of the learning vector */
  parse_vector_format(line);

  /* Eat the opening braket */
  fscanf(decision_file," [ %n",&position);

  if (position!=0)
	 my_decision=parse_Switch(decision_file);
  else
	 {
		fprintf(stderr,"Lack [\n");
		exit(1);
	 }
  return(my_decision);
}

Phoneme run_Switch(Switch* sw, TestVector vector, int depth)
	  /* Run the decision tree */
{
  int i;
  
  if (depth<max_depth)
	 {
		/* Linear search in the cases... may be dichotomic later ? */
		for(i=0; i<nb_case(sw); i++)
		  {
			 if (strcmp( feature(cases(sw)[i]) , vector[feature_index(sw)]) ==0)
				{
				  if (trace)
					 { 
						if ( (depth!=0) && 
							  (trace==2))
						  printf("(%s) ",default_phoneme(sw));
						
						display_debug( feature_index(sw) );
						printf("=%s ",vector[feature_index(sw)]);
					 }
				  return( run_Case( cases(sw)[i], vector, depth) );
				}
			 
		  }
	 }
  /* default case */
  if (trace)
	 {
		if (trace==2)
		  printf("(%s) ",default_phoneme(sw));
		display_debug( feature_index(sw) );
		printf("=%s -> (%s)\n", vector[feature_index(sw)], default_phoneme(sw));
	 }
  return( default_phoneme(sw) );
}

void Evaluation(FILE* test_file, Switch* decision)
	  /* Transcribe a serie of words */
{
  int nb_word=0;
  int nb_word_success=0;
  int nb_phone=0;
  int nb_phone_success=0;
  double stat;
  char line[1024];

  while (fgets(line,sizeof(line), test_file))
	 {
		char* solution[100]; /* Solution of the dictionnary */
		int nb_solution=0;   /* number of elements in the solution */

		char str_solution[255]="";
		char str_phoneme[255]="";

		char* phoneme[100];    /* Computed solution, including trailing '-' */
		int nb_phoneme=0;

		char* guess[100];    /* Computed solution */
		
		char* grapheme[100]; /* graphemic feature, including leading and trailing '-' */
		int nb_grapheme;

		char* word; /* word to transcribe */
		Feature vector[100];
		int index=1; /* because 1st was phoneme during training */
		
		char temp[1024];
		int j,i;

		int increment, begin_at,end_at; /* to deal with RL or LR transcription */
		 

		/* First come the word graphemic */
		word= strtok(line," \n\r");
		if (!word)
		  break; /* empty line */
		
		/* Eventually build extra feature list */
		for (j=0; j<nb_skip; j++)
		  vector[index++]=strtok(NULL," \n\r");

		/*
		 * Build Solution (phonemic transcription)
		 */
		while ( (solution[nb_solution++]=strtok(NULL," \n\r")) != NULL)
		  {/* empty */
		  }
		nb_solution-=2; /* the last one was NULL */
		
		/*
		 * Build Grapheme
		 */
		for(j=0; j<left_grapheme; j++)
		  grapheme[j]=strdup("-");

		temp[1]=0;

		for(nb_grapheme=0; word[nb_grapheme]!=0; nb_grapheme++,j++)
		  {
			 temp[0]=word[nb_grapheme];
			 grapheme[j]=strdup(temp);
		  }
		
		for(i=0 ; i<right_grapheme; i++,j++)
		  grapheme[j]=strdup("-");
		
		/*
		 * Build Phoneme feedback trailing '-'
		 */
		for (nb_phoneme=0; nb_phoneme< nb_feedback; nb_phoneme++)
		  phoneme[nb_phoneme]=strdup("-");
		
		/* 
		 * Let's transcribe each letter, from end to beginning !
		 */

		if (left_to_right)
		  {
			 increment=1;
			 begin_at=0;
			 end_at=nb_grapheme;
		  }
		else
		  {
			 increment=-1;
			 begin_at=nb_grapheme-1;
			 end_at=-1;
		  }
		
		/* MAIN ITERATION */
		for(j=begin_at; j!=end_at; j+=increment)
		  {
			 /* Leave the beginning untouched through the loop, contain external
			  * features such as PART_OF_SPEECH 
			  */
			 index= nb_skip+1;
			 
			 /* LLLTRRRR */
			 for(i=0; i< left_grapheme+right_grapheme+1; i++)
				vector[index++]=grapheme[i+j];

			 /* Phonemic loop */
			 for(i=0; i<nb_feedback; i++)
				vector[index++]= phoneme[nb_phoneme-1-i];

			 /* transcribe the vector ! */
			 guess[j]= run_Switch(decision,vector,0);
			 
			 if (strcmp(guess[j],solution[j])==0)
				nb_phone_success++;
			 
			 /* progress if non nul transcription */
			 if (strcmp(guess[j],"_")!=0)
				{ 
				  phoneme[nb_phoneme++]= guess[j];
				  
				  if (left_to_right)
					 {
						strcat(str_phoneme,guess[j]);
						strcat(str_phoneme," ");
					 }
				  else
					 {
						char local[100]="";
						sprintf(local,"%s %s",guess[j],str_phoneme);
						strcpy(str_phoneme,local);
					 }
				}

			 /* build the phonemic solution */
			 if (strcmp(solution[j],"_")!=0)
				{ 
				  if (left_to_right)
					 {
						strcat(str_solution,solution[j]);
						strcat(str_solution," ");
					 }
				  else
					 {
						char local[100]="";
						sprintf(local,"%s %s",solution[j],str_solution);
						strcpy(str_solution,local);
					 }
				}
			 nb_phone++;
		  }

		/* Number of words correctly transcribed */
		if (strcmp(str_solution, str_phoneme)==0)
		  {
			 nb_word_success++;
			 if ( (modulo>0) &&
					( (nb_word % modulo)==0))
				printf("Good\n%s %s\n",word,str_solution);
		  }
		else
		  {
			 printf("Wrong "); 
			 printf("%s \n",word);
			 if (nb_skip!=0)
				printf("%s\n",vector[1]);
			 printf("%s\n%s\n",str_phoneme,str_solution);
		  }
		nb_word++;
	 }

  stat= 100.0 * (double) nb_word_success / (double)nb_word;
  fprintf(stdout,"Read %i words, %i success (%f/100)\n",nb_word,nb_word_success,stat);

  stat= 100.0 * (double) nb_phone_success / (double)nb_phone;
  fprintf(stdout,"Read %i letters, %i success (%f/100)\n",nb_phone,nb_phone_success,stat);

  fprintf(stdout,"Nbnode=%i\n", count_Switch(decision,0));
}

int main(int argc, char **argv)
{
  char* decision_name;
  char* test_name;
  FILE* test_file;
  Switch* decision;
  
  if (argc==1)
	 {
		fprintf(stderr,ProgInfo);
		fprintf(stderr,"USAGE: %s decision.tree test [max_depth] [modulo] [trace]\n",argv[0]);
		return(1);
	 }

  decision_name=argv[1];
  
  if (argc>=4)
	 max_depth=atoi(argv[3]);
  else
	 max_depth=320000;

  if (argc==5)
	 modulo=atoi(argv[4]);
  else
	 modulo=-1;

  /* Display the decision's path */
  trace=0;
  if (argc==6)
	 trace=  atoi(argv[5]);

  decision= read_Switch(decision_name);

  if (argc==2)
	 {
		print_Switch(decision,0);
		return 0;
	 }

  test_name=argv[2];
  if (strcmp(test_name,"-")==0)
	 {
		test_file=stdin;
	 }
  else
	 {
		if (!(test_file= fopen(test_name,"r")))
		  {
			 fprintf(stderr,"Can't open %s\n",test_name);
			 exit(1);
		  }
	 }
  
  Evaluation(test_file,decision);

  /* Destructor */
  close_Switch(decision);
  return 0;
}
