/*
 * File:    dtw.hpp
 * Purpose: Dynamic Time Warping alignement between letters/phoneme strings 
 *          (use an affinity matrix between phoneme | letters^n )
 * Author:  Vincent Pagel ( pagel@tcts.fpms.ac.be )
 * Version : 0.99
 * Time-stamp: <1999-03-03 21:25:06 pagel>
 * 
 * Copyright (c) 1999 Faculte Polytechnique de Mons (TCTS lab)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 1
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * 19/02/98: Created
 * 02/03/99: P(phoneme| letter^n + extra context letter)
 */

#ifndef DTW_HPP
#define DTW_HPP

#include "Affinity.hpp"

#define MAX_SIZE 50 // Size of the big DTW matrix ... ugly but quick and safe implementation

class Dtw
{
public:
  Dtw(istream& affinity_in, bool back=false, double thrs=1, bool is_debug=false): affin(affinity_in), threshold(thrs), back_off(back), debug(is_debug), dummy_char_string("*")
	 {
	 }
  
  // Compute the best alignment path between Letters and Phonemes 
  // from the probablity affinity matrix by Dynamic Time Warping
  void align(const char* head_word, const char* phonet, const string& tags);
  
protected:

  // Probability of phoneme knowing 'letters' (which is one or more letters) + the next letter
  double probability(const string& letters, const char context, const string& phone);

  // Probability of phoneme knowing 'letter' + the next letter
  double probability(const char, const char context, const string&);

  // Back-off probability of phoneme knowing 'letter'
  double back_probability(const char letter, const string& phone);
  
  // Members
  Affinity affin;	  // Letters phoneme affinity matrix
  double threshold; // is there a threshold on the alignment score to reject lower end?
  bool back_off;   //  use only 1 letter| 1phoneme proba (during initialization)
  bool debug;       // Dump DTW search

  // Begin UGLY DON'T TOUCH
  double best_cumul[MAX_SIZE][MAX_SIZE]; // cumulated weights
  string best_path[MAX_SIZE][MAX_SIZE]; // best alignment path to go up there
  
  // a preformed string to avoid allocating strings each time we want 
  // to compute the probability of a char
  string dummy_char_string; 
};

#endif
