package info.magnolia.projects.lm.commands;

import info.magnolia.cms.util.QueryUtil;
import info.magnolia.commands.MgnlCommand;
import info.magnolia.context.Context;
import info.magnolia.context.MgnlContext;
import info.magnolia.jcr.util.NodeTypes;
import info.magnolia.jcr.util.NodeUtil;
import info.magnolia.jcr.util.PropertyUtil;
import info.magnolia.projects.lm.model.Index;
import info.magnolia.projects.lm.model.VolksfestData;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;

@SuppressWarnings("unused")
public class EventImporterCommand extends MgnlCommand {

    private static final Logger log = LoggerFactory.getLogger(EventImporterCommand.class);

    public static final String EVENT_PREFIX = "event-";
    public static final String EVENT_OVERVIEW_TEMPLATE = "linux-magazin-website:pages/lmEventsOverview";
    public static final String EVENT_TEMPLATE = "linux-magazin-website:pages/lmEvent";

    private static final String[] dateFormats = new String[]{"dd.MM.yyyy", "yyyy-MM-dd", "MM/dd/yyyy"};

    private Session session;

    private String repository;
    private String contentRepository;
    private String path;
    private String importURL;
    private String contentPath;
    private boolean importContent;

    /**
     * Ausfuehrungs-Methode des Magnolia Commands zum Import von Berliner Volksfesten
     *
     * @param context Magnolia Context
     * @return Ergebnis der Import-Methode (true/false)
     */
    @Override
    public boolean execute(Context context) throws Exception {
        log.debug("Starte Event Import... ");

        return importEvents();
    }

    /**
     * Import von Berliner Volksfest Daten
     * Legt fuer jedes nicht vorhandene Event eine neue Seite an
     * sowie optinal Daten fuer die Events Content App
     *
     * @return true bei Erfolg, ansonsten false
     */
    protected boolean importEvents() {
        boolean importResult = true;

        try {
            // Start des Daten-Imports
            InputStream input = new URL(importURL).openStream();
            BufferedReader br = new BufferedReader(new InputStreamReader(input, "UTF-8"));

            Gson gson = new Gson();

            VolksfestData data = gson.fromJson(br, VolksfestData.class);

            // Ende des eigentlichen Daten-Imports, importierte Events werden verarbeitet
            for (Index event : data.getIndex()) {
                if (validEvent(event)) {
                    try {
                        // Erstellung einer Seite im Website-Tree, falls das Event gueltige Daten hat und noch nicht
                        // vorhanden ist
                        session = MgnlContext.getJCRSession(repository);

                        Node parentNode = NodeUtil.createPath(session.getRootNode(),
                                path, NodeTypes.Page.NAME, false);
                        PropertyUtil.setProperty(parentNode, "mgnl:template", EVENT_OVERVIEW_TEMPLATE);
                        PropertyUtil.setProperty(parentNode, "title", "Berlin Events");

                        if (!eventExists(parentNode.getPath(), event.getId(), repository)) {
                            createEventNode(parentNode, event, NodeTypes.Page.NAME);
                        }

                        // Dieser Teil ist optional und erstellt Daten in der Events Content App
                        if (importContent) {
                            session = MgnlContext.getJCRSession(contentRepository);
                            Node contentNode = NodeUtil.createPath(session.getRootNode(),
                                    contentPath, NodeTypes.Folder.NAME, false);

                            if (!eventExists(contentNode.getPath(), event.getId(), contentRepository)) {
                                createEventNode(contentNode, event, NodeTypes.Content.NAME);
                            }
                        }

                        session.save();
                    } catch (Exception e) {
                        log.error("Problem beim erstellen von Content fuer ein Event: " + e.getMessage());
                    }
                } else {
                    log.error("Fehlende Pflichtfelder bei Event (kein Import):");
                    log.error(event.toString());
                }
            }

        } catch (Exception e) {
            log.error("Problem beim Import von Events:" + e.getMessage());
            importResult = false;
        }

        log.debug("Event-Import beendet.");

        return importResult;
    }

    /**
     * Prueft, ob beim gelieferten Event alle Pflichtfelder gefuellt wurden
     *
     * @param event Einzelnes Event aus REST-Import
     * @return true falls Event gueltig, ansonsten false
     */
    protected boolean validEvent(Index event) {

        return !(StringUtils.isEmpty(event.getId()) || StringUtils.isEmpty(event.getBezeichnung())
                || StringUtils.isEmpty(event.getVon()) || StringUtils.isEmpty(event.getBis())
                || StringUtils.isEmpty(event.getStrasse()) || StringUtils.isEmpty(event.getBezirk()));
    }

    /**
     * Baut die Event-Location aus Strasse und PLZ zusammen
     *
     * @param event Einzelnes Event aus REST-Import
     * @return String mit der Event-Location
     */
    protected String getLocation(Index event) {
        String location = event.getStrasse();
        if (StringUtils.isNotEmpty(event.getPlz())) {
            location += " (PLZ: " + event.getPlz() + ")";
        }
        return location;
    }

    /**
     * Erstellt eine Event Page oder ein Event fuer die Event Content App
     *
     * @param parentNode Parent-Knoten im Repository fuer das neue Event
     * @param event Einzelnes Event aus REST-Import
     * @param eventType Definiert den Typ des Events (Webseite oder Content)
     */
    protected void createEventNode(Node parentNode, Index event, String eventType) {
        try {
            Node eventNode = NodeUtil.createPath(parentNode, EVENT_PREFIX + event.getId(), eventType, false);
            if (StringUtils.equals(NodeTypes.Page.NAME, eventType)) {
                PropertyUtil.setProperty(eventNode, "mgnl:template", EVENT_TEMPLATE);
                PropertyUtil.setProperty(eventNode, "abstract", event.getBezeichnung());
            }
            PropertyUtil.setProperty(eventNode, "id", event.getId());
            PropertyUtil.setProperty(eventNode, "title", event.getBezeichnung());
            PropertyUtil.setProperty(eventNode, "eventTitle", event.getBezeichnung());
            PropertyUtil.setProperty(eventNode, "date", DateUtils.parseDate(event.getVon(), dateFormats));
            PropertyUtil.setProperty(eventNode, "dateEnd", DateUtils.parseDate(event.getVon(), dateFormats));
            PropertyUtil.setProperty(eventNode, "zeit", StringUtils.replace(event.getZeit(), "\n", " "));
            PropertyUtil.setProperty(eventNode, "location", getLocation(event));
            PropertyUtil.setProperty(eventNode, "bezirk", event.getBezirk());

            PropertyUtil.setProperty(eventNode, "link", event.getWww());
            PropertyUtil.setProperty(eventNode, "veranstalter", event.getVeranstalter());
            PropertyUtil.setProperty(eventNode, "mail", event.getMail());
            PropertyUtil.setProperty(eventNode, "bemerkungen", event.getBemerkungen());
            session.save();
        } catch (Exception e) {
            log.error("Event konnte nicht erstellt werden, Event ID " + event.getId() + ": " + e.getMessage());
        }
    }

    /**
     * Prueft, ob ein Event bereits existiert
     *
     * @param eventPath Repository-Pfad, in dem das Event gesucht werden soll
     * @param eventId Eindeutige ID des Events aus dem REST-Import
     * @param repo Repository, in dem gesucht werden soll
     */
    protected boolean eventExists(String eventPath, String eventId, String repo) {
        boolean found = false;

        String sql = "select * from [nt:base] as t where ISDESCENDANTNODE([" + eventPath + "]) " +
                "and (t.id='" + eventId + "')";

        try {
            NodeIterator posts = QueryUtil.search(repo, sql);
            if (posts.hasNext() && posts.nextNode() != null) {
                found = true;
            }
        } catch (RepositoryException e) {
            log.error("Problem bei der Suche nach einem Event: " + e.getMessage());
        }

        return found;
    }

    // Getter und Setter fuer die Konfigurationsparameter des Import Event Commands

    public String getRepository() {
        return repository;
    }

    public void setRepository(String repository) {
        this.repository = repository;
    }

    public String getContentRepository() {
        return contentRepository;
    }

    public void setContentRepository(String contentRepository) {
        this.contentRepository = contentRepository;
    }

    public String getPath() {
        return path;
    }

    public void setPath(String path) {
        this.path = path;
    }

    public String getImportURL() {
        return importURL;
    }

    public void setImportURL(String importURL) {
        this.importURL = importURL;
    }

    public String getContentPath() {
        return contentPath;
    }

    public void setContentPath(String contentPath) {
        this.contentPath = contentPath;
    }

    public boolean isImportContent() {
        return importContent;
    }

    public void setImportContent(boolean importContent) {
        this.importContent = importContent;
    }
}
