// Project: WP7Blocks

using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Input.Touch;
using Microsoft.Xna.Framework.Media;

namespace WP7Blocks
{
	/// <summary>
	/// Simple game with just a background image and blocks falling down.
	/// </summary>
	public class BlocksGame : Microsoft.Xna.Framework.Game
	{
		#region Variables
		GraphicsDeviceManager graphics;
		SpriteBatch spriteBatch;
		Texture2D background;
		Texture2D block;
		// Define a field of 9x9 blocks
		const int FieldWidth = 9, FieldHeight = 9;
		bool[,] field = new bool[FieldWidth, FieldHeight];
		int currentBlockX = 4, currentBlockY = 0;
		// Update 4 times a second.
		const double UpdateTime = 0.25f;
		double nextUpdateTime = 0.0f;
		#endregion

		#region Constructor
		/// <summary>
		/// Create blocks game
		/// </summary>
		public BlocksGame()
		{
			graphics = new GraphicsDeviceManager(this);
			Content.RootDirectory = "Content";

			// Frame rate is 30 fps by default for Windows Phone.
			TargetElapsedTime = TimeSpan.FromTicks(333333);
		} // BlocksGame()
		#endregion

		#region LoadContent
		/// <summary>
		/// LoadContent will be called once per game and is the place to load
		/// all of your content.
		/// </summary>
		protected override void LoadContent()
		{
			// Create a new SpriteBatch, which can be used to draw textures.
			spriteBatch = new SpriteBatch(GraphicsDevice);

			background = Content.Load<Texture2D>("SpaceBackground");
			block = Content.Load<Texture2D>("Block");
		} // LoadContent()
		#endregion

		#region Update
		/// <summary>
		/// Allows the game to run logic such as updating the world,
		/// checking for collisions, gathering input, and playing audio.
		/// </summary>
		/// <param name="gameTime">Provides a snapshot of timing values.</param>
		protected override void Update(GameTime gameTime)
		{
			// Allows the game to exit
			if (GamePad.GetState(PlayerIndex.One).Buttons.Back == ButtonState.Pressed)
				this.Exit();

			// Move left right depending on where the user touched the screen
			foreach (TouchLocation touchPos in TouchPanel.GetState())
			{
				if (touchPos.State == TouchLocationState.Released)
				{
					// Move block left or right
					if (touchPos.Position.X < 350 &&
						currentBlockX > 0)
					{
						currentBlockX--;
					}
					else if (touchPos.Position.X > 450 &&
						currentBlockX < FieldWidth - 1)
					{
						currentBlockX++;
					}
				} // if
			} // foreach

			// Update current block and create a new one if we reached the ground
			nextUpdateTime += gameTime.ElapsedGameTime.TotalSeconds;
			// Only handle this every 0.25 seconds.
			if (nextUpdateTime > UpdateTime)
			{
				nextUpdateTime -= UpdateTime;
				// Check if we can move down
				if (currentBlockY < FieldHeight - 1 &&
					field[currentBlockX, currentBlockY + 1] == false)
				{
					// Move block one down.
					currentBlockY++;
				}
				// Otherwise fix this block and create a new one at the top
				else
				{
					// Fix this block
					field[currentBlockX, currentBlockY] = true;

					// Quick check if the bottom line is now full
					int blocksInLine = 0;
					for (int x = 0; x < FieldWidth; x++)
						if (field[x, FieldHeight - 1])
							blocksInLine++;
					if (blocksInLine == FieldWidth)
					{
						// Move everything one down (we have to do it upside down)!
						for (int y = FieldHeight - 1; y > 0; y--)
							for (int x = 0; x < FieldWidth; x++)
								field[x, y] = field[x, y - 1];
					}

					// Create a new block at the top
					currentBlockX = 4;
					currentBlockY = 0;
				} // else
			} // if

			base.Update(gameTime);
		} // Update(gameTime)
		#endregion

		#region GetBlockRect
		/// <summary>
		/// Get block position and rectangle for the current block or a field block
		/// </summary>
		private Rectangle GetBlockRect(int x, int y)
		{
			// Place blocks in center (800 pixels width - 480 pixels height) / 2
			// Note: Reduced to 477 to make 9*53 fit into 477 (480 would have spaces)
			return new Rectangle(162 + x * 477 / FieldWidth, y * 477 / FieldHeight,
				477 / FieldWidth, 477 / FieldHeight);
		} // GetBlockRect(x, y)
		#endregion

		#region Draw
		/// <summary>
		/// This is called when the game should draw itself.
		/// </summary>
		/// <param name="gameTime">Provides a snapshot of timing values.</param>
		protected override void Draw(GameTime gameTime)
		{
			// Clear background in black
			GraphicsDevice.Clear(Color.Black);

			spriteBatch.Begin();
			// Place background in center (800 pixels width - 512 pixels) / 2 = 144
			spriteBatch.Draw(background, new Vector2(144, 0), Color.White);
			// Draw all field blocks
			for (int x = 0; x < FieldWidth; x++)
				for (int y = 0; y < FieldHeight; y++)
					if (field[x,y])
						spriteBatch.Draw(block, GetBlockRect(x, y), Color.DarkGray);
			// And finally also add the current block
			spriteBatch.Draw(block, GetBlockRect(currentBlockX, currentBlockY),
				Color.White);
			spriteBatch.End();

			base.Draw(gameTime);
		} // Draw(gameTime)
		#endregion
	} // class BlocksGame
} // namespace WP7Blocks
