# Copyright (c) 2004 by Intevation GmbH
# Authors:
# Jan-Oliver Wagnber <jan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

__version__ = "$Revision: 1.1 $"
# $Source: /thubanrepository/thuban/Thuban/UI/colordialog.py,v $
# $Id: colordialog.py,v 1.1 2004/05/07 20:12:16 jan Exp $

from wxPython.wx import wxDialog, wxColourDialog, wxID_OK, wxID_CANCEL, \
                        wxBoxSizer, wxVERTICAL, wxHORIZONTAL, wxALL, \
                        wxALIGN_CENTER_HORIZONTAL, wxButton, EVT_BUTTON, \
                        wxInitAllImageHandlers

from Thuban import _

from Thuban.UI.common import Color2wxColour, wxColour2Color

from Thuban.Model.color import Transparent


# determine whether the pyColourChooserDialog is available
# It was not available with the very first versions of wxWindows 2.4
# (I don't know the exact version though)
try:
    from wxPython.lib.colourchooser import wxPyColourChooser
    _wxPyColourChooser = True
    wxInitAllImageHandlers() # should be somewhere at Thuban startup?
except:
    _wxPyColourChooser = False


class PyColorChooserDialog(wxDialog):
    """
    A Dialog that uses the wxPyColourChooser Frame and simply
    adds OK and Cancel button to form a modal color selection dialog.
    """

    def __init__(self, parent):
        wxDialog.__init__(self, parent, -1, _("Select Color"))

        self.parent = parent
        self.dialog_layout()

    def dialog_layout(self):
        top_box = wxBoxSizer(wxVERTICAL)

        self.chooser = wxPyColourChooser(self, -1)

        top_box.Add(self.chooser, 1, wxALL | wxALIGN_CENTER_HORIZONTAL, 5)

        box = wxBoxSizer(wxHORIZONTAL)
        box.Add(wxButton(self, wxID_OK, _("OK")), 0, wxALL, 4) 
        box.Add(wxButton(self, wxID_CANCEL, _("Cancel")), 0, wxALL, 4) 
        top_box.Add(box, 0, wxALIGN_CENTER_HORIZONTAL|wxALL, 10)

        EVT_BUTTON(self, wxID_OK, self.OnOK)
        EVT_BUTTON(self, wxID_CANCEL, self.OnCancel)

        self.SetAutoLayout(True)
        self.SetSizer(top_box)
        top_box.Fit(self)
        top_box.SetSizeHints(self)

    def GetValue(self):
        return self.chooser.GetValue()

    def SetValue(self, color):
        return self.chooser.SetValue(color)

    def OnOK(self, event):
        self.EndModal(wxID_OK)

    def OnCancel(self, event):
        self.EndModal(wxID_CANCEL)


class ColorDialog:
    """
    The color dialog raises one of the available color selection
    dialogs. ColorDialog is no derived from any GUI class though.

    Furthermore, wxColour class is mapped to the Thuban
    Color class already through this class.

    Eventually it should be configurable globally for Thuban
    which color dialog to use since they might differ in
    functionality, translation or stability.
    """

    def __init__(self, parent):
        if _wxPyColourChooser:
            self.dlg = PyColorChooserDialog(parent)
        else:
            self.dlg = wxColourDialog(parent)

    def GetColor(self):
        if _wxPyColourChooser:
            return wxColour2Color(self.dlg.GetValue())
        else:
            return wxColour2Color(self.dlg.GetColourData().GetColour())

    def SetColor(self, color):
        if color is not Transparent:
            if _wxPyColourChooser:
                self.dlg.SetValue(Color2wxColour(color))
            else:
                self.dlg.GetColourData().SetColour(Color2wxColour(color))
        
    def ShowModal(self):
        return self.dlg.ShowModal()

    def Destroy(self):
        return self.dlg.Destroy()


if __name__ == "__main__":
    # Test routine to run the dialog without Thuban

    from wxPython.wx import wxApp, NULL

    wxInitAllImageHandlers()

    class _TestApp(wxApp):
        def OnInit(self):
            dialog = ColorDialog(NULL)

            if dialog.ShowModal() == wxID_OK:
                print "Selected color:", dialog.GetColor()
            else:
                print "No color selected"

            dialog.Destroy()
            return True

    app = _TestApp()
    app.MainLoop()
