# Copyright (c) 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test the Table classes

These tests test the old table interface. They're only here for some
intermediate backwards compatibility testing. The tests for the new
interfaces are in test_dbf_table.py and test_memory_table.py.
"""

__version__ = "$Revision: 1.6 $"
# $Source: /thubanrepository/thuban/test/test_table.py,v $
# $Id: test_table.py,v 1.6 2003/07/10 14:56:38 jonathan Exp $

import os
import unittest

# Ignore the deprecation warnings for the old table interface. Testing
# the old interface is the purpose of this module, after all, so all
# calls are intended
import warnings
warnings.filterwarnings("ignore", ".*old table interface.*",
                        DeprecationWarning, "test_table")

import support
support.initthuban()

from Thuban.Model.table import DBFTable as Table, MemoryTable, \
     FIELDTYPE_STRING, FIELDTYPE_INT, FIELDTYPE_DOUBLE
import dbflib

class TableTest(unittest.TestCase, support.FileTestMixin):

    def checkColumn(self, col, data):
        self.assertEquals((col.type, col.name, col.width, col.prec), data)

    def test_read(self):
        """Reading from DBF tables"""
        eq = self.assertEquals
        table = Table(os.path.join("..", "Data", "iceland", "political.dbf"))
        eq(table.NumRows(), 156)
        eq(table.NumColumns(), 8)

        # Check one each of the possible field types
        self.checkColumn(table.Column(0), ('double', 'AREA', 12, 3))
        self.checkColumn(table.Column(3), ('int', 'PONET_ID', 11, 0))
        self.checkColumn(table.Column(6), ('string', 'POPYCOUN', 2, 0))

        # Read an `interesting` record
        eq(table.ReadRowAsDict(144),
           {'POPYCOUN': 'IC', 'POPYADMIN': '', 'PONET_': 146, 'AREA': 19.462,
            'POPYTYPE': 1, 'PERIMETER': 88.518000000000001, 'POPYREG': '1',
            'PONET_ID': 145})

        table.Destroy()

    def test_write(self):
        """Writing to DBF tables"""
        eq = self.assertEquals

        # First create a DBF file
        dbffilename = self.temp_file_name("table_write.dbf")
        dbf = dbflib.create(dbffilename)
        dbf.add_field("NAME", dbflib.FTString, 20, 0)
        dbf.add_field("INT", dbflib.FTInteger, 10, 0)
        dbf.add_field("FLOAT", dbflib.FTDouble, 10, 4)
        dbf.write_record(0, {'NAME': "Weatherwax", "INT":1,
                             "FLOAT":3.1415926535})
        dbf.close()

        # Create the table
        table = Table(dbffilename)
        record = table.ReadRowAsDict(0)
        # The FLOAT value is different from above because of rounding
        eq(record, {'NAME': "Weatherwax", "INT":1, "FLOAT":3.1415999999999999})

        # change only one field
        table.write_record(0, {"NAME": "Ogg"})
        # check whether it has been written immediately
        dbf = dbflib.DBFFile(dbffilename)
        control = dbf.read_record(0)
        eq(control, {'NAME': "Ogg", "INT":1, "FLOAT":3.1415999999999999})
        dbf.close()

        # check whether the table itself returns the new value
        eq(table.ReadRowAsDict(0),
           {'NAME': "Ogg", "INT":1, "FLOAT":3.1415999999999999})

        # Check whether we can specify the record as a tuple
        table.write_record(0, ("Garlick", 2, 1.5))
        eq(table.ReadRowAsDict(0), {"NAME": "Garlick", "INT": 2, "FLOAT": 1.5})

        table.Destroy()

class MemoryTableTest(unittest.TestCase, support.FileTestMixin):

    def checkColumn(self, col, data):
        self.assertEquals((col.type, col.name), data)

    def setUp(self):
        """Create a small table we will work on."""
        self.table = MemoryTable([("type", FIELDTYPE_STRING),
                                  ("height", FIELDTYPE_DOUBLE),
                                  ("code", FIELDTYPE_INT)],
                                 [("OTHER/UNKNOWN", 200.22, 0),
                                  ("RUINS", 300.33, 1),
                                  ("FARM", 400.44, 2),
                                  ("BUILDING", 500.55, 3),
                                  ("HUT", 600.66, 4),
                                  ("LIGHTHOUSE", 700.77, 5)])

    def test_read(self):
        """Reading from MemoryTables"""
        eq = self.assertEquals
        table = self.table

        eq(table.NumRows(), 6)
        eq(table.NumColumns(), 3)

        # Check one each of the possible field types
        self.checkColumn(table.Column(0), ('string', 'type'))
        self.checkColumn(table.Column(1), ('double', 'height'))
        self.checkColumn(table.Column(2), ('int', 'code'))

        # Read an `interesting` record
        eq(table.ReadRowAsDict(3),
           {'type': 'BUILDING', 'height': 500.55, 'code': 3})

    def test_write(self):
        """Writing to a MemoryTable"""
        eq = self.assertEquals
        table = self.table

        # change only one field
        # TODO: acticate when implemented
        # table.write_record(2, {"type": "FARMS"})

        # check whether the table returns the new value
        # TODO: acticate when implemented
        #eq(table.read_record(2),
        #   {'type': "FARMS", "height": 400.44, "code": 2})

        # Check whether we can specify the record as a tuple
        table.write_record(4, ("HUTS", 111.11, 42))
        eq(table.ReadRowAsDict(4), {"type": "HUTS", "height": 111.11, "code": 42})


if __name__ == "__main__":
    unittest.main()
