# Copyright (c) 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test the Thuban-specific Projection class
"""

__version__ = "$Revision: 1.11 $"
# $Source: /thubanrepository/thuban/test/test_proj.py,v $
# $Id: test_proj.py,v 1.11 2003/08/29 10:02:16 bh Exp $

import unittest
import os

import support
support.initthuban()

from Thuban import _
from Thuban.Model.proj import Projection, ProjFile

import Thuban.Model.resource as resource

from xmlsupport import sax_eventlist

from xml.sax import SAXParseException


class TestProjection(unittest.TestCase, support.FloatComparisonMixin):

    """Test cases for the Thuban-specific Projection class
    """

    def test(self):
        """Test Projection"""
        params = ["zone=26", "proj=utm", "ellps=clrk66"]
        proj = Projection(params)
        self.assertEquals(proj.params, params)

        # It's not clear whether this value is really the correct one
        # but a test failure here probably still means a bug somewhere
        self.assertFloatSeqEqual(proj.Forward(0, 0),
                                 [3623101.8103431347, 0.0],
                                 epsilon = 1e-5)
        self.assertFloatSeqEqual(proj.Inverse(3623101.8103431347, 0.0),
                                 [-0.00065775699878736467, 0])

        self.assertFloatSeqEqual(proj.ForwardBBox((0, 0, 2, 2)),
                                 (3620891.3077618643, 0.0,
                                  3875381.8535437919, 252962.10480170773),
                                 epsilon = 1e-5)

        # GetName()
        self.assertEquals(proj.GetName(), _("Unknown"))

        # GetParameter()
        self.assertEquals(proj.GetParameter("zone"), "26")
        self.assertEquals(proj.GetParameter("proj"), "utm")
        self.assertEquals(proj.GetParameter("ellps"), "clrk66")
        self.assertEquals(proj.GetParameter("hallo"), "")

        # GetAllParameters()
        self.assertEquals(proj.GetAllParameters(), params)

        # GetName()
        proj = Projection(params, "MyName")
        self.assertEquals(proj.GetName(), "MyName")


sample_projfile = '''\
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE projfile SYSTEM "thuban.dtd">
<projectionlist>
    <projection name="Transverse Mercartor">
        <parameter value="proj=tmerc"/>
        <parameter value="ellps=clrk66"/>
        <parameter value="lat_0=90w"/>
        <parameter value="lon_0=90w"/>
        <parameter value="k=1"/>
    </projection>
    <projection name="Transverse Mercartor">
        <parameter value="proj=tmerc"/>
        <parameter value="ellps=clrk66"/>
        <parameter value="lat_0=30w"/>
        <parameter value="lon_0=30w"/>
        <parameter value="k=1"/>
    </projection>
    <projection name="Universal Transverse Mercartor">
        <parameter value="proj=utm"/>
        <parameter value="ellps=clrk66"/>
        <parameter value="zone=1"/>
    </projection>
</projectionlist>
'''

sample_projfile_data = [("Transverse Mercartor", ["proj=tmerc",
                                                  "ellps=clrk66",
                                                  "lat_0=90w",
                                                  "lon_0=90w",
                                                  "k=1"]),
                        ("Transverse Mercartor", ["proj=tmerc",
                                                  "ellps=clrk66",
                                                  "lat_0=30w",
                                                  "lon_0=30w",
                                                  "k=1"]),
                        ("Universal Transverse Mercartor", ["proj=utm",
                                                            "ellps=clrk66",
                                                            "zone=1"])]

sample_projfile2 = '''\
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE projfile SYSTEM "thuban.dtd">
<projectionlist>
</projectionlist>
'''

sample_projfile_data2 = []

class TestProjFile(unittest.TestCase, support.FileTestMixin):

    """Test cases for reading and writing projection files.
    """

    def compare_xml(self, xml1, xml2):
        self.assertEquals(sax_eventlist(xml1), sax_eventlist(xml2))

    def test(self):
        """Test ProjFile"""

        proj0 = Projection(["proj=tmerc", "ellps=clrk66"])
        proj1 = Projection(["proj=utm", "ellps=clrk66"])
        proj2 = Projection(["proj=lcc", "ellps=clrk66",
                            "lat_1=0", "lat_2=20"])

        eq = self.assertEquals


        #
        # __init__()
        # GetFilename()
        # SetFilename()
        #
        for name in ["", "hello_world"]:
            projFile = ProjFile(name)
            eq(projFile.GetFilename(), name)

            projFile.SetFilename("XXX")
            projFile.SetFilename(name)
            eq(projFile.GetFilename(), name)

        # initial number of projections should be 0
        eq(len(projFile.GetProjections()), 0)

        #
        # Add()
        # Remove()
        #
        projFile.Add(proj0)
        eq(len(projFile.GetProjections()), 1)
        projFile.Remove(proj0)
        eq(len(projFile.GetProjections()), 0)

        # try to remove something that doesn't exist
        self.assertRaises(ValueError, projFile.Remove, None)
        self.assertRaises(ValueError, projFile.Remove, proj0)

        projFile.Add(proj0)
        projFile.Add(proj1)
        projFile.Add(proj2)
        eq(len(projFile.GetProjections()), 3)

        # GetProjections() -- tests order
        projs = projFile.GetProjections()
        eq(projs[0], proj0)
        eq(projs[1], proj1)
        eq(projs[2], proj2)

        projFile.Remove(proj2)
        projFile.Remove(proj1)

        # Replace()
        projFile.Replace(proj0, proj1)
        projs = projFile.GetProjections()
        eq(projs[0], proj1)

        # replace a non-existent projection
        self.assertRaises(ValueError, projFile.Replace, None, proj2)
        self.assertRaises(ValueError, projFile.Replace, proj0, proj2)

    def testRead(self):
        """Test read_proj_file"""

        self.doTestRead(sample_projfile_data, sample_projfile)
        self.doTestRead(sample_projfile_data2, sample_projfile2)

        #
        # file doesn't exist
        #
        self.assertRaises(IOError,
            resource.read_proj_file, self.temp_file_name("nonexistent.proj"))

        #
        # file isn't readable
        #
        filename = self.temp_file_name("projfile.proj")
        file = open(filename, "w")
        file.close()
        os.chmod(filename, 0200) # write-only
        self.assertRaises(IOError, resource.read_proj_file, filename)
        os.chmod(filename, 0600) # read/write so we reuse the file

        #
        # file has invalid XML (or none at all)
        #
        filename = self.temp_file_name("projfile.proj")
        file = open(filename, "w")
        file.close()

        self.assertRaises(SAXParseException, resource.read_proj_file, filename)

    def testWrite(self):
        """Test write_proj_file"""

        self.doTestWrite(sample_projfile_data, sample_projfile)
        self.doTestWrite(sample_projfile_data2, sample_projfile2)

    def doTestWrite(self, data, expected):

        filename = self.temp_file_name("projfile.proj")

        pf = ProjFile(filename)
        for proj in data:
            pf.Add(Projection(proj[1], proj[0]))

        resource.write_proj_file(pf)

        file = open(filename)
        written_contents = file.read()
        file.close()
        self.compare_xml(written_contents, expected)

    def doTestRead(self, data, input):

        filename = self.temp_file_name("projfile.proj")
        file = open(filename, "w")
        file.write(input)
        file.close()

        pf = resource.read_proj_file(filename)

        eq = self.assertEquals

        eq(pf.GetFilename(), filename)

        for proj, d in zip(pf.GetProjections(), data):
            eq(proj.GetName(), d[0])
            for param in proj.GetAllParameters():
                self.assert_(param in d[1])


if __name__ == "__main__":
    unittest.main()
