# Copyright (c) 2003 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
# Frank Koormann <frank.koormann@intevation.de>
# Jan-Oliver Wagner <jan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""Projection dialog"""

__version__ = "$Revision: 1.24 $"
# $Source: /thubanrepository/thuban/Thuban/UI/projdialog.py,v $
# $Id: projdialog.py,v 1.24 2003/08/12 12:34:07 bh Exp $

import os, sys, math
from wxPython.wx import *

from Thuban import _

from Thuban.Model.proj import Projection, ProjFile

from Thuban.Model.resource import get_user_proj_files, get_system_proj_files, \
                                  read_proj_file, write_proj_file
from Thuban.UI.dialogs import NonModalNonParentDialog


ID_PROJ_ADVANCED  = 4001
ID_PROJ_PROJCHOICE = 4002
ID_PROJ_ADDTOLIST    = 4003
ID_PROJ_NEW       = 4004
ID_PROJ_REVERT    = 4006
ID_PROJ_AVAIL     = 4009
ID_PROJ_SAVE      = 4010
ID_PROJ_IMPORT    = 4011
ID_PROJ_EXPORT    = 4012
ID_PROJ_REMOVE    = 4013
ID_PROJ_PROJNAME  = 4014

CLIENT_PROJ = 0
CLIENT_PROJFILE = 1

class ProjFrame(NonModalNonParentDialog):

    def __init__(self, parent, name, title, receiver):
        """Initialize the projection dialog.

        receiver -- An object that implements the following methods:
                        SetProjection(projection)
                        GetProjection()
        """
        
        self.receiver = receiver
        self.haveTried = False
        self.curProjPanel = None

        self.projPanels = []
        self.projPanels.append(
            ("tmerc", _("Transverse Mercator"), TMPanel))
        self.projPanels.append(
            ("utm", _("Universal Transverse Mercator"), UTMPanel))
        self.projPanels.append(
            ("lcc", _("Lambert Conic Conformal"), LCCPanel))
        self.projPanels.append(
            ("latlong", _("Geographic"), GeoPanel))

        NonModalNonParentDialog.__init__(self, parent, name, title)
        # originally generate by wxGlade
        self.panel_1 = wxPanel(self, -1)
        self.panel_edit = wxPanel(self, -1)
        self.label_5 = wxStaticText(self.panel_1, -1, 
                        _("Available Projections:"))

        # Projection List specific actions (Import/Export/Remove)
        self.button_import = wxButton(self.panel_1, ID_PROJ_IMPORT, 
                                      _("Import..."))
        self.button_export = wxButton(self.panel_1, ID_PROJ_EXPORT, 
                                      _("Export..."))
        self.button_remove = wxButton(self.panel_1, ID_PROJ_REMOVE, 
                                      _("Remove"))

        # The Projection List
        self.availprojs = wxListBox(self.panel_1, ID_PROJ_AVAIL, 
                                    style=wxLB_EXTENDED|wxLB_SORT)
        self.projfilepath = wxStaticText(self.panel_1, -1, "")

        # Projection Specific Entries (Name/Projection)
        self.label_2 = wxStaticText(self.panel_edit, -1, _("Name:"))
        self.projname = wxTextCtrl(self.panel_edit, ID_PROJ_PROJNAME, "")
        self.label_3 = wxStaticText(self.panel_edit, -1, _("Projection:"))
        self.projchoice = wxChoice(self.panel_edit, ID_PROJ_PROJCHOICE)

        # Projection Specific actions (New/Save/Add)
        self.button_new = wxButton(self.panel_edit, ID_PROJ_NEW, _("New"))
        self.button_add = wxButton(self.panel_edit, ID_PROJ_ADDTOLIST,
                                      _("Add to List"))
        self.button_save = wxButton(self.panel_edit, ID_PROJ_SAVE,_("Update"))

        # Main Action buttons (Try/Revert/OK/Close)
        self.button_try = wxButton(self, wxID_APPLY, _("Try"))
        self.button_revert = wxButton(self, ID_PROJ_REVERT, 
                                      _("Revert"))
        self.button_ok = wxButton(self, wxID_OK, _("OK"))
        self.button_ok.SetDefault()
        self.button_close = wxButton(self, wxID_CANCEL, 
                                     _("Close"))

        self.__set_properties()
        self.__do_layout()

        # wxGlade

        # Fill the projection choice list.
        for proj, name, clazz in self.projPanels:
            self.projchoice.Append(name, [clazz, None])

        self.originalProjection = self.receiver.GetProjection()

        self.__DoOnProjAvail()
        self.button_ok.SetFocus()
        self.availprojs.SetFocus()
        
        EVT_BUTTON(self, wxID_APPLY, self.OnApply)
        EVT_BUTTON(self, ID_PROJ_REVERT, self._OnRevert)
        EVT_BUTTON(self, wxID_OK, self.OnOK)
        EVT_BUTTON(self, wxID_CANCEL, self.OnCancel)
        EVT_CHOICE(self, ID_PROJ_PROJCHOICE, self._OnProjChoice)
        EVT_LISTBOX(self, ID_PROJ_AVAIL, self._OnProjAvail)
        EVT_BUTTON(self, ID_PROJ_IMPORT, self._OnImport)
        EVT_BUTTON(self, ID_PROJ_EXPORT, self._OnExport)
        EVT_BUTTON(self, ID_PROJ_REMOVE, self._OnRemove)

        EVT_BUTTON(self, ID_PROJ_NEW, self._OnNew)
        EVT_BUTTON(self, ID_PROJ_SAVE, self._OnSave)
        EVT_BUTTON(self, ID_PROJ_ADDTOLIST, self._OnAddToList)

        EVT_TEXT(self, ID_PROJ_PROJNAME, self._OnProjName)

    def OnApply(self, event):
        self.__SetProjection()
        self.haveTried = True

    def OnOK(self, event):
        self.__SetProjection()
        self.Close()

    def OnCancel(self, event):
        """Cancel just closes the dialog, but we call it cancel so we
        can overload the functionality of wxDialog.
        """
        self.Close()

    def _OnRevert(self, event):
        if self.haveTried:
            self.receiver.SetProjection(self.originalProjection)
            self.haveTried = False

    def _OnNew(self, event):

        # clear all selections
        sel = self.availprojs.GetSelections()
        for index in sel:
            self.availprojs.SetSelection(index, False)

        self.projname.Clear()

        # supply a projection panel if there wasn't one
        if self.curProjPanel is None:
            self.projchoice.SetSelection(0)
            self.__DoOnProjChoice()

        self.curProjPanel.Clear()

    def _OnSave(self, event):

        sel = self.availprojs.GetSelections()
        assert len(sel) == 1,  "button shouldn't be enabled"

        proj, projfile = self.availprojs.GetClientData(sel[0])

        assert proj is not None and projfile is not None

        newproj = self.__GetProjection()

        if newproj is not None:
            projfile.Replace(proj, newproj)
            try:
                write_proj_file(projfile)
            except IOError, (errno, errstr):
                self.__ShowError(projfile.GetFilename(), errstr)
            self.availprojs.SetClientData(sel[0], [newproj, projfile])

            self.__FillAvailList( selectProj = newproj.GetName())

    def _OnAddToList(self, event):

        proj = self.__GetProjection()
        if proj is not None:
            self.__usrProjFile.Add(proj)
            try:
                write_proj_file(self.__usrProjFile)
            except IOError, (errno, errstr):
                self.__ShowError(self.__userProjFile.GetFilename(), errstr)

            self.__FillAvailList( selectProj = proj.GetName())

    def _OnProjAvail(self, event):
        self.__DoOnProjAvail()

    def _OnImport(self, event):

        dlg = wxFileDialog(self, _("Import"), style = wxOPEN)

        if dlg.ShowModal() == wxID_OK:
            path = dlg.GetPath()

            try:
                projFile = read_proj_file(path)
                for proj in projFile.GetProjections():
                    self.__usrProjFile.Add(proj)
                write_proj_file(self.__usrProjFile)
            except IOError, (errno, errstr):
                self.__ShowError(dlg.GetPath(), errstr)

            self.__FillAvailList()

        dlg.Destroy()

    def _OnExport(self, event):

        sel = self.availprojs.GetSelections()
        assert len(sel) != 0, "button should be disabled"

        dlg = wxFileDialog(self, _("Export"), 
                        style = wxSAVE|wxOVERWRITE_PROMPT)

        if dlg.ShowModal() == wxID_OK:
            path = dlg.GetPath()

            projFile = ProjFile(path)

            for i in sel:
                proj = self.availprojs.GetClientData(i)[CLIENT_PROJ]
                if proj is not None:
                    projFile.Add(proj)

            try:
                write_proj_file(projFile)
            except IOError, (errno, errstr):
                self.__ShowError(dlg.GetPath(), errstr)

        dlg.Destroy()

    def _OnRemove(self, event):

        sel = self.availprojs.GetSelections()
        assert len(sel) != 0, "button should be disabled!"

        #
        # remove the items backwards so the indices don't change
        #
        sel = list(sel)
        sel.sort()
        sel.reverse()

        newselection = -1
        if len(sel) == 1:
            newselection = sel[0] - 1
            if newselection < 0:
                newselection = 0

        for i in sel:
            proj, projfile = self.availprojs.GetClientData(i)

            #
            # this could be the case if they selected <None> or
            # the currently used projection
            #
            if proj is not None and projfile is not None:
                projfile.Remove(proj)

        try:
            write_proj_file(projfile)
        except IOError, (errno, errstr):
            self.__ShowError(projfile.GetFilename(), errstr)

        self.__FillAvailList()

        #
        # this *could* produce incorrect results if the .proj files 
        # change between the last list update and this selection
        # because the list has been repopulated.
        #
        if newselection != -1 and self.availprojs.GetCount() > 0:
            self.availprojs.SetSelection(newselection)

        self.__VerifyButtons()

    def _OnProjName(self, event):
        self.__VerifyButtons()

    def __ShowError(self, filename, errstr):
        wxMessageDialog(self, 
            _("The following error occured:\n") + 
            filename + "\n" + errstr, 
            _("Error"), wxOK | wxICON_ERROR).ShowModal()

    def __VerifyButtons(self):
        """Enable or disable the appropriate buttons based on the 
        current state of the dialog.
        """

        sel = self.availprojs.GetSelections()

        self.button_import.Enable(True)
        self.button_export.Enable(True)
        self.button_save.Enable(True)
        self.button_remove.Enable(True)

        self.panel_edit.Enable(True)

        for ctrl in [self.button_import,
                     self.button_export,
                     self.button_remove,
                     self.button_save,
                     self.button_add,
                     self.projchoice,
                     self.projname,
                     self.panel_edit]:
            ctrl.Enable(True)

        if self.curProjPanel is not None:
            self.curProjPanel.Enable(True)

        if len(sel) == 0:
            self.button_import.Enable(True)
            self.button_export.Enable(False)
            self.button_remove.Enable(False)
            self.button_save.Enable(False)

        elif len(sel) == 1:

            proj, projFile = self.availprojs.GetClientData(sel[0])

            self.button_save.Enable(len(self.projname.GetValue()) > 0)
            self.button_add.Enable(len(self.projname.GetValue()) > 0)

            if proj is None:
                # <None> is selected
                for ctrl in [self.button_export,
                             self.button_remove,
                             self.button_save,
                             self.button_add,
                             self.projchoice,
                             self.projname]:
                    ctrl.Enable(False)

                if self.curProjPanel is not None:
                    self.curProjPanel.Enable(False)

            elif proj is self.originalProjection:
                self.button_remove.Enable(False)

            if projFile is None:
                self.button_save.Enable(False)

        else:
            self.panel_edit.Enable(False)

    def __DoOnProjAvail(self):

        sel = self.availprojs.GetSelections()
        if len(sel) == 1: 

            proj = self.availprojs.GetClientData(sel[0])[CLIENT_PROJ]
            projfile = self.availprojs.GetClientData(sel[0])[CLIENT_PROJFILE]

            if proj is None:
                # user selected <None>
                self.projname.Clear()
                self.projfilepath.SetLabel(_("Projection File: "))
            else:
            
                if projfile is not None:
                    self.projfilepath.SetLabel(_("Projection File: ") +
                        os.path.basename(projfile.GetFilename()))
                else:
                    # only None if the currently used projection is selected
                    self.projfilepath.SetLabel(_("Projection File: "))

                self.projname.SetValue(proj.GetName())

                myProjType = proj.GetParameter("proj")
                i = 0
                for projType, name, clazz in self.projPanels:
                    if myProjType == projType:
                        self.projchoice.SetSelection(i)
                        self.__DoOnProjChoice()

                        #
                        # self.curProjPanel should not be null
                        # after a call to __DoOnProjChoice
                        #
                        assert self.curProjPanel is not None

                        self.curProjPanel.SetProjection(proj)
                    i += 1

        self.__VerifyButtons()

    def _OnProjChoice(self, event):
        self.__DoOnProjChoice()

    def __DoOnProjChoice(self):
        """Create and layout a projection panel based on the selected
        projection type.

        This is necessary to have in seperate method since calls to
        wxChoice.SetSelection() do not trigger an event.

        At the end of this method self.curProjPanel will not be None
        if there was a item selected.
        """

        choice = self.projchoice

        sel = choice.GetSelection()
        if sel != -1:

            clazz, obj = choice.GetClientData(sel)

            if obj is None:
                obj = clazz(self.panel_edit, self.receiver)
                choice.SetClientData(sel, [clazz, obj])

            if self.curProjPanel is not None:
                self.curProjPanel.Hide()
                self.sizer_projctrls.Remove(self.curProjPanel)

            self.curProjPanel = obj
            self.curProjPanel.Show()

            self.sizer_projctrls.Add(self.curProjPanel, 1, 
                wxALL|wxEXPAND|wxADJUST_MINSIZE, 3)
            self.sizer_projctrls.Layout()
            self.Layout()
            self.topBox.SetSizeHints(self)

    def __SetProjection(self):
        """Set the receiver's projection."""

        #
        # save the original projection only once in case 
        # we try to apply several different projections
        #
        self.receiver.SetProjection(self.__GetProjection())

    def __GetProjection(self):
        """Return a suitable Projection object based on the current
        state of the dialog box selections.

        Could be None.
        """

        sel = self.availprojs.GetSelections()
        assert len(sel) < 2, "button should be disabled"


        if len(sel) == 1:
            proj = self.availprojs.GetClientData(sel[0])[CLIENT_PROJ]
            if proj is None:
                # <None> is selected
                return None

        #
        # self.curProjPanel should always contain the most 
        # relevant data for a projection
        #
        if self.curProjPanel is not None:
            return Projection(self.curProjPanel.GetParameters(),
                              self.projname.GetValue())

        return None

    def __FillAvailList(self, selectCurrent = False, selectProj = None):
        """Populate the list of available projections.
        
        selectCurrent -- if True, select the projection used by
                         the receiver, otherwise select nothing.
        selectProj    -- if set, select the projection found under the
                         specified name. This overwrites any other 
                         selection estimate.
        """

        self.availprojs.Clear()

        #
        # the list can never be empty. there will always be 
        # at least this one item
        #
        self.availprojs.Append("<None>", (None, None))

        projfile = get_system_proj_files()
        projfile = projfile[0]
        for proj in projfile.GetProjections():
            self.availprojs.Append(proj.GetName(), [proj, projfile])
        self.__sysProjFile = projfile

        projfile = get_user_proj_files()
        projfile = projfile[0]
        for proj in projfile.GetProjections():
            self.availprojs.Append(proj.GetName(), [proj, projfile])
        self.__usrProjFile = projfile

        #
        # We add the current projection to the list at last.
        # Since the list is resorted immediately a following FindString()
        # determines the index correctly.
        #
        proj = self.receiver.GetProjection()
        if proj is not None:
            proj_item = _("%s (current)") % proj.GetName()
            self.availprojs.Append(proj_item, [proj, None])
            if selectCurrent:
                self.availprojs.SetSelection(
                        self.availprojs.FindString(proj_item)
                    )
        else:
            if selectCurrent:
                self.availprojs.SetSelection(
                        self.availprojs.FindString("<None>")
                    )
        if selectProj:
            self.availprojs.SetSelection(
                        self.availprojs.FindString(selectProj)
                    )

                

    def __set_properties(self):

        #self.availprojs.SetSelection(0)
        self.projchoice.SetSelection(0)

        self.__FillAvailList(selectCurrent = True)

        self.projname.SetMaxLength(32)

    def __do_layout(self):
        # originally generated by wxGlade

        self.topBox = wxBoxSizer(wxVERTICAL)
        self.sizer_panel = wxBoxSizer(wxVERTICAL)
        sizer_6 = wxBoxSizer(wxHORIZONTAL)
        self.sizer_mainbttns = wxBoxSizer(wxHORIZONTAL)
        self.sizer_mainctrls = wxBoxSizer(wxHORIZONTAL)
        self.sizer_edit = wxStaticBoxSizer(wxStaticBox(self.panel_edit, -1, _("Edit")), wxHORIZONTAL)
        sizer_11 = wxBoxSizer(wxVERTICAL)
        self.sizer_projctrls = wxBoxSizer(wxVERTICAL)
        sizer_14 = wxBoxSizer(wxHORIZONTAL)
        sizer_13 = wxBoxSizer(wxHORIZONTAL)
        sizer_15 = wxBoxSizer(wxVERTICAL)
        sizer_15.Add(self.button_import, 0, wxALL, 4)
        sizer_15.Add(self.button_export, 0, wxALL, 4)
        sizer_15.Add(20, 20, 0, wxEXPAND, 0)
        sizer_15.Add(self.button_remove, 0, wxALL|wxALIGN_BOTTOM, 4)

        # list controls
        grid_sizer_1 = wxFlexGridSizer(3, 2, 0, 0)
        grid_sizer_1.Add(self.label_5, 0, wxLEFT|wxRIGHT|wxTOP, 4)
        grid_sizer_1.Add(20, 20, 0, wxEXPAND, 0)
        grid_sizer_1.Add(self.availprojs, 1, wxALL|wxEXPAND|wxADJUST_MINSIZE, 4)
        grid_sizer_1.Add(sizer_15, 0, wxALL|wxEXPAND, 4)
        grid_sizer_1.Add(self.projfilepath, 0, wxEXPAND|wxALL|wxADJUST_MINSIZE, 4)
        grid_sizer_1.AddGrowableRow(1)
        grid_sizer_1.AddGrowableCol(0)

        # edit controls
        sizer_13.Add(self.label_2, 0, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        sizer_13.Add(self.projname, 1, wxALL, 4)
        self.sizer_projctrls.Add(sizer_13, 0, wxEXPAND, 0)
        sizer_14.Add(self.label_3, 0, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        sizer_14.Add(self.projchoice, 1, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        self.sizer_projctrls.Add(sizer_14, 0, wxEXPAND, 0)
        self.sizer_edit.Add(self.sizer_projctrls, 1, wxEXPAND, 0)
        sizer_11.Add(self.button_new, 0, wxEXPAND|wxALL, 4)
        sizer_11.Add(self.button_add, 0, wxEXPAND|wxALL, 4)
        sizer_11.Add(20, 20, 0, wxEXPAND, 0)
        sizer_11.Add(self.button_save, 0, wxEXPAND|wxALL|wxALIGN_BOTTOM, 4)
        self.sizer_edit.Add(sizer_11, 0, wxALL|wxEXPAND, 4)

        sizer_6.Add(self.button_try, 0, wxRIGHT| wxEXPAND, 10)
        sizer_6.Add(self.button_revert, 0, wxRIGHT| wxEXPAND, 10)
        sizer_6.Add(self.button_ok, 0, wxRIGHT| wxEXPAND, 10)
        sizer_6.Add(self.button_close, 0, wxRIGHT| wxEXPAND, 10)

        self.panel_1.SetAutoLayout(1)
        self.panel_1.SetSizer(grid_sizer_1)
        grid_sizer_1.Fit(self.panel_1)
        grid_sizer_1.SetSizeHints(self.panel_1)

        self.panel_edit.SetAutoLayout(1)
        self.panel_edit.SetSizer(self.sizer_edit)
        self.sizer_edit.Fit(self.panel_edit)
        self.sizer_edit.SetSizeHints(self.panel_edit)

        self.sizer_mainctrls.Add(self.panel_1, 0, 
            wxALL|wxEXPAND|wxADJUST_MINSIZE, 0)
        self.sizer_mainctrls.Add(self.panel_edit, 1, 
            wxALL|wxEXPAND|wxADJUST_MINSIZE, 0)

        self.sizer_mainbttns.Add(sizer_6, 0, 
            wxALL|wxEXPAND|wxADJUST_MINSIZE, 10)

        self.topBox.Add(self.sizer_mainctrls, 1, wxALL|wxEXPAND, 4)
        self.topBox.Add(self.sizer_mainbttns, 0, wxALIGN_RIGHT|wxBOTTOM, 0)

        self.SetAutoLayout(1)
        self.SetSizer(self.topBox)
        self.topBox.Fit(self)
        self.topBox.SetSizeHints(self)
        self.Layout()

# end of class ProjFrame


class ProjPanel(wxPanel):
    """Base class for all projection panels."""

    def __init__(self, parent):
        wxPanel.__init__(self, parent, -1)

        self.__ellps = wxChoice(self, -1)
        self.ellpsData = [("airy"  , _("Airy")),
                          ("bessel", _("Bessel 1841")),
                          ("clrk66", _("Clarke 1866")), 
                          ("clrk80", _("Clarke 1880")),
                          ("GRS80" , _("GRS 1980 (IUGG, 1980)")),
                          ("intl"  , _("International 1909 (Hayford)")),
                          ("WGS84" , _("WGS 84"))]

        for tag, name in self.ellpsData:
            self.__ellps.Append(name, tag)

        self.__ellps.SetSelection(0)
        
    def _DoLayout(self, childPanel = None):

        panelSizer = wxBoxSizer(wxVERTICAL)

        sizer = wxBoxSizer(wxHORIZONTAL)
        sizer.Add(wxStaticText(self, -1, _("Ellipsoid:")), 0,
                                    wxALL|wxALIGN_CENTER_VERTICAL, 4)
        sizer.Add(self.__ellps, 1, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        panelSizer.Add(sizer, 0, wxALL|wxEXPAND, 4)

        if childPanel is not None:
            panelSizer.Add(childPanel, 0, wxEXPAND, 0)
            
        self.SetAutoLayout(1)
        self.SetSizer(panelSizer)
        panelSizer.Fit(self)
        panelSizer.SetSizeHints(self)
        self.Layout()

    def SetProjection(self, proj):
        if proj is not None:
            param = proj.GetParameter("ellps")
            i = 0
            for tag, name in self.ellpsData:
                if param == tag:
                    self.__ellps.SetSelection(i)
                    return # returning early!
                i += 1

        #
        # if proj is none, or the parameter couldn't be found...
        #
        self.__ellps.SetSelection(0)

    def GetParameters(self):
        return ["ellps=" + self.__ellps.GetClientData(
                                        self.__ellps.GetSelection())]


ID_TMPANEL_LAT = 4001
ID_TMPANEL_LONG = 4002
ID_TMPANEL_FASLE_EAST = 4003
ID_TMPANEL_FALSE_NORTH = 4004
ID_TMPANEL_SCALE = 4005

class UnknownProjPanel(ProjPanel):
    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self._DoLayout()

    def _DoLayout(self):
        sizer = wxBoxSizer(wxVERTICAL)

        sizer.Add(wxStaticText(self, -1,
                               _("Thuban does not know the parameters for the"
                                 " current projection and cannot display a"
                                 " configuration panel.")))

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return "Unknown"

    def SetProjection(self, proj):
        pass

    def GetParameters(self):
        return None

class TMPanel(ProjPanel):
    """Projection panel for Transverse Mercator."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.__latitude = wxTextCtrl(self, ID_TMPANEL_LAT)
        self.__longitude = wxTextCtrl(self, ID_TMPANEL_LONG)
        self.__scale = wxTextCtrl(self, ID_TMPANEL_SCALE)
        self.__falseEast = wxTextCtrl(self, ID_TMPANEL_FASLE_EAST)
        self.__falseNorth = wxTextCtrl(self, ID_TMPANEL_FALSE_NORTH)

        self._DoLayout()

    def _DoLayout(self):

        sizer = wxFlexGridSizer(4, 4, 0, 0)
        sizer.Add(wxStaticText(self, -1, _("Latitude:")), 0, wxALL, 4)
        sizer.Add(self.__latitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Easting:")), 0, wxALL, 4)
        sizer.Add(self.__falseEast, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Longitude:")), 0, wxALL, 4)
        sizer.Add(self.__longitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Northing:")), 0, wxALL, 4)
        sizer.Add(self.__falseNorth, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Scale Factor:")), 0, wxALL, 4)
        sizer.Add(self.__scale, 0, wxALL, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return _("Transverse Mercator")

    def SetProjection(self, proj):
        ProjPanel.SetProjection(self, proj)

        self.__latitude.SetValue(proj.GetParameter("lat_0"))
        self.__longitude.SetValue(proj.GetParameter("lon_0"))
        self.__falseEast.SetValue(proj.GetParameter("x_0"))
        self.__falseNorth.SetValue(proj.GetParameter("y_0"))
        self.__scale.SetValue(proj.GetParameter("k"))

        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=tmerc",
                  "lat_0=" + self.__latitude.GetValue(),
                  "lon_0=" + self.__longitude.GetValue(),
                  "x_0="   + self.__falseEast.GetValue(),
                  "y_0="   + self.__falseNorth.GetValue(),
                  "k="     + self.__scale.GetValue()]
        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        self.__latitude.Clear()
        self.__longitude.Clear()
        self.__falseEast.Clear()
        self.__falseNorth.Clear()
        self.__scale.Clear()

        ProjPanel.Clear(self)

ID_UTMPANEL_ZONE = 4001
ID_UTMPANEL_SOUTH = 4002
ID_UTMPANEL_PROP = 4003

class UTMPanel(ProjPanel):
    """Projection Panel for Universal Transverse Mercator."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.receiver = receiver

        self.__zone = wxSpinCtrl(self, ID_UTMPANEL_ZONE, "1", min=1, max=60)
        self.__propButton = wxButton(self, ID_UTMPANEL_PROP, _("Propose"))
        self.__south = wxCheckBox(self, ID_UTMPANEL_SOUTH, 
                                  _("Southern Hemisphere"))

        self._DoLayout()

        EVT_BUTTON(self, ID_UTMPANEL_PROP, self._OnPropose)

    def _DoLayout(self):

        sizer = wxBoxSizer(wxVERTICAL)
        psizer = wxBoxSizer(wxHORIZONTAL)
        psizer.Add(wxStaticText(self, -1, _("Zone:")), 0, wxALL, 4)
        psizer.Add(self.__zone, 0, wxALL, 4)
        psizer.Add(self.__propButton, 0, wxALL, 4)
        sizer.Add(psizer, 0, wxALL, 4)
        sizer.Add(self.__south, 0, wxALL, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return _("Universal Transverse Mercator")

    def SetProjection(self, proj):
        self.__zone.SetValue(int(proj.GetParameter("zone")))
        self.__south.SetValue(proj.GetParameter("south") != "")
        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=utm", "zone=" + str(self.__zone.GetValue())]
        if self.__south.IsChecked():
            params.append("south")

        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        self.__zone.SetValue(1)
        self.__south.SetValue(False)
        ProjPanel.Clear(self)

    def _OnPropose(self, event):
        """Call the propose dialog.
        If the receiver (e.g. the current map) has no bounding box,
        inform the user accordingly.
        """
        bb = self.receiver.BoundingBox()
        if bb is None:
            dlg = wxMessageDialog(self,
                    _("Can not propose: No bounding box found."),
                    _("Projection: Propose UTM Zone"),
                    wxOK | wxICON_INFORMATION)
            dlg.CenterOnParent()
            result = dlg.ShowModal()
            dlg.Destroy()
            return

        dlg = UTMProposeZoneDialog(self, self.receiver.BoundingBox())
        if dlg.ShowModal() == wxID_OK:
            self.__zone.SetValue(dlg.GetProposedZone())

class LCCPanel(ProjPanel): 
    """Projection Panel for Lambert Conic Conformal."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)
        
        self.__fspLatitude = wxTextCtrl(self, -1)
        self.__sspLatitude = wxTextCtrl(self, -1)
        self.__originLat   = wxTextCtrl(self, -1)
        self.__meridian    = wxTextCtrl(self, -1)
        self.__falseEast   = wxTextCtrl(self, -1)
        self.__falseNorth  = wxTextCtrl(self, -1)

        self._DoLayout()

    def _DoLayout(self):

        sizer = wxFlexGridSizer(6, 2, 0, 0)
        sizer.Add(wxStaticText(self, -1, 
            _("Latitude of first standard parallel:")))
        sizer.Add(self.__fspLatitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1,
            _("Latitude of second standard parallel:")))
        sizer.Add(self.__sspLatitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Central Meridian:")))
        sizer.Add(self.__meridian, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Latitude of origin:")))
        sizer.Add(self.__originLat, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Easting:")))
        sizer.Add(self.__falseEast, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Northing:")))
        sizer.Add(self.__falseNorth, 0, wxALL, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return _("Lambert Conic Conformal")
        
    def SetProjection(self, proj):
        self.__fspLatitude.SetValue(proj.GetParameter("lat_1"))
        self.__sspLatitude.SetValue(proj.GetParameter("lat_2"))
        self.__originLat.SetValue(proj.GetParameter("lat_0"))
        self.__meridian.SetValue(proj.GetParameter("lon_0"))
        self.__falseEast.SetValue(proj.GetParameter("x_0"))
        self.__falseNorth.SetValue(proj.GetParameter("y_0"))

        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=lcc", 
                  "lat_1=" + self.__fspLatitude.GetValue(),
                  "lat_2=" + self.__sspLatitude.GetValue(),
                  "lat_0=" + self.__originLat.GetValue(),
                  "lon_0=" + self.__meridian.GetValue(),
                  "x_0=" + self.__falseEast.GetValue(),
                  "y_0=" + self.__falseNorth.GetValue()]

        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        self.__fspLatitude.Clear()
        self.__sspLatitude.Clear()
        self.__originLat.Clear()
        self.__meridian.Clear()
        self.__falseEast.Clear()
        self.__falseNorth.Clear()

        ProjPanel.Clear(self)

class GeoPanel(ProjPanel): 
    """Projection Panel for a Geographic Projection."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.__choices = [(_("Degrees"), "0.017453"),
                          (_("Radians"), "1")]

        self.__scale = wxChoice(self, -1)
        for choice, value in self.__choices:
            self.__scale.Append(choice, value)

        self._DoLayout()

    def GetProjName(self):
        return _("Geographic")
        
    def SetProjection(self, proj):
        value = proj.GetParameter("to_meter")
        for i in range(len(self.__choices)):
            choice, data = self.__choices[i]
            if value == data:
                self.__scale.SetSelection(i)
        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=latlong", 
                  "to_meter=%s" % self.__scale.GetClientData(
                                  self.__scale.GetSelection())]

        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        ProjPanel.Clear(self)

    def _DoLayout(self):
        sizer = wxBoxSizer(wxHORIZONTAL)

        sizer.Add(wxStaticText(self, -1, _("Source Data is in: ")), 
                  0, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        sizer.Add(self.__scale, 1, wxEXPAND|wxALL, 4)

        self.__scale.SetSelection(0)

        ProjPanel._DoLayout(self, sizer)


ID_UTM_PROPOSE_ZONE_DIALOG_TAKE   = 4001
ID_UTM_PROPOSE_ZONE_DIALOG_CANCEL = 4002
class UTMProposeZoneDialog(wxDialog):

    """Propose a sensible Zone considering the current map extent."""

    def __init__(self, parent, (x, y, x2, y2)):
        wxDialog.__init__(self, parent, -1, _("Projection: Propose UTM Zone"),
                          wxDefaultPosition, wxSize(200, 100))
        self.parent = parent
        x = x + 180
        x2 = x2 + 180
        center = (x2 - x) / 2 + x
        self.proposedZone = int(center / 6 + 1)
        self.dialogLayout()

    def dialogLayout(self):
        topBox = wxBoxSizer(wxVERTICAL)

        textBox = wxBoxSizer(wxVERTICAL)
        textBox.Add(wxStaticText(self, -1, _("The current map extent center "
                                             "lies in UTM Zone")),
                    0, wxALIGN_CENTER|wxALL, 4)
        textBox.Add(wxStaticText(self, -1, str(self.proposedZone)),
                    0, wxALIGN_CENTER|wxALL, 4)

        topBox.Add(textBox, 1, wxEXPAND|wxALL, 4)

        buttonBox = wxBoxSizer(wxHORIZONTAL)
        buttonBox.Add(wxButton(self, ID_UTM_PROPOSE_ZONE_DIALOG_TAKE,
                      _("Take")), 0, wxALL, 4)
        buttonBox.Add(wxButton(self, ID_UTM_PROPOSE_ZONE_DIALOG_CANCEL,
                               _("Cancel")), 0, wxALL, 4)
        topBox.Add(buttonBox, 0, wxALIGN_CENTER_HORIZONTAL|wxALL, 10)
        EVT_BUTTON(self, ID_UTM_PROPOSE_ZONE_DIALOG_TAKE, self.OnTake)
        EVT_BUTTON(self, ID_UTM_PROPOSE_ZONE_DIALOG_CANCEL, self.OnCancel)

        self.SetAutoLayout(True)
        self.SetSizer(topBox)
        topBox.Fit(self)
        topBox.SetSizeHints(self)

    def OnTake(self, event):
        self.EndModal(wxID_OK)

    def OnCancel(self, event):
        self.EndModal(wxID_CANCEL)

    def GetProposedZone(self):
        return self.proposedZone
