# Copyright (c) 2001, 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
# Jonathan Coles <jonathan@intevation.de>
# Frank Koormann <frank.koormann@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""Basic rendering logic for Thuban maps

The code in this module is completely independend of wx so that it can
be tested reasonably easily and it could make it easier to write non-wx
renderers.
"""

__version__ = "$Revision: 1.3 $"
# $Source: /thubanrepository/thuban/Thuban/UI/baserenderer.py,v $
# $Id: baserenderer.py,v 1.3 2003/08/15 14:10:27 bh Exp $

import traceback

from Thuban.Model.layer import Layer, RasterLayer
from Thuban.Model.data import SHAPETYPE_ARC, SHAPETYPE_POINT
from Thuban.Model.label import ALIGN_CENTER, ALIGN_TOP, ALIGN_BOTTOM, \
     ALIGN_LEFT, ALIGN_RIGHT

import Thuban.Model.resource

if Thuban.Model.resource.has_gdal_support():
    from gdalwarp import ProjectRasterFile


class BaseRenderer:

    """Basic Renderer Infrastructure for Thuban Maps

    This class can't be used directly to render because it doesn't know
    anything about real DCs such as how to create pens or brushes. That
    functionality has to be provided by derived classes. The reason for
    this is that it makes the BaseRenderer completely independend of wx
    and thus it's quite easy to write test cases for it.
    """
    # If true the render honors the visibility flag of the layers
    honor_visibility = 1

    # Transparent brushes and pens. Derived classes should define these
    # as appropriate.
    TRANSPARENT_PEN = None
    TRANSPARENT_BRUSH = None

    def __init__(self, dc, scale, offset, resolution = 72.0,
                 honor_visibility = None):
        """Inititalize the renderer.

        dc -- the device context to render on.

        scale, offset -- the scale factor and translation to convert
                between projected coordinates and the DC coordinates

        resolution -- the assumed resolution of the DC. Used to convert
                absolute lengths like font sizes to DC coordinates. The
                defauult is 72.0

        honor_visibility -- boolean. If true, honor the visibility flag
                of the layers, otherwise draw all layers. If None (the
                default), use the renderer's default.
        """
        # resolution in pixel/inch

        self.dc = dc
        self.scale = scale
        self.offset = offset
        if honor_visibility is not None:
            self.honor_visibility = honor_visibility
        # store the resolution in pixel/point because it's more useful
        # later.
        self.resolution = resolution / 72.0

    def tools_for_property(self, prop):
        """Return a suitable pen and brush for the property

        This method must be implemented in derived classes. The return
        value should be a tuple (pen, brush).
        """
        raise NotImplementedError

    def render_map(self, map):
        """Render the map onto the DC the renderer was instantiated with

        Iterate through all layers and draw them. Layers containing
        vector data are drawn with the draw_shape_layer method, raster
        layers are drawn with draw_raster_layer. The label layer is
        drawn last with draw_label_layer.

        During execution of this method, the map is bound to self.map so
        that methods especially in derived classes have access to the
        map if necessary.
        """
        # Some method have to have access to the map so we store it in
        # self.map.
        self.map = map

        # Whether the raster layer has already been drawn. See below for
        # the optimization this is used for
        seenRaster = True

        self.dc.BeginDrawing()

        try:
            #
            # This is only a good optimization if there is only one
            # raster layer and the image covers the entire window (as
            # it currently does). We note if there is a raster layer
            # and only begin drawing layers once we have drawn it.
            # That way we avoid drawing layers that won't be seen.
            #
            if Thuban.Model.resource.has_gdal_support():
                for layer in map.Layers():
                    if isinstance(layer, RasterLayer) and layer.Visible():
                        seenRaster = False
                        break

            for layer in map.Layers():
                # if honor_visibility is true, only draw visible layers,
                # otherwise draw all layers
                if not self.honor_visibility or layer.Visible():
                    if isinstance(layer, Layer) and seenRaster:
                        self.draw_shape_layer(layer)
                    elif isinstance(layer, RasterLayer) \
                        and Thuban.Model.resource.has_gdal_support():
                        self.draw_raster_layer(layer)
                        seenRaster = True

            self.draw_label_layer(map.LabelLayer())
        finally:
            self.dc.EndDrawing()

    def draw_shape_layer(self, layer):
        """Draw the shape layer layer onto the map.

        Automatically called by render_map. Iterate through all shapes
        as indicated by self.layer_shapes() and draw them, using
        low-level renderers returned by self.low_level_renderer().
        """
        scale = self.scale
        offx, offy = self.offset

        map_proj = self.map.projection
        layer_proj = layer.projection

        brush = self.TRANSPARENT_BRUSH
        pen   = self.TRANSPARENT_PEN

        old_prop = None
        old_group = None
        lc = layer.GetClassification()
        field = layer.GetClassificationColumn()
        defaultGroup = lc.GetDefaultGroup()
        table = layer.ShapeStore().Table()

        # Determine which render function to use.
        useraw, draw_func, draw_func_param = self.low_level_renderer(layer)

        # Iterate through all shapes that have to be drawn.
        for shape in self.layer_shapes(layer):

            if field is None:
                group = defaultGroup
            else:
                record = table.ReadRowAsDict(shape.ShapeID())
                assert record is not None
                group = lc.FindGroup(record[field])

            if not group.IsVisible():
                continue

            # don't recreate new objects if they are the same as before
            if group is not old_group:
                old_group = group

                prop = group.GetProperties()

                if prop != old_prop:
                    pen, brush = self.tools_for_property(prop)

            if useraw:
                data = shape.RawData()
            else:
                data = shape.Points()
            draw_func(draw_func_param, data, pen, brush)

    def layer_shapes(self, layer):
        """Return an iterable over the shapes to be drawn from the given layer.

        The default implementation simply returns all ids in the layer.
        Override in derived classes to be more precise.
        """
        return layer.ShapeStore().AllShapes()

    def low_level_renderer(self, layer):
        """Return the low-level renderer for the layer for draw_shape_layer

        The low level renderer to be returned by this method is a tuple
        (useraw, func, param) where useraw is a boolean indicating
        whether the function uses the raw shape data, func is a callable
        object and param is passed as the first parameter to func. The
        draw_shape_layer method will call func like this:

            func(param, shapedata, pen, brush)

        where shapedata is the return value of the RawData method of the
        shape object if useraw is true or the return value of the Points
        method if it's false. pen and brush are the pen and brush to use
        to draw the shape on the dc.

        The default implementation returns one of
        self.draw_polygon_shape, self.draw_arc_shape or
        self.draw_point_shape as func and layer as param. None of the
        method use the raw shape data. Derived classes can override this
        method to return more efficient low level renderers.
        """
        shapetype = layer.ShapeType()
        if shapetype == SHAPETYPE_POINT:
            func = self.draw_point_shape
        elif shapetype == SHAPETYPE_ARC:
            func = self.draw_arc_shape
        else:
            func = self.draw_polygon_shape
        return False, func, layer

    def make_point(self, x, y):
        """Convert (x, y) to a point object.

        Derived classes must override this method.
        """
        raise NotImplementedError

    def projected_points(self, layer, points):
        """Return the projected coordinates of the points taken from layer.

        Transform all the points in the list of lists of coordinate
        pairs in points.

        The transformation applies the inverse of the layer's projection
        if any, then the map's projection if any and finally applies
        self.scale and self.offset.

        The returned list has the same structure as the one returned the
        shape's Points method.
        """
        proj = self.map.GetProjection()
        if proj is not None:
            forward = proj.Forward
        else:
            forward = None
        proj = layer.GetProjection()
        if proj is not None:
            inverse = proj.Inverse
        else:
            inverse = None
        result = []
        scale = self.scale
        offx, offy = self.offset
        make_point = self.make_point
        for part in points:
            result.append([])
            for x, y in part:
                if inverse:
                    x, y = inverse(x, y)
                if forward:
                    x, y = forward(x, y)
                result[-1].append(make_point(x * scale + offx,
                                             -y * scale + offy))
        return result

    def draw_polygon_shape(self, layer, points, pen, brush):
        """Draw a polygon shape from layer with the given brush and pen

        The shape is given by points argument which is a the return
        value of the shape's Points() method. The coordinates in the
        DC's coordinate system are determined with
        self.projected_points.
        """
        points = self.projected_points(layer, points)

        if brush is not self.TRANSPARENT_BRUSH:
            polygon = []
            for part in points:
                polygon.extend(part)

            insert_index = len(polygon)
            for part in points[:-1]:
                polygon.insert(insert_index, part[0])

            self.dc.SetBrush(brush)
            self.dc.SetPen(self.TRANSPARENT_PEN)
            self.dc.DrawPolygon(polygon)

        if pen is not self.TRANSPARENT_PEN:
            # At last draw the boundarys of the simple polygons
            self.dc.SetBrush(self.TRANSPARENT_BRUSH)
            self.dc.SetPen(pen)
            for part in points:
                self.dc.DrawLines(part)

    def draw_arc_shape(self, layer, points, pen, brush):
        """Draw an arc shape from layer with the given brush and pen

        The shape is given by points argument which is a the return
        value of the shape's Points() method. The coordinates in the
        DC's coordinate system are determined with
        self.projected_points.
        """
        points = self.projected_points(layer, points)
        self.dc.SetBrush(brush)
        self.dc.SetPen(pen)
        for part in points:
            self.dc.DrawLines(part)

    def draw_point_shape(self, layer, points, pen, brush):
        """Draw a point shape from layer with the given brush and pen

        The shape is given by points argument which is a the return
        value of the shape's Points() method. The coordinates in the
        DC's coordinate system are determined with
        self.projected_points.

        The point is drawn as a circle centered on the point.
        """
        points = self.projected_points(layer, points)
        if not points:
            return

        radius = self.resolution * 5
        self.dc.SetBrush(brush)
        self.dc.SetPen(pen)
        for part in points:
            for p in part:
                self.dc.DrawEllipse(p.x - radius, p.y - radius,
                                    2 * radius, 2 * radius)

    def draw_raster_layer(self, layer):
        """Draw the raster layer

        This implementation does the projection and scaling of the data
        as required by the layer's and map's projections and the scale
        and offset of the renderer and then hands the transformed data
        to self.draw_raster_data() which has to be implemented in
        derived classes.
        """
        offx, offy = self.offset
        width, height = self.dc.GetSizeTuple()

        in_proj = ""
        proj = layer.GetProjection()
        if proj is not None:
            for p in proj.GetAllParameters():
                in_proj += "+" + p + " "

        out_proj = ""
        proj = self.map.GetProjection()
        if proj is not None:
            for p in proj.GetAllParameters():
                out_proj += "+" + p + " "

        xmin = (0 - offx) / self.scale
        ymin = (offy - height) / self.scale
        xmax = (width - offx) / self.scale
        ymax = (offy - 0) / self.scale

        try:
            data = ProjectRasterFile(layer.GetImageFilename(),
                                     in_proj, out_proj,
                                     (xmin, ymin, xmax, ymax), "",
                                     (width, height))
        except (IOError, AttributeError, ValueError):
            # Why does this catch AttributeError and ValueError?
            # FIXME: The exception should be communicated to the user
            # better.
            traceback.print_exc()
        else:
            self.draw_raster_data(data)

    def draw_raster_data(self, data):
        """Draw the raster image in data onto the DC

        The raster image data is a string holding the data in BMP
        format. The data is exactly the size of the dc and covers it
        completely.

        This method has to be implemented by derived classes.
        """
        raise NotImplementedError

    def label_font(self):
        """Return the font object for the label layer"""
        raise NotImplementedError

    def draw_label_layer(self, layer):
        """Draw the label layer

        All labels are draw in the font returned by self.label_font().
        """
        scale = self.scale
        offx, offy = self.offset

        self.dc.SetFont(self.label_font())

        map_proj = self.map.projection
        if map_proj is not None:
            forward = map_proj.Forward
        else:
            forward = None

        for label in layer.Labels():
            x = label.x
            y = label.y
            text = label.text
            if forward:
                x, y = forward(x, y)
            x = x * scale + offx
            y = -y * scale + offy
            width, height = self.dc.GetTextExtent(text)
            if label.halign == ALIGN_LEFT:
                # nothing to be done
                pass
            elif label.halign == ALIGN_RIGHT:
                x = x - width
            elif label.halign == ALIGN_CENTER:
                x = x - width/2
            if label.valign == ALIGN_TOP:
                # nothing to be done
                pass
            elif label.valign == ALIGN_BOTTOM:
                y = y - height
            elif label.valign == ALIGN_CENTER:
                y = y - height/2
            self.dc.DrawText(text, x, y)
