# Copyright (c) 2003 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""Handle resources loaded from files such as projections"""

__version__ = "$Revision: 1.13 $"
# $Source: /thubanrepository/thuban/Thuban/Model/resource.py,v $
# $Id: resource.py,v 1.13 2003/08/01 16:13:05 bh Exp $


import os
import os.path

import Thuban
from Thuban import _

from Thuban.Lib.fileutil import get_application_dir

from Thuban.Model.xmlreader import XMLReader
from Thuban.Model.xmlwriter import XMLWriter
from Thuban.Model.proj import Projection, ProjFile
from xml.sax import SAXParseException

projdir = \
        os.path.join(Thuban.__path__[0], os.pardir, "Resources", "Projections")


PROJ_EXT = ".proj"

has_gdal_support = lambda: False
try:
    # first try to see if our extension module is there. if it 
    # wasn't even compiled, we obviously don't have gdal support.
    import gdalwarp

    # now try to import the python extension. if this doesn't
    # exist then we can't do anything.
    import gdal

    """Return True if the gdal library is available."""
    has_gdal_support = lambda: True
except ImportError:
    pass

def read_proj_file(filename):
    """Read a single .proj file and return a ProjFile object.
    
    Raises IOError if the file cannot be opened.
    Raises OSError if the file cannot be read.
    Raises SAXParseException if the file is not valid XML.
    """

    handler = ProjFileReader()
    handler.read(filename)
    return handler.GetProjFile()

def write_proj_file(pf):
    """Write a single .proj file

    Raises IOError if the file cannot be written.
    """

    saver = ProjFileSaver(pf)
    saver.write(pf.GetFilename())

def get_proj_files(dir):
    """Read all .proj files in the given directory and 
    return a list of ProjFile objects.
    """

    list = []
    try:
        dirlist = os.listdir(dir)
    except OSError:
        pass # if we can't get a directory listing just return []
    else:
        for file in filter(lambda s: s.endswith(PROJ_EXT), dirlist):
            try:
                filename = os.path.join(dir, file)
                list.append(read_proj_file(filename))
            except (OSError, IOError, SAXParseException):
                pass # just move onto the next file

    return list

def get_system_proj_files():
    """Return a list of ProjFile objects from files that are 
    supplied by Thuban.

    If no files could not be opened return a list with one
    empty projection file set to store data in the default file.
    """
    filename = os.path.join(projdir, "defaults.proj")
    try:
        return [read_proj_file(filename)]
    except (OSError, IOError, SAXParseException):
        return [ProjFile(filename)]

def get_user_proj_files():
    """Return a list of ProjFile objects from files that are user-defined.

    If no files could not be opened return a list with one
    empty projection file set to store data in the default file.
    """

    usrdir  = get_application_dir()
    filename = os.path.join(usrdir, "user.proj")
    try:
        return [read_proj_file(filename)]
    except (OSError, IOError, SAXParseException):
        return [ProjFile(filename)]

class ProjFileReader(XMLReader):

    def __init__(self):
        XMLReader.__init__(self)
        self.__pf = ProjFile("")

        XMLReader.AddDispatchers(self,
            {'projection': ("start_projection", "end_projection"),
             'parameter':  ("start_parameter", None)})
            
    def read(self, file_or_filename):
        XMLReader.read(self, file_or_filename)

        self.__pf.SetFilename(XMLReader.GetFilename(self))
        
    def start_projection(self, name, qname, attrs):
        self.params = []
        self.name = self.encode(attrs.get((None, 'name'), _("Unknown")))

    def end_projection(self, name, qname):
        self.__pf.Add(Projection(self.params, self.name))

    def start_parameter(self, name, qname, attrs):
        s = attrs.get((None, 'value'))
        s = str(s) # we can't handle unicode in proj
        self.params.append(s)

    def GetProjFile(self):
        return self.__pf

class ProjFileSaver(XMLWriter):

    def __init__(self, pf):
        XMLWriter.__init__(self)
        self.__pf = pf

    def write(self, file_or_filename):
        XMLWriter.write(self, file_or_filename)

        self.write_header("projfile", "projfile.dtd")
        self.write_projfile(self.__pf)
        self.close()

    def write_projfile(self, pf):

        self.open_element("projectionlist")

        for p in pf.GetProjections():
            self.open_element("projection", {"name": p.GetName()})

            for param in p.GetAllParameters():
                self.write_element("parameter", {"value": param})

            self.close_element("projection")

        self.close_element("projectionlist")


