# Copyright (c) 2002, 2003 by Intevation GmbH
# Authors:
# Bernhard Herzog <bh@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test saving a thuban session as XML
"""

__version__ = "$Revision: 1.14 $"
# $Source: /home/thuban/jail/thubanrepository/thuban/test/test_save.py,v $
# $Id: test_save.py,v 1.14 2003/06/20 16:10:12 bh Exp $

import os
import unittest
from StringIO import StringIO

import xml.sax
import xml.sax.handler
from xml.sax import make_parser, ErrorHandler, SAXNotRecognizedException

import xmlsupport

import support
support.initthuban()

import dbflib

from Thuban.Lib.fileutil import relative_filename
from Thuban.Model.save import XMLWriter, save_session, sort_data_stores
from Thuban.Model.session import Session
from Thuban.Model.map import Map
from Thuban.Model.layer import Layer, RasterLayer
from Thuban.Model.proj import Projection
from Thuban.Model.table import DBFTable
from Thuban.Model.transientdb import TransientJoinedTable
from Thuban.Model.data import DerivedShapeStore

from Thuban.Model.classification import ClassGroupSingleton, ClassGroupRange, \
    ClassGroupProperties

from Thuban.Model.range import Range


class XMLWriterTest(unittest.TestCase):

    def testEncode(self):
        """Test XMLWriter.encode"""
        writer = XMLWriter()
        eq = self.assertEquals

        eq(writer.encode("hello world"), "hello world")
        eq(writer.encode(unicode("hello world")), unicode("hello world"))

        eq(writer.encode("\x80\x90\xc2\x100"), 
                         "\xc2\x80\xc2\x90\xc3\x82\x100")
        eq(writer.encode(u"\x80\x90\xc2\x100"), 
                         "\xc2\x80\xc2\x90\xc3\x82\x100")
        eq(writer.encode(u"\xFF5E"), "\xc3\xbf5E")

        eq(writer.encode('&"\'<>'), "&amp;&quot;&apos;&lt;&gt;")
        eq(writer.encode(unicode('&"\'<>')), "&amp;&quot;&apos;&lt;&gt;")

class SaveSessionTest(unittest.TestCase, support.FileTestMixin,
                      xmlsupport.ValidationTest):

    dtd = "http://thuban.intevation.org/dtds/thuban-0.8.dtd"
    thubanids = [((dtd, n), (None, "id")) for n in
                 ["fileshapesource", "filetable", "jointable",
                  "derivedshapesource"]]
    thubanidrefs = [((dtd, n), (None, m)) for n, m in
                    [("layer", "shapestore"),
                     ("jointable", "left"),
                     ("jointable", "right"),
                     ("derivedshapesource", "table"),
                     ("derivedshapesource", "shapesource")]]
    del n, m, dtd

    def compare_xml(self, xml1, xml2):
        if 0:
            for a, b in zip(sax_eventlist(xml1, self.thubanids,
                                        self.thubanidrefs),
                          sax_eventlist(xml2, self.thubanids,
                                        self.thubanidrefs)):
              print a == b and ' ' or '*****'
              print a
              print b
        self.assertEquals(xmlsupport.sax_eventlist(xml1, ids = self.thubanids,
                                                   idrefs = self.thubanidrefs),
                          xmlsupport.sax_eventlist(xml2, ids = self.thubanids,
                                                   idrefs = self.thubanidrefs))

    def testEmptySession(self):
        """Save an empty session"""
        session = Session("empty session")
        filename = self.temp_file_name("save_emptysession.thuban")
        save_session(session, filename)
        session.Destroy()

        file = open(filename)
        written_contents = file.read()
        file.close()
        self.compare_xml(written_contents,
                         '<?xml version="1.0" encoding="UTF-8"?>\n'
                         '<!DOCTYPE session SYSTEM "thuban-0.8.dtd">\n'
                         '<session title="empty session" '
         'xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">'
                         '\n</session>\n')

        self.validate_data(written_contents)

    def testSingleLayer(self):
        """Save a session with a single map with a single layer"""
        # deliberately put an apersand in the title :)
        session = Session("single map&layer")
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        map = Map("Test Map", projection = proj)
        session.AddMap(map)
        # use shapefile from the example data
        shpfile = os.path.join(os.path.dirname(__file__),
                               os.pardir, "Data", "iceland", "political.shp")
        layer = Layer("My Layer", session.OpenShapefile(shpfile))
        map.AddLayer(layer)

        filename = self.temp_file_name("save_singlemap.thuban")
        save_session(session, filename)

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_template = '''<?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE session SYSTEM "thuban-0.8.dtd">
        <session title="single map&amp;layer"
           xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">
            <fileshapesource id="D1" filename="%s" filetype="shapefile"/>
            <map title="Test Map">
                <projection name="Unknown">
                    <parameter value="zone=26"/>
                    <parameter value="proj=utm"/>
                    <parameter value="ellps=clrk66"/>
                </projection>
                <layer title="My Layer" shapestore="D1"
                fill="None" stroke="#000000" stroke_width="1" visible="%s"/>
            </map>
        </session>'''

        expected_contents = expected_template % \
            (os.path.join("..", "..", "Data", "iceland", "political.shp"),
             "true")

        self.compare_xml(written_contents, expected_contents)

        self.validate_data(written_contents)

        layer.SetVisible(False)
        save_session(session, filename)

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_contents = expected_template % \
            (os.path.join("..", "..", "Data", "iceland", "political.shp"),
             "false")
        self.compare_xml(written_contents, expected_contents)
        self.validate_data(written_contents)

        session.Destroy()

    def testLayerProjection(self):
        """Test saving layers with projections"""
        # deliberately put an apersand in the title :)
        session = Session("single map&layer")
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        map = Map("Test Map", projection = proj)
        session.AddMap(map)
        # use shapefile from the example data
        shpfile = os.path.join(os.path.dirname(__file__),
                               os.pardir, "Data", "iceland", "political.shp")
        layer = Layer("My Layer", session.OpenShapefile(shpfile))
        proj = Projection(["proj=lcc", "ellps=clrk66"], "Layer Projection")
        layer.SetProjection(proj)
        map.AddLayer(layer)

        filename = self.temp_file_name("save_layerproj.thuban")
        save_session(session, filename)
        session.Destroy()

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_contents = '''<?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE session SYSTEM "thuban-0.8.dtd">
        <session title="single map&amp;layer"
           xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">
            <fileshapesource id="D1" filename="%s" filetype="shapefile"/>
            <map title="Test Map">
                <projection name="Unknown">
                    <parameter value="zone=26"/>
                    <parameter value="proj=utm"/>
                    <parameter value="ellps=clrk66"/>
                </projection>
                <layer title="My Layer" shapestore="D1"
                fill="None" stroke="#000000" stroke_width="1" visible="true">
                    <projection name="Layer Projection">
                        <parameter value="proj=lcc"/>
                        <parameter value="ellps=clrk66"/>
                    </projection>
                </layer>
            </map>
        </session>''' % os.path.join("..", "..", "Data", "iceland",
                                     "political.shp")
        #print written_contents
        #print "********************************************"
        #print expected_contents
        self.compare_xml(written_contents, expected_contents)

        self.validate_data(written_contents)

    def testRasterLayer(self):
        # deliberately put an apersand in the title :)
        session = Session("single map&layer")
        map = Map("Test Map")
        session.AddMap(map)
        # use shapefile from the example data
        imgfile = os.path.join(os.path.dirname(__file__),
                               os.pardir, "Data", "iceland", "island.tif")
        layer = RasterLayer("My RasterLayer", imgfile)
        map.AddLayer(layer)

        filename = self.temp_file_name("save_singlemap.thuban")
        save_session(session, filename)
        session.Destroy()

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_contents = '''<?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE session SYSTEM "thuban-0.8.dtd">
        <session title="single map&amp;layer"
           xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">
            <map title="Test Map">
                <rasterlayer title="My RasterLayer" filename="%s"
                             visible="true">
                </rasterlayer>
            </map>
        </session>''' % os.path.join(os.path.dirname(__file__),
                                     os.pardir, "Data", "iceland",
                                     "island.tif")
        #print written_contents
        #print "********************************************"
        #print expected_contents
        self.compare_xml(written_contents, expected_contents)

        self.validate_data(written_contents)

    def testClassifiedLayer(self):
        """Save a session with a single map with a single layer
           with a classificaton.
        """
        # deliberately put an apersand in the title :)
        session = Session("single map&layer")
        proj = Projection(["zone=26", "proj=utm", "ellps=clrk66"])
        map = Map("Test Map", projection = proj)
        session.AddMap(map)
        # use shapefile from the example data
        shpfile = os.path.join(os.path.dirname(__file__),
                               os.pardir, "Data", "iceland", "political.shp")
        layer = Layer("My Layer", session.OpenShapefile(shpfile))
        map.AddLayer(layer)

        clazz = layer.GetClassification()

        clazz.SetField("AREA")

        clazz.AppendGroup(ClassGroupSingleton(42, 
                                           ClassGroupProperties(),
                                           "single"))
        clazz.AppendGroup(ClassGroupSingleton("text", 
                                           ClassGroupProperties(),
                                           "single-text"))

        clazz.AppendGroup(ClassGroupRange(0, 42,
                                           ClassGroupProperties(),
                                           "range"))

        range = ClassGroupRange(Range("[0;42]"))
        range.SetProperties(ClassGroupProperties())
        range.SetLabel("new-range")
        clazz.AppendGroup(range)

        filename = self.temp_file_name("save_singlemap.thuban")
        save_session(session, filename)

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_template = '''<?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE session SYSTEM "thuban-0.8.dtd">
        <session title="single map&amp;layer"
           xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">
            <fileshapesource id="D1" filename="%s" filetype="shapefile"/>
            <map title="Test Map">
                <projection name="Unknown">
                    <parameter value="zone=26"/>
                    <parameter value="proj=utm"/>
                    <parameter value="ellps=clrk66"/>
                </projection>
                <layer title="My Layer" shapestore="D1"
                fill="None" stroke="#000000" stroke_width="1" visible="%s">
                    <classification field="AREA" field_type="double">
                        <clnull label="">
                            <cldata fill="None" stroke="#000000" stroke_width="1"/>
                        </clnull>
                        <clpoint value="42" label="single">
                            <cldata fill="None" stroke="#000000" stroke_width="1"/>
                        </clpoint>
                        <clpoint value="text" label="single-text">
                            <cldata fill="None" stroke="#000000" stroke_width="1"/>
                        </clpoint>
                        <clrange range="[0;42[" label="range">
                            <cldata fill="None" stroke="#000000" stroke_width="1"/>
                        </clrange>
                        <clrange range="[0;42]" label="new-range">
                            <cldata fill="None" stroke="#000000" stroke_width="1"/>
                        </clrange>
                    </classification>
                </layer>
            </map>
        </session>'''

        expected_contents = expected_template % \
            (os.path.join("..", "..", "Data", "iceland", "political.shp"),
             "true")

        #print written_contents
        #print "********************************************"
        #print expected_contents
        self.compare_xml(written_contents, expected_contents)

        self.validate_data(written_contents)

        session.Destroy()

    def test_dbf_table(self):
        """Test saving a session with a dbf table link"""
        session = Session("a DBF Table session")
        # use shapefile from the example data
        dbffile = os.path.join(os.path.dirname(__file__),
                               os.pardir, "Data", "iceland", "political.dbf")
        table = session.AddTable(DBFTable(dbffile))

        filename = self.temp_file_name("save_singletable.thuban")
        save_session(session, filename)

        file = open(filename)
        written_contents = file.read()
        file.close()
        expected_template = '''<?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE session SYSTEM "thuban-0.8.dtd">
        <session title="a DBF Table session"
           xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">
            <filetable id="D1" filename="%s" filetype="DBF" title="political"/>
        </session>'''

        expected_contents = expected_template % dbffile
        self.compare_xml(written_contents, expected_contents)

    def test_joined_table(self):
        """Test saving a session with joined table"""
        # Create a simple table to use in the join
        dbffile = self.temp_file_name("save_joinedtable.dbf")
        dbf = dbflib.create(dbffile)
        dbf.add_field("RDTYPE", dbflib.FTInteger, 10, 0)
        dbf.add_field("TEXT", dbflib.FTString, 10, 0)
        dbf.write_record(0, {'RDTYPE': 8, "TEXT": "foo"})
        dbf.write_record(1, {'RDTYPE': 2, "TEXT": "bar"})
        dbf.write_record(2, {'RDTYPE': 3, "TEXT": "baz"})
        dbf.close()

        # Create the session and a map
        session = Session("A Joined Table session")
        try:
            map = Map("Test Map")
            session.AddMap(map)

            # Add the dbf file to the session
            dbftable = session.AddTable(DBFTable(dbffile))

            # Create a layer with the shapefile to use in the join
            shpfile = os.path.join(os.path.abspath(os.path.dirname(__file__)),
                                   os.pardir, "Data", "iceland",
                                   "roads-line.shp")
            layer = Layer("My Layer", session.OpenShapefile(shpfile))
            map.AddLayer(layer)

            # Do the join
            store = layer.ShapeStore()
            #for col in store.Table().Columns():
            #    print col.name
            joined = TransientJoinedTable(session.TransientDB(),
                                          store.Table(), "RDLNTYPE",
                                          dbftable, "RDTYPE")
            store = session.AddShapeStore(DerivedShapeStore(store, joined))
            layer.SetShapeStore(store)

            # Save the session
            filename = self.temp_file_name("save_joinedtable.thuban")
            save_session(session, filename)

            # Read it back and compare
            file = open(filename)
            written_contents = file.read()
            file.close()
            expected_template = '''<?xml version="1.0" encoding="UTF-8"?>
            <!DOCTYPE session SYSTEM "thuban-0.8.dtd">
            <session title="A Joined Table session"
             xmlns="http://thuban.intevation.org/dtds/thuban-0.8.dtd">
                <fileshapesource filename="%(shpfile)s"
                                 filetype="shapefile" id="D142197204"/>
                <filetable filename="%(dbffile)s"
                           title="save_joinedtable"
                           filetype="DBF" id="D141881756"/>
                <jointable id="D142180284"
                           title="Join of roads-line and save_joinedtable"
                           leftcolumn="RDLNTYPE" left="D142197204"
                           rightcolumn="RDTYPE" right="D141881756"/>
                <derivedshapesource id="D141915644"
                                    table="D142180284"
                                    shapesource="D142197204"/>
                <map title="Test Map">
                    <layer title="My Layer"
                           shapestore="D141915644" visible="true"
                           stroke="#000000" stroke_width="1" fill="None"/>
                </map>
            </session>'''

            expected_contents = expected_template % {
                "dbffile": relative_filename(self.temp_dir(), dbffile),
                "shpfile": relative_filename(self.temp_dir(), shpfile)
                }
            self.compare_xml(written_contents, expected_contents)
        finally:
            session.Destroy()
            session = None


class MockDataStore:

    """A very simple data store that only has dependencies"""

    def __init__(self, name, *dependencies):
        self.name = name
        self.dependencies = dependencies

    def __repr__(self):
        return self.name

    def Dependencies(self):
        return self.dependencies


class TestStoreSort(unittest.TestCase):

    def check_sort(self, containers, sorted):
        """Check whether the list of data containers is sorted"""
        # check whether sorted is in the right order
        seen = {}
        for container in sorted:
            self.failIf(id(container) in seen,
                        "Container %r at least twice in %r" % (container,
                                                               sorted))
            for dep in container.Dependencies():
                self.assert_(id(dep) in seen,
                             "Dependency %r of %r not yet seen" % (dep,
                                                                   container))
            seen[id(container)] = 1
        # check whether all of containers is in sorted
        for container in containers:
            self.assert_(id(container) in seen,
                         "Container %r in containers but not in sorted")
        self.assertEquals(len(containers), len(sorted))

    def test_sort_data_stores(self):
        """Test Thuban.Model.save.sort_data_stores"""
        d1 = MockDataStore("d1")
        d2 = MockDataStore("d2")
        d3 = MockDataStore("d3", d1)
        d4 = MockDataStore("d4", d1, d3)

        containers = [d4, d1, d2, d3]
        self.check_sort(containers, sort_data_stores(containers))
        containers = [d1, d3, d2, d4]
        self.check_sort(containers, sort_data_stores(containers))



if __name__ == "__main__":
    # Fake the __file__ global because it's needed by a test
    import sys
    __file__ = sys.argv[0]
    support.run_tests()
