# Copyright (c) 2002, 2003 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""
Test the Classification class
"""

__version__ = "$Revision: 1.12 $"
# $Source: /home/thuban/jail/thubanrepository/thuban/test/test_classification.py,v $
# $Id: test_classification.py,v 1.12 2003/05/09 16:34:53 jonathan Exp $

from __future__ import nested_scopes

import unittest

import support
support.initthuban()

import os
from Thuban.Model.color import Color
from Thuban.Model.table import FIELDTYPE_INT, FIELDTYPE_STRING, FIELDTYPE_DOUBLE
from Thuban.Model.classification import \
    Classification, ClassGroup, \
    ClassGroupDefault, ClassGroupSingleton, ClassGroupRange,\
    ClassGroupProperties

from Thuban.Model.session import Session
from Thuban.Model.layer import Layer

import copy


class TestClassification(unittest.TestCase):

    def test_ClassGroupProperties(self):
        """Test ClassGroupProperties"""

        props = ClassGroupProperties()
        self.assertEqual(props.GetLineColor(), Color.Black)
        self.assertEqual(props.GetLineWidth(), 1)
        self.assertEqual(props.GetFill(), Color.Transparent)

        red = Color(1, 0, 0)
        props.SetLineColor(red)
        self.assertEqual(props.GetLineColor(), red)

        blue = Color(0, 0, 1)
        props.SetLineColor(blue)
        self.assertEqual(props.GetLineColor(), blue)

        props.SetLineWidth(10)
        self.assertEqual(props.GetLineWidth(), 10)

        self.assertRaises(ValueError, props.SetLineWidth, -10)
        self.assertEqual(props.GetLineWidth(), 10)

        newProps1 = ClassGroupProperties()
        newProps2 = ClassGroupProperties()
        self.assertNotEqual(newProps1, props)
        self.assertEqual(newProps1, newProps2)

    def test_ClassGroup(self):
        """Test ClassGroup"""

        # test constructor with no label
        group = ClassGroup()
        self.assertEqual(group.GetLabel(), "")

        # test constructor with label
        group = ClassGroup("hallo")
        self.assertEqual(group.GetLabel(), "hallo")

        # test SetLabel()/GetLabel()
        group = ClassGroup("welt")
        group.SetLabel("hallo")
        self.assertEqual(group.GetLabel(), "hallo")

        group.SetLabel("")
        self.assertEqual(group.GetLabel(), "")

        # test Matches
        # Matches() is a virtual function...can't test it here
        #
        #self.assertEqual(group.Matches(None), False)
        #self.assertEqual(group.Matches(1), False)
        #self.assertEqual(group.Matches("hallo"), False)
        #self.assertEqual(group.Matches([]), False)

        # test GetProperties...also a virtual function
        #self.assertEqual(group.GetProperties(), None)

    def test_ClassGroupDefault(self):
        """Test ClassGroupDefault"""

        defProps = ClassGroupProperties()

        newProps = ClassGroupProperties()
        newProps.SetLineColor(Color(.25, .5, .75))
        newProps.SetLineWidth(5)
        newProps.SetFill(Color(.12, .24, .36))

        # test constructor

        group = ClassGroupDefault(newProps)
        self.assertEqual(group.GetProperties(), newProps)

        group = ClassGroupDefault(newProps, "hallo")
        self.assertEqual(group.GetProperties(), newProps)
        self.assertEqual(group.GetLabel(), "hallo")

        # test empty constructor
        group = ClassGroupDefault()
        props = group.GetProperties()

        self.assertEqual(group.GetLabel(), "")
        self.assertEqual(defProps, props)

        # test Matches()
        self.assertEqual(group.Matches(None), True)
        self.assertEqual(group.Matches(1), True)
        self.assertEqual(group.Matches("hallo"), True)
        self.assertEqual(group.Matches([]), True)

        # test SetProperties()/GetProperties()
        group.SetProperties(newProps)
        self.assertEqual(group.GetProperties(), newProps)

        # test copy
        groupCopy = copy.copy(group)
        self.assertEqual(group, groupCopy)

    def test_ClassGroupRange(self):
        """Test ClassGroupRange"""

        defProps = ClassGroupProperties()
        newProps = ClassGroupProperties()
        newProps.SetLineColor(Color(.25, .5, .75))
        newProps.SetLineWidth(5)
        newProps.SetFill(Color(.12, .24, .36))

        # test empty constructor
        group = ClassGroupRange()

        self.assertEqual(group.GetMin(), 0)
        self.assertEqual(group.GetMax(), 1)
        self.assertEqual(group.GetProperties(), defProps)
        self.assertEqual(group.GetLabel(), "")
        
        # test SetMax()
        self.assertRaises(ValueError, group.SetMax, 0)
        self.assertRaises(ValueError, group.SetMax, -1)
        self.assertEquals(group.GetMax(), 1)
        group.SetMax(2)
        self.assertEquals(group.GetMax(), 2)

        # test SetMin()
        self.assertRaises(ValueError, group.SetMin, 2)
        self.assertRaises(ValueError, group.SetMin, 3)
        self.assertEquals(group.GetMin(), 0)
        group.SetMin(-5)
        self.assertEquals(group.GetMin(), -5)

        # test SetProperties()/GetProperties()
        group.SetProperties(newProps)
        self.assertEqual(group.GetProperties(), newProps)

        # test SetRange()
        self.assertRaises(ValueError, group.SetRange, 1, 0)
        group.SetRange(-5, 5)
        self.assertEqual(group.GetRange(), "[-5;5[")

        # test Matches()
        self.assertEqual(group.Matches(-6), False)
        self.assertEqual(group.Matches(-5), True)
        self.assertEqual(group.Matches(0), True)
        self.assertEqual(group.Matches(4), True)
        self.assertEqual(group.Matches(5), False)

        # test copy
        groupCopy = copy.copy(group)
        self.assertEqual(group, groupCopy)

    def test_ClassGroupSingleton(self):
        """Test ClassGroupSingleton"""

        defProps = ClassGroupProperties()
        newProps = ClassGroupProperties()
        newProps.SetLineColor(Color(.25, .5, .75))
        newProps.SetLineWidth(5)
        newProps.SetFill(Color(.12, .24, .36))

        # test empty constructor
        group = ClassGroupSingleton()

        self.assertEqual(group.GetValue(), 0)
        self.assertEqual(group.GetProperties(), defProps)
        self.assertEqual(group.GetLabel(), "")

        # test SetProperties()/GetProperties()
        group.SetProperties(newProps)
        self.assertEqual(group.GetProperties(), newProps)

        # test SetValue()
        group.SetValue(5)
        self.assertEqual(group.GetValue(), 5)

        # test Matches()
        self.assertEqual(group.Matches(0), False)
        self.assertEqual(group.Matches(5), True)

        group.SetValue("5")
        self.assertNotEqual(group.GetValue(), 5)

        # test Matches()
        self.assertEqual(group.Matches(5), False)
        self.assertEqual(group.Matches("5"), True)

        group.SetValue("hallo")
        self.assertEqual(group.GetValue(), "hallo")

        # test Matches()
        self.assertEqual(group.Matches("HALLO"), False)
        self.assertEqual(group.Matches("hallo"), True)

        # test copy
        groupCopy = copy.copy(group)
        self.assertEqual(group, groupCopy)
        

    def test_ClassIterator(self):
        """Test ClassIterator"""

        groups = [ClassGroupSingleton(5), ClassGroupSingleton(5),
                  ClassGroupRange(-3, 3), ClassGroupSingleton(-5),
                  ClassGroupDefault()]

        clazz = Classification()

        for g in groups:
            clazz.AppendGroup(g)

        def convert(clazz):
            if isinstance(clazz, ClassGroupDefault):   return 0
            if isinstance(clazz, ClassGroupSingleton): return 1
            if isinstance(clazz, ClassGroupRange):     return 2

        list = []
        for g in clazz:
            list.append(convert(g))

        self.assertEquals(list, [0, 1, 1, 2, 1, 0])

    def test_classification(self):
        """Test Classification"""

        defProps = ClassGroupProperties()
        red   = Color(1, 0, 0)
        green = Color(0, 1, 0)
        blue  = Color(0, 0, 1)

        session = Session("Test session")
        filename = os.path.join("..", "Data", "iceland", "political.dbf")
        layer = Layer("asdf", session.OpenShapefile(filename))

        #
        # init with no params
        #
        c = Classification()
        self.assertEqual(c.GetField(), None)
        self.assertEqual(c.GetFieldType(), None)
        self.assertEqual(c.FindGroup(-1), c.GetDefaultGroup())

        c.SetDefaultLineColor(red)
        self.assertEqual(c.GetDefaultLineColor(), red)
        self.assertEqual(c.GetDefaultFill(), Color.Transparent)

        c.SetDefaultFill(green)
        self.assertEqual(c.GetDefaultFill(), green)
        self.assertEqual(c.GetDefaultLineColor(), red)

        c.SetField("hallo")
        self.assertEqual(c.GetField(), "hallo")

        c.SetFieldType(FIELDTYPE_STRING)
        self.assertEqual(c.GetFieldType(), FIELDTYPE_STRING)

        # should raise an exception because 'hallo' doesn't 
        # exist in the table
        self.assertRaises(ValueError, c.SetLayer, layer)
        
        c.SetField("AREA")
        c.SetLayer(layer)
        self.assertEqual(c.GetLayer(), layer)
        self.assertEqual(c.GetField(), "AREA")
        self.assertEqual(c.GetFieldType(), FIELDTYPE_DOUBLE)

        c.SetField(None)
        self.assertEquals(c.GetFieldType(), None)
        self.assertEquals(c.FindGroup(5), c.GetDefaultGroup())

        c.SetField("AREA")
        s = ClassGroupSingleton(5)
        c.AppendGroup(s)
        self.assertEquals(c.FindGroup(5), s)
        self.assertEquals(c.FindGroup(0), c.GetDefaultGroup())

        r = ClassGroupRange(-10, 10)
        c.AppendGroup(r)
        self.assertEquals(c.FindGroup(-11), c.GetDefaultGroup())
        self.assertEquals(c.FindGroup(-10), r)
        self.assertEquals(c.FindGroup(9), r)
        self.assertEquals(c.FindGroup(5), s)
        self.assertEquals(c.FindGroup(10), c.GetDefaultGroup())

        clazz = copy.deepcopy(c)

        self.assertEquals(clazz.GetNumGroups(), c.GetNumGroups())

        for i in range(clazz.GetNumGroups()):
            self.assertEquals(clazz.GetGroup(i), c.GetGroup(i))

        layer.Destroy()

if __name__ == "__main__":
    support.run_tests()
